/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.rules.filters.contition;

import android.text.TextUtils;
import android.util.Pair;

import org.json.JSONObject;

import java.util.Calendar;

import cn.thinkingdata.strategy.core.period.PeriodLimitFactory;
import cn.thinkingdata.strategy.core.period.TimeCycleUnit;
import cn.thinkingdata.strategy.rules.annotation.Action;
import cn.thinkingdata.strategy.rules.annotation.Condition;
import cn.thinkingdata.strategy.rules.annotation.Rule;
import cn.thinkingdata.strategy.rules.api.Facts;
import cn.thinkingdata.strategy.utils.AnalyticUtils;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.StrategyConstants;

/**
 * @author liulongbing
 * @since 2024/3/19
 */
@Rule(name = StrategyConstants.C14, description = "相对事件时间")
public class RELATIVE_EVENT_TIMERule extends PropertiesRule {

    public RELATIVE_EVENT_TIMERule(JSONObject ruleJson) {
        super(ruleJson);
    }

    @Condition
    public boolean when(Facts facts) {
        return isCondition(facts);
    }

    @Override
    public boolean executeWhen(Object value, Facts facts) {
        if (StrategyConstants.DATE_TYPE.equals(columnType)) {
            String v = value.toString();
            Object zOffset = facts.get(StrategyConstants.EVENT_ZONE_OFFSET);
            long timeStamp = CommonUtils.getTimeStampForObj(v, zOffset);
            String timeRelative = this.ruleJson.optString("timeRelative");
            if (TextUtils.equals(timeRelative, "range")) {
                if (ftv.size() == 2) {
                    long targetTimeStamp = getTargetTimeStamp(facts);
                    int before = Integer.parseInt(ftv.get(0).toString());
                    int after = Integer.parseInt(ftv.get(1).toString());
                    int millis = getMillis();
                    long start = targetTimeStamp + ( long ) before * millis;
                    long end = targetTimeStamp + ( long ) after * millis;
                    return timeStamp >= start && timeStamp <= end;
                } else {
                    return false;
                }
            } else {
                Pair<Long, Integer> pair = null;
                if (TextUtils.equals(timeRelative, "that_day")) {
                    pair = getDayTimeStamp(facts);
                } else if (TextUtils.equals(timeRelative, "that_week")) {
                    pair = getWeekTimeStamp(facts);
                } else if (TextUtils.equals(timeRelative, "that_month")) {
                    pair = getMonthTimeStamp(facts);
                }
                if (null != pair) {
                    long endStamp = pair.first + ( long ) pair.second * 24 * 60 * 60 * 1000;
                    return timeStamp >= pair.first && timeStamp < endStamp;
                }
            }
        }
        return false;
    }

    private int getMillis() {
        TimeCycleUnit timeCycleUnit = PeriodLimitFactory.getTimeCycleUnit(this.ruleJson.optString("timeUnit"));
        if (timeCycleUnit == null) return 0;
        if (timeCycleUnit == TimeCycleUnit.DAY) {
            return 24 * 60 * 60 * 1000;
        } else if (timeCycleUnit == TimeCycleUnit.HOUR) {
            return 60 * 60 * 1000;
        } else if (timeCycleUnit == TimeCycleUnit.MINUTE) {
            return 60 * 1000;
        }
        return 0;
    }

    @Action
    public void then(Facts facts) {

    }

    private Pair<Long, Integer> getDayTimeStamp(Facts facts) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(getTargetTimeStamp(facts));
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return new Pair<>(calendar.getTimeInMillis(), 1);
    }

    private Pair<Long, Integer> getWeekTimeStamp(Facts facts) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(getTargetTimeStamp(facts));
        int currentDayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
        int daysToSubtract = currentDayOfWeek - Calendar.SUNDAY;
        calendar.add(Calendar.DAY_OF_MONTH, -daysToSubtract);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return new Pair<>(calendar.getTimeInMillis(), 7);
    }

    private Pair<Long, Integer> getMonthTimeStamp(Facts facts) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(getTargetTimeStamp(facts));
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        long stamp = calendar.getTimeInMillis();
        int max = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);
        return new Pair<>(stamp, max);
    }

    private long getTargetTimeStamp(Facts facts) {
        try {
            Object eventTime = facts.get(StrategyConstants.EVENT_TIME_KEY);
            Object zoneOffset = facts.get(StrategyConstants.EVENT_ZONE_OFFSET);
            return CommonUtils.getTimeStampForObj(eventTime, zoneOffset);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return AnalyticUtils.getCurrentTimeStamp();
    }
}
