/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.rules.filters.analytic;

import android.text.TextUtils;
import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import cn.thinkingdata.strategy.core.period.ITimeCycle;
import cn.thinkingdata.strategy.core.trigger.AbstractTriggerRule;
import cn.thinkingdata.strategy.core.trigger.entity.RelationProperties;
import cn.thinkingdata.strategy.rules.annotation.Action;
import cn.thinkingdata.strategy.rules.annotation.Condition;
import cn.thinkingdata.strategy.rules.annotation.Rule;
import cn.thinkingdata.strategy.rules.api.Facts;
import cn.thinkingdata.strategy.rules.filters.contition.EQRule;
import cn.thinkingdata.strategy.rules.support.composite.UnitRuleGroup;
import cn.thinkingdata.strategy.storage.db.StrategyDataBaseHelper;
import cn.thinkingdata.strategy.storage.db.TriggerResultInfo;
import cn.thinkingdata.strategy.utils.AnalyticUtils;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.StrategyConstants;
import cn.thinkingdata.strategy.utils.TDStrategyLog;

/**
 * @author liulongbing
 * @since 2024/3/11
 */
@Rule(name = StrategyConstants.A200, description = "总次数")
public class EventCountRule {

    private String key;
    private int num;
    private String eventName;

    private final AbstractTriggerRule triggerRule;

    public EventCountRule(JSONObject ruleJson, AbstractTriggerRule triggerRule) {
        try {
            key = ruleJson.optString("key");
            num = ruleJson.optInt("num");
            eventName = ruleJson.optString("eventName");
        } catch (Exception e) {
            e.printStackTrace();
        }
        this.triggerRule = triggerRule;
    }

    @Condition
    public boolean when(Facts facts) {
        try {
            ITimeCycle timeCycle = triggerRule.getTimeCycle();
            if (timeCycle == null) return false;
            Pair<Boolean, JSONObject> relationPair = getRelationProperties(facts);
            if (!relationPair.first) return false;
            Pair<Long, Long> pair = timeCycle.getCurrentTimeCycle();
            TriggerResultInfo info = new TriggerResultInfo();
            info.appId = triggerRule.mTaskFlow.appId;
            info.userId = triggerRule.clientUserId;
            info.taskId = triggerRule.mTaskFlow.taskId;
            info.triggerId = triggerRule.mTriggerId;
            TDStrategyLog.i("EventName：" + eventName + " Start ：" + CommonUtils.getLogFormatTime(pair.first) + " End ：" + CommonUtils.getLogFormatTime(pair.second));
            List<TriggerResultInfo> lists = StrategyDataBaseHelper.getHelper().getTriggerResultList(info, pair.first, pair.second);
            Pair<Integer, List<String>> countPairs = getEventCount(lists, relationPair.second);
            int count = countPairs.first;
            if (this.triggerRule.isWithInPeriod()) {
                count++;
            }
            TDStrategyLog.i("EventName：" + eventName + " Trigger Count：" + count + " Target Count：" + num);
            if (count >= num) {
                StrategyDataBaseHelper.getHelper().deleteTriggerResultByIds(countPairs.second);
                return true;
            }
            info.key = key;
            info.result = 1;
            info.relationProperties = relationPair.second.toString();
            StrategyDataBaseHelper.getHelper().addTriggerResult(info);
            return false;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    private Pair<Integer, List<String>> getEventCount(List<TriggerResultInfo> lists, JSONObject relationJson) {
        int i = 0;
        List<String> ids = new ArrayList<>();
        for (TriggerResultInfo info : lists) {
            if (TextUtils.equals(info.key, key) && isRelationEq(relationJson, info.relationProperties)) {
                i++;
                ids.add(info._id + "");
            } else {
                if (triggerRule.disableEventList.contains(info.key)) {
                    break;
                }
            }
        }
        return new Pair<>(i, ids);
    }

    @Action
    public void then(Facts facts) {

    }

    private boolean isRelationEq(JSONObject relationJson, String str) {
        if (triggerRule.relationProperties.size() == 0) return true;
        if (TextUtils.isEmpty(str) && relationJson.length() == 0) return true;
        try {
            JSONObject targetJson = new JSONObject(str);
            if (relationJson.length() != targetJson.length()) return false;
            UnitRuleGroup unitRuleGroup = new UnitRuleGroup();
            Facts relationFacts = new Facts();
            for (RelationProperties relationProperty : triggerRule.relationProperties) {
                JSONObject itemJson = new JSONObject();
                String columnName = relationProperty.columnName;
                Object columnValue = relationJson.opt(columnName);
                if (null == columnValue) continue;
                itemJson.put("calcuSymbol", StrategyConstants.C00);
                itemJson.put("columnName", columnName);
                itemJson.put("columnType", relationProperty.columnType);
                JSONArray itemArray = new JSONArray();
                itemArray.put(targetJson.opt(columnName));
                itemJson.put("ftv", itemArray);
                EQRule eqRule = new EQRule(itemJson);
                unitRuleGroup.addRule(eqRule);
                relationFacts.put(columnName, columnValue);
            }
            TDStrategyLog.i("TriggerID：" + triggerRule.mTriggerId + "------start relation properties------");
            boolean isRelation = unitRuleGroup.evaluate(relationFacts);
            TDStrategyLog.i("TriggerID：" + triggerRule.mTriggerId + "------end relation properties------result is " + isRelation);
            return isRelation;
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return false;
    }

    private Pair<Boolean, JSONObject> getRelationProperties(Facts facts) {
        JSONObject json = new JSONObject();
        for (RelationProperties relationProperty : triggerRule.relationProperties) {
            Object obj = facts.get(relationProperty.columnName);
            if (obj == null) continue;
            try {
                switch (relationProperty.columnType) {
                    case StrategyConstants.STR_TYPE:
                    case StrategyConstants.DATE_TYPE:
                        String v1 = ( String ) obj;
                        json.put(relationProperty.columnName, v1);
                        break;
                    case StrategyConstants.INTEGER_TYPE:
                        int v2 = Integer.parseInt(obj.toString());
                        json.put(relationProperty.columnName, v2);
                        break;
                    case StrategyConstants.BIGINT_TYPE:
                        long v3 = Long.parseLong(obj.toString());
                        json.put(relationProperty.columnName, v3);
                        break;
                    case StrategyConstants.DOUBLE_TYPE:
                        double v4 = Double.parseDouble(obj.toString());
                        json.put(relationProperty.columnName, v4);
                        break;
                    case StrategyConstants.DECIMAL_TYPE:
                        BigDecimal v5 = ( BigDecimal ) obj;
                        json.put(relationProperty.columnName, v5);
                        break;
                    default:
                        json.put(relationProperty.columnName, obj);
                        break;
                }
            } catch (Exception e) {
                e.printStackTrace();
                //类型强转失败 丢弃事件
                return new Pair<>(false, null);
            }
        }
        return new Pair<>(true, json);
    }

}
