/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.core.trigger;

import android.text.TextUtils;
import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import cn.thinkingdata.strategy.core.period.ITimeCycle;
import cn.thinkingdata.strategy.core.period.PeriodLimitFactory;
import cn.thinkingdata.strategy.core.period.PeriodScrollInfo;
import cn.thinkingdata.strategy.core.period.SlideTimeCycle;
import cn.thinkingdata.strategy.core.period.TaskPeriodInfo;
import cn.thinkingdata.strategy.core.period.TimeCycleUnit;
import cn.thinkingdata.strategy.core.task.TaskFlow;
import cn.thinkingdata.strategy.core.trigger.entity.RelationProperties;
import cn.thinkingdata.strategy.rules.api.Facts;
import cn.thinkingdata.strategy.rules.filters.factory.RuleFactory;
import cn.thinkingdata.strategy.rules.support.composite.ActivationRuleGroup;
import cn.thinkingdata.strategy.rules.support.composite.CompositeRule;
import cn.thinkingdata.strategy.rules.support.composite.UnitRuleGroup;
import cn.thinkingdata.strategy.storage.db.StrategyDataBaseHelper;
import cn.thinkingdata.strategy.storage.db.TriggerResultInfo;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.StrategyConstants;
import cn.thinkingdata.strategy.utils.TDStrategyLog;

/**
 * @author liulongbing
 * @since 2024/3/26
 * 连续完成A
 */
public class ContinueTriggerRule extends AbstractTriggerRule {

    private PeriodScrollInfo periodScrollInfo;
    private ActivationRuleGroup blackListGroup;

    public ContinueTriggerRule(TaskFlow mTaskFlow, JSONObject rule, int triggerId) {
        super(mTaskFlow, rule, triggerId);
    }

    @Override
    protected void parseFilters(JSONObject rule) throws Exception {
        super.parseFilters(rule);
        parseTimeCycleHour(rule);
        parseRelationProperties(rule);
        parseDisableEvent(rule);
    }

    private void parseTimeCycleHour(JSONObject rule) {
        int windowGap = rule.optInt("windowGap");
        String timeCycleUnit = rule.optString("windowGapTimeUnit");
        TimeCycleUnit timeUnit = PeriodLimitFactory.getTimeCycleUnit(timeCycleUnit);
        if (windowGap > 0 && timeUnit != null) {
            periodScrollInfo = new PeriodScrollInfo();
            periodScrollInfo.timeCycleNumber = windowGap;
            periodScrollInfo.timeCycleUnit = timeUnit;
        }
    }

    private void parseRelationProperties(JSONObject rule) {
        JSONArray associatedFilters = rule.optJSONArray("relationProps");
        if (associatedFilters == null) return;
        for (int i = 0; i < associatedFilters.length(); i++) {
            JSONObject property = associatedFilters.optJSONObject(i);
            if (property == null) continue;
            String columnName = property.optString("columnName");
            String columnType = property.optString("columnType");
            if (TextUtils.isEmpty(columnName) || TextUtils.isEmpty(columnType)) continue;
            RelationProperties properties = new RelationProperties();
            properties.columnName = columnName;
            properties.columnType = columnType;
            relationProperties.add(properties);
        }
    }

    private void parseDisableEvent(JSONObject rule) throws JSONException {
        JSONArray blackList = rule.optJSONArray("blackList");
        if (blackList == null) return;
        blackListGroup = new ActivationRuleGroup();
        for (int i = 0; i < blackList.length(); i++) {
            JSONObject item = blackList.optJSONObject(i);
            if (item == null) continue;
            String eventName = item.optString("eventName");
            disableEventList.add(eventName);
            UnitRuleGroup unitRuleGroup = new UnitRuleGroup();
            int relation = item.optInt("relation");
            JSONArray fits = item.optJSONArray("filts");
            CompositeRule compositeRule = CommonUtils.parseCompound(relation, fits, this.mTaskFlow.zoneOffset);
            unitRuleGroup.addRule(CommonUtils.buildEventNameRule(eventName));
            if (compositeRule != null) {
                unitRuleGroup.addRule(compositeRule);
            }
            blackListGroup.addRule(unitRuleGroup);
        }
    }

    @Override
    public boolean isTargetEvent(String event) {
        return super.isTargetEvent(event) || disableEventList.contains(event);
    }

    @Override
    public boolean triggerEventRules(Facts facts) {
        if (null == facts) return false;
        Object eventName = facts.get(StrategyConstants.EVENT_NAME_KEY);
        if (eventName != null && disableEventList.contains(eventName.toString())) {
            TDStrategyLog.i("BlackList Event:" + eventName + "------start trigger rules------");
            boolean isBlackTrigger = blackListGroup.evaluate(facts);
            TDStrategyLog.i("BlackList Event:" + eventName + "------end trigger rules------result is " + isBlackTrigger);
            if (isBlackTrigger) {
                TriggerResultInfo info = new TriggerResultInfo();
                info.appId = mTaskFlow.appId;
                info.userId = clientUserId;
                info.taskId = mTaskFlow.taskId;
                info.key = eventName.toString();
                info.triggerId = mTriggerId;
                info.result = 1;
                StrategyDataBaseHelper.getHelper().addTriggerResult(info);
            }
            return false;
        }
        return triggerEventFilters(facts);
    }

    @Override
    public ITimeCycle getTimeCycle() {
        if (mTriggerTimeCycle == null) return null;
        if (periodScrollInfo == null) return mTriggerTimeCycle;
        Pair<Long, Long> pair = mTriggerTimeCycle.getCurrentTimeCycle();
        TaskPeriodInfo tInfo = new TaskPeriodInfo();
        tInfo.periodStartStamp = pair.first;
        tInfo.periodEndStamp = pair.second;
        return new SlideTimeCycle(tInfo, periodScrollInfo);
    }
}