/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.core.trigger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import cn.thinkingdata.remoteconfig.TDRemoteConfig;
import cn.thinkingdata.strategy.core.period.ITimeCycle;
import cn.thinkingdata.strategy.core.period.PeriodLimitFactory;
import cn.thinkingdata.strategy.core.task.TaskFlow;
import cn.thinkingdata.strategy.core.trigger.entity.RelationProperties;
import cn.thinkingdata.strategy.rules.api.Facts;
import cn.thinkingdata.strategy.rules.filters.factory.RuleFactory;
import cn.thinkingdata.strategy.rules.support.composite.ActivationRuleGroup;
import cn.thinkingdata.strategy.rules.support.composite.CompositeRule;
import cn.thinkingdata.strategy.rules.support.composite.UnitRuleGroup;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.TDStrategyLog;

/**
 * @author liulongbing
 * @since 2024/3/26
 */
public abstract class AbstractTriggerRule implements ITriggerRule {

    public int mTriggerId;
    protected final List<String> targetEventList;
    public final TaskFlow mTaskFlow;
    protected ActivationRuleGroup mEventFilters;
    protected ITimeCycle mTriggerTimeCycle;
    public final List<String> disableEventList;
    public String clientUserId;
    public List<RelationProperties> relationProperties;

    public AbstractTriggerRule(TaskFlow mTaskFlow, JSONObject rule, int mTriggerId) {
        targetEventList = new ArrayList<>();
        disableEventList = new ArrayList<>();
        relationProperties = new ArrayList<>();
        this.mTaskFlow = mTaskFlow;
        clientUserId = mTaskFlow.clientUserId;
        this.mTriggerId = mTriggerId;
        mEventFilters = new ActivationRuleGroup();
        mTriggerTimeCycle = PeriodLimitFactory.createTriggerTimeCycle(rule, mTaskFlow.zoneOffset);
        try {
            parseFilters(rule);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public boolean isWithInPeriod() {
        if (this.mTriggerTimeCycle != null) {
            return this.mTriggerTimeCycle.isWithInPeriod();
        }
        return false;
    }

    protected void parseFilters(JSONObject rule) throws Exception {
        parseEventFilters(rule);
    }

    private void parseEventFilters(JSONObject json) throws JSONException {
        JSONArray events = json.optJSONArray("events");
        if (events == null) return;
        for (int i = 0; i < events.length(); i++) {
            JSONObject eventItem = events.optJSONObject(i);
            if (eventItem == null) continue;
            String eventName = eventItem.optString("eventName");
            targetEventList.add(eventName);
            Object eventObj = RuleFactory.createAnalyticsRule(eventItem, this);
            if (null == eventObj) continue;
            UnitRuleGroup unitRuleGroup = new UnitRuleGroup();
            //1 表示且，0 表示或
            int relation = eventItem.optInt("relation");
            JSONArray fits = eventItem.optJSONArray("filts");
            CompositeRule compositeRule = CommonUtils.parseCompound(relation, fits, this.mTaskFlow.zoneOffset);
            unitRuleGroup.addRule(CommonUtils.buildEventNameRule(eventName));
            if (compositeRule != null) {
                unitRuleGroup.addRule(compositeRule);
            }
            unitRuleGroup.addRule(eventObj);
            mEventFilters.addRule(unitRuleGroup);
        }
    }

    protected boolean triggerEventFilters(Facts facts) {
        TDStrategyLog.i("TriggerID：" + mTriggerId + "------start trigger rules------");
        boolean isTrigger = mEventFilters.evaluate(facts);
        TDStrategyLog.i("TriggerID：" + mTriggerId + "------end trigger rules------result is " + isTrigger);
        return isTrigger;
    }

    @Override
    public boolean isTargetEvent(String event) {
        return targetEventList.contains(event);
    }
}
