/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.core.task;

import android.os.SystemClock;
import android.text.TextUtils;
import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import cn.thinkingdata.remoteconfig.TDRemoteConfig;
import cn.thinkingdata.remoteconfig.user.ClientUserManager;
import cn.thinkingdata.strategy.TDStrategy;
import cn.thinkingdata.strategy.core.result.TDStrategyResult;
import cn.thinkingdata.strategy.core.result.TDStrategyResultObservable;
import cn.thinkingdata.strategy.rules.api.Facts;
import cn.thinkingdata.strategy.storage.db.StrategyDataBaseHelper;
import cn.thinkingdata.strategy.storage.db.StrategyTaskInfo;
import cn.thinkingdata.strategy.task.StrategyTaskManager;
import cn.thinkingdata.strategy.utils.AnalyticUtils;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.SpUtils;
import cn.thinkingdata.strategy.utils.StrategyConstants;
import cn.thinkingdata.strategy.utils.TDStrategyLog;

/**
 * @author liulongbing
 * @since 2024/3/12
 */
public class TaskFlowManager {

    public static final Map<String, TaskFlowManager> instances = new HashMap<>();
    private final String appId;
    private final Map<String, TaskFlow> taskFlows;
    private boolean mUserTaskStatus = false;//用户任务状态

    public static long clientStartTime = 0L;//单次启动时间
    public static long clientStartElapsedRealtime;//单次启动时的开机时间，解决客户启动时间不准的问题

    private final List<String> mDebugTriggerList;

    private final List<Pair<Long, JSONObject>> preEventCaches;
    private boolean loadTaskSuccess = false;

    private TaskFlowManager(String appId) {
        this.appId = appId;
        mDebugTriggerList = new ArrayList<>();
        preEventCaches = new ArrayList<>();
        StrategyDataBaseHelper.addDatabaseChangeListener(new StrategyDataBaseHelper.OnDatabaseChangeListener() {
            @Override
            public void onDbModified() {
                clearAllTasks();
            }
        });
        taskFlows = new HashMap<>();
    }

    public static TaskFlowManager getInstance(String appId) {
        TaskFlowManager instance = instances.get(appId);
        if (instance == null) {
            synchronized (TaskFlowManager.class) {
                if (instances.get(appId) == null) {
                    instance = new TaskFlowManager(appId);
                    instances.put(appId, instance);
                }
            }
        }
        return instance;
    }

    public void deleteInvalidTasks() {
        if (CommonUtils.isStrategyStop(appId)) return;
        //删除31天之前的疲劳数据
        long invalidTime = AnalyticUtils.getCurrentTimeStamp() - 31 * 24 * 60 * 60 * 1000L;
        StrategyDataBaseHelper.getHelper().deleteTouchLimitByTime(invalidTime);
    }

    public void loadLocalTask() {
        if (CommonUtils.isStrategyStop(appId)) return;
        clientStartElapsedRealtime = SystemClock.elapsedRealtime();
        try {
            StrategyDataBaseHelper.getHelper().deleteTaskByLimit();
            StrategyDataBaseHelper.getHelper().deleteTriggerResultLimit();
            StrategyDataBaseHelper.getHelper().deleteFrequencyByLimit();
            Future<String> userIdFuture = ClientUserManager.getInstance(TDStrategy.mContext).getClientUserId(appId,
                    AnalyticUtils.getAccountId(appId), AnalyticUtils.getDistinctId(appId));
            String userId = userIdFuture.get(3, TimeUnit.SECONDS);
            mUserTaskStatus = SpUtils.getInstance().getBoolean(appId + userId, true);
            Map<String, StrategyTaskInfo> taskMap = StrategyDataBaseHelper.getHelper().getTasksByUser(appId, userId);
            for (String key : taskMap.keySet()) {
                StrategyTaskInfo info = taskMap.get(key);
                if (info == null || TextUtils.isEmpty(info.task)) continue;
                taskFlows.put(key, new TaskFlow(this.appId, userId, new JSONObject(info.task)));
            }
            if (taskFlows.size() > 0) {
                clearPreEvent();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void stopTasks(String templateCode, String userId) {
        if (CommonUtils.isStrategyStop(appId)) return;
        if (!TextUtils.equals(templateCode, StrategyConstants.STRATEGY_TEMPLATE_CODE)) return;
        taskFlows.clear();
        mUserTaskStatus = false;
        SpUtils.getInstance().putBoolean(appId + userId, false);
    }

    public void applyNewConfig(String templateCode, String userId) {
        if (CommonUtils.isStrategyStop(appId)) return;
        if (!TextUtils.equals(templateCode, StrategyConstants.STRATEGY_TEMPLATE_CODE)) return;
        mUserTaskStatus = true;
        SpUtils.getInstance().putBoolean(appId + userId, true);
        JSONArray taskJsonArray = TDRemoteConfig.getData(appId,templateCode).get(StrategyConstants.STRATEGY_CONFIG_KEY).arrayValue();
        if (taskJsonArray == null) {
            taskJsonArray = new JSONArray();
        }
        Map<String, StrategyTaskInfo> taskMap = StrategyDataBaseHelper.getHelper().getTasksByUser(appId, userId);
        List<String> taskIds = new ArrayList<>();
        List<String> pauseTaskIds = new ArrayList<>();
        List<String> testTaskIds = new ArrayList<>();
        for (int i = 0; i < taskJsonArray.length(); i++) {
            JSONObject tJson = taskJsonArray.optJSONObject(i);
            if (tJson != null) {
                String taskId = tJson.optString("taskId");
                long version = tJson.optLong("currentVersion");
                int taskStatus = tJson.optInt("status");
                StrategyTaskInfo newTask = new StrategyTaskInfo();
                newTask.appId = this.appId;
                newTask.userId = userId;
                newTask.taskId = taskId;
                newTask.taskVersion = version;
                if (taskStatus != 1) {
                    pauseTaskIds.add(taskId);
                    newTask.task = "";
                    StrategyDataBaseHelper.getHelper().updateTask(newTask);
                    continue;
                }
                newTask.task = tJson.toString();
                StrategyTaskInfo taskInfo = taskMap.get(taskId);
                if (taskInfo == null) {
                    StrategyDataBaseHelper.getHelper().createNewTask(newTask);
                } else if (version >= taskInfo.taskVersion) {
                    StrategyDataBaseHelper.getHelper().updateTask(newTask);
                    //任务升级之后需要删除任务计算结果
                    if (version > taskInfo.taskVersion) {
                        StrategyDataBaseHelper.getHelper().deleteTriggerResult(appId, userId, taskId);
                    }
                }
                taskIds.add(taskId);
                TaskFlow flow = taskFlows.get(taskId);
                if (flow == null || version >= flow.taskVersion) {
                    flow = new TaskFlow(this.appId, userId, tJson);
                    taskFlows.put(taskId, flow);
                }
                if (CommonUtils.isTestTask(flow.taskMode)) {
                    testTaskIds.add(taskId);
                }
            }
        }
        //删除数据库多余的任务
        for (String tId : taskMap.keySet()) {
            //暂停的任务暂时不需要删除
            if (!taskIds.contains(tId) && !pauseTaskIds.contains(tId)) {
                //删除任务
                StrategyDataBaseHelper.getHelper().deleteTask(appId, userId, tId);
                //删除任务计算结果
                StrategyDataBaseHelper.getHelper().deleteTriggerResult(appId, userId, tId);
                //删除任务频控
                StrategyDataBaseHelper.getHelper().deleteFrequencyLimit(appId, userId, tId);
            }
        }
        //删除内存中多余的任务 暂停的任务也需要从内存中删除
        Iterator<Map.Entry<String, TaskFlow>> iterator = taskFlows.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, TaskFlow> entry = iterator.next();
            String taskId = entry.getKey();
            if (!taskIds.contains(taskId)) {
                iterator.remove();
            }
        }
        //清除初始化前置事件
        clearPreEvent();

        //发送debug任务拉取成功日志
        AnalyticUtils.trackClientDebugEvent(appId, testTaskIds);
        //如果是通道测试任务 直接触发
        for (String testTaskId : testTaskIds) {
            TaskFlow t = taskFlows.get(testTaskId);
            if (t == null) continue;
            //如果是通道测试任务 需要主动发送
            if (!CommonUtils.isChannelTestTask(t.taskMode)) continue;
            taskTriggerFlow(t, AnalyticUtils.CHANNEL_TEST_EVENT, AnalyticUtils.buildAnalyticPresetFacts(appId), 0);
        }
    }

    private void clearPreEvent() {
        loadTaskSuccess = true;
        if (preEventCaches.size() > 0) {
            for (Pair<Long, JSONObject> preEvent : preEventCaches) {
                triggerEvent(preEvent.second, preEvent.first);
            }
            preEventCaches.clear();
        }
    }

    public void triggerEvent(JSONObject eventJson, long eventEnqueueTime) {
        if (CommonUtils.isStrategyStop(appId)) return;
        if (eventJson == null) return;
        if (!mUserTaskStatus) return;
        if (!CommonUtils.isCommonEvent(eventJson)) return;
        if (!loadTaskSuccess) {
            //如果任务没加载成功 先缓存起来 然后进行事件回放
            if (preEventCaches.size() < 20) {
                preEventCaches.add(new Pair<>(SystemClock.elapsedRealtime(), eventJson));
            }
            return;
        }
        String eventName = eventJson.optString(StrategyConstants.EVENT_NAME_KEY);
        if (AnalyticUtils.isBlackEvent(eventName)) return;
        Facts facts = CommonUtils.jsonToFacts(eventJson);
        TDStrategyLog.i("Event received:" + eventName);
        for (String key : taskFlows.keySet()) {
            TaskFlow flow = taskFlows.get(key);
            if (flow == null) continue;
            taskTriggerFlow(flow, eventName, facts, eventEnqueueTime);
        }
    }

    private void taskTriggerFlow(TaskFlow flow, String eventName, Facts facts, long eventEnqueueTime) {
        //如果是通道任务测试 只允许触发一次
        if (mDebugTriggerList.contains(flow.taskId)) return;
        //开始重置埋点参数
        flow.startClientRecord();
        //判断是否是任务事件
        if (!flow.isTargetEvent(eventName)) return;
        //判断是否超时
        if (flow.isEventTimeout(eventEnqueueTime)) return;
        //判断是否在任务周期内
        if (!flow.isValidityPeriod()) return;
        //属性筛选 次数判断
        if (!flow.triggerRules(facts)) return;
        //受众判断
        if (!flow.isTargetAudience(facts)) return;
        //实验条件判断
        if (!flow.isExpRelease()) return;
        //频控 + 抽样淘汰 + 疲劳 + 跳过推送
        if (flow.isTaskLimit()) return;
        //计算推送内容
        TDStrategyResult result = flow.calculate(facts);
        if (null != result) {
            TDStrategyLog.i("Task (" + flow.taskId + ") will be pushed soon");
            TDStrategyResultObservable.getInstance().triggerResult(result);
            if (CommonUtils.isChannelTestTask(flow.taskMode)) {
                mDebugTriggerList.add(flow.taskId);
            }
            //推送埋点结束
            flow.finishClientRecord();
        }
    }

    private void clearAllTasks() {
        StrategyTaskManager.getInstance().addTask(new Runnable() {
            @Override
            public void run() {
                //加入到队列任务中去删除 否则会报错
                taskFlows.clear();
                AnalyticUtils.trackRiskEvent(appId);
            }
        });
    }


}
