/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.core.task;

import android.os.SystemClock;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import cn.thinkingdata.strategy.core.audience.TargetAudienceManager;
import cn.thinkingdata.strategy.core.exp.ExpConfigManager;
import cn.thinkingdata.strategy.core.frequency.FrequencyLimit;
import cn.thinkingdata.strategy.core.frequency.ITaskLimit;
import cn.thinkingdata.strategy.core.frequency.UserFatigueControl;
import cn.thinkingdata.strategy.core.period.ITimeCycle;
import cn.thinkingdata.strategy.core.period.StartEndLimit;
import cn.thinkingdata.strategy.core.period.TaskPeriodInfo;
import cn.thinkingdata.strategy.core.result.PushResultManager;
import cn.thinkingdata.strategy.core.result.TDStrategyResult;
import cn.thinkingdata.strategy.core.result.TDStrategyResultObservable;
import cn.thinkingdata.strategy.core.trigger.ITriggerRule;
import cn.thinkingdata.strategy.core.trigger.TriggerRuleFactory;
import cn.thinkingdata.strategy.core.trigger.entity.ClientTriggerRecord;
import cn.thinkingdata.strategy.rules.api.Facts;
import cn.thinkingdata.strategy.storage.db.StrategyDataBaseHelper;
import cn.thinkingdata.strategy.utils.AnalyticUtils;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.StrategyConstants;
import cn.thinkingdata.strategy.utils.TDStrategyLog;

/**
 * @author liulongbing
 * @since 2024/3/11
 */
public class TaskFlow {

    public String appId;
    public String projectId;
    public String taskId;
    public String clientUserId;
    public long taskVersion;
    public String channelMsgType;
    public String channelId;
    public ITimeCycle mTimeCycle;
    public double zoneOffset;
    private final List<ITriggerRule> triggerRules = new ArrayList<>();
    private ITaskLimit mTaskFrequencyLimit;
    private ITaskLimit mTaskTouchLimit;
    private TargetAudienceManager mTargetAudienceManager;
    private PushResultManager mPushResultManger;
    private ExpConfigManager mExpConfigManager;
    public ClientTriggerRecord mTriggerRecord;
    public int taskMode;
    private long eventTimeout;

    public TaskFlow(String appId, String clientUserId, JSONObject taskJson) {
        this.appId = appId;
        this.clientUserId = clientUserId;
        try {
            parseTaskInfo(taskJson);
            parseTriggerRule(taskJson);
            parseTargetAudience(taskJson);
            parseExpConfig(taskJson);
            parseTaskLimit(taskJson);
            parseContentList(taskJson);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void parseTaskInfo(JSONObject taskJson) {
        taskId = taskJson.optString("taskId");
        projectId = taskJson.optString("projectId");
        mTriggerRecord = new ClientTriggerRecord(appId, taskId, clientUserId);
        zoneOffset = taskJson.optDouble("tzOffset");
        taskVersion = taskJson.optLong("currentVersion");
        channelMsgType = taskJson.optString("channelMsgType");
        channelId = taskJson.optString("channelId");
        taskMode = taskJson.optInt("pushMode", StrategyConstants.NORMAL_TYPE_TASK);
        String startDate = taskJson.optString("startDate");
        String endDate = taskJson.optString("endDate");
        eventTimeout = taskJson.optLong("eventTimeout",5000);
        TaskPeriodInfo tInfo = new TaskPeriodInfo();
        tInfo.periodStart = startDate;
        tInfo.periodEnd = endDate;
        tInfo.zoneOffset = zoneOffset;
        tInfo.timePattern = StrategyConstants.TIME_PATTERN;
        mTimeCycle = new StartEndLimit(tInfo);
    }

    private void parseContentList(JSONObject taskJson) {
        mPushResultManger = new PushResultManager(this, taskJson);
    }

    private void parseExpConfig(JSONObject taskJson) {
        mExpConfigManager = new ExpConfigManager(taskJson, zoneOffset);
    }

    private void parseTargetAudience(JSONObject taskJson) {
        boolean isTargetUser = taskJson.optBoolean("isTargetUser");
        String clientQpStr = taskJson.optString("clientQp");
        JSONObject clientQpJson;
        try {
            clientQpJson = new JSONObject(clientQpStr);
        } catch (JSONException e) {
            clientQpJson = new JSONObject();
        }
        String pushId = taskJson.optString("pushId");
        mTargetAudienceManager = new TargetAudienceManager(this, isTargetUser, clientQpJson, pushId);
    }

    private void parseTriggerRule(JSONObject taskJson) {
        String ruleStr = taskJson.optString("triggerRule");
        JSONArray ruleArray = null;
        try {
            ruleArray = new JSONArray(ruleStr);
        } catch (Exception ignore) {
        }
        if (ruleArray == null) return;
        for (int i = 0; i < ruleArray.length(); i++) {
            JSONObject item = ruleArray.optJSONObject(i);
            if (item == null) continue;
            int eventTriggerType = item.optInt("eventTriggerType");
            ITriggerRule triggerRule = TriggerRuleFactory.createTriggerRule(this, item, eventTriggerType, i);
            if (triggerRule == null) continue;
            triggerRules.add(triggerRule);
        }
    }

    private void parseTaskLimit(JSONObject taskJson) {
        //频控
        mTaskFrequencyLimit = new FrequencyLimit(this, taskJson);
        //疲劳
        mTaskTouchLimit = new UserFatigueControl(this, taskJson);
    }


    public boolean triggerRules(Facts facts) {
        boolean isTrigger = false;
        //通道测试跳过触发规则
        if (CommonUtils.isChannelTestTask(taskMode)) {
            isTrigger = true;
        } else {
            for (ITriggerRule triggerRule : triggerRules) {
                if (triggerRule.triggerEventRules(facts)) {
                    isTrigger = true;
                    break;
                }
            }
        }
        if (isTrigger) {
            //记录触发时间
            mTriggerRecord.triggerTime = new Date(AnalyticUtils.getCurrentTimeStamp());
        }
        return isTrigger;
    }

    public boolean isValidityPeriod() {
        if (CommonUtils.isChannelTestTask(taskMode)) return true;
        boolean isInPeriod = mTimeCycle != null && mTimeCycle.isWithInPeriod();
        if (!isInPeriod) {
            TDStrategyLog.i("Task (" + this.taskId + ") is not in the validity period");
        }
        return isInPeriod;
    }

    public boolean isTaskLimit() {
        if (mTaskFrequencyLimit != null && mTaskFrequencyLimit.isLimit()) {
            //达到频控
            mTriggerRecord.status = AnalyticUtils.FREQUENCY_CONTROL;
            AnalyticUtils.trackTriggerEvent(mTriggerRecord);
            return true;
        }
        if (mExpConfigManager != null && mExpConfigManager.isExpSampleOut()) {
            //抽样淘汰
            mTriggerRecord.status = AnalyticUtils.SAMPLE;
            AnalyticUtils.trackTriggerEvent(mTriggerRecord);
            TDStrategyLog.i("Task (" + this.taskId + ") is sampling and elimination");
            return true;
        }
        if (mTaskTouchLimit != null && mTaskTouchLimit.isLimit()) {
            //达到疲劳
            mTriggerRecord.status = AnalyticUtils.FATIGUE_CONTROL;
            AnalyticUtils.trackTriggerEvent(mTriggerRecord);
            return true;
        }
        if (mExpConfigManager != null && !mExpConfigManager.isPush()) {
            mTriggerRecord.status = AnalyticUtils.EXP_SKIP_PUSH;
            AnalyticUtils.trackTriggerEvent(mTriggerRecord);
            TDStrategyLog.i("Task (" + this.taskId + ") is skip push");
            return true;
        }
        return false;
    }

    public void startClientRecord() {
        mTriggerRecord.pushId = null;
        mTriggerRecord.expGroupId = null;
        mTriggerRecord.triggerTime = null;
        mTriggerRecord.actualPushTime = null;
        mTriggerRecord.status = -1;
        mTriggerRecord.taskMode = taskMode;
    }

    public void finishClientRecord() {
        mTriggerRecord.actualPushTime = new Date(AnalyticUtils.getCurrentTimeStamp());
        //增加频控和疲劳的次数
        if (mTaskFrequencyLimit != null) {
            mTaskFrequencyLimit.addLimitCount();
        }
        if (mTaskTouchLimit != null) {
            mTaskTouchLimit.addLimitCount();
        }
        if (TDStrategyResultObservable.getInstance().isObserverEmpty()) {
            mTriggerRecord.status = AnalyticUtils.FAIL;
        } else {
            mTriggerRecord.status = AnalyticUtils.SUCCESS;
        }
        AnalyticUtils.trackTriggerEvent(mTriggerRecord);
    }

    public boolean isTargetEvent(String eventName) {
        //白名单事件
        if (AnalyticUtils.isWhiteEvent(eventName)) return true;
        boolean isTargetEvent = false;
        for (ITriggerRule triggerRule : triggerRules) {
            if (triggerRule.isTargetEvent(eventName)) {
                isTargetEvent = true;
                break;
            }
        }
        if (isTargetEvent) {
            TDStrategyLog.i("Task (" + this.taskId + ") contains target event");
        }
        return isTargetEvent;
    }

    public boolean isEventTimeout(long enqueueTime) {
        if (enqueueTime == 0) return false;
        return SystemClock.elapsedRealtime() - enqueueTime > eventTimeout;
    }


    public boolean isTargetAudience(Facts facts) {
        if (mTargetAudienceManager == null) return false;
        boolean isTarget = mTargetAudienceManager.isTargetUser(facts);
        if (isTarget) {
            mTriggerRecord.pushId = getPushId();
        }
        return isTarget;
    }

    public String getPushId() {
        if (mTargetAudienceManager == null) return "";
        return mTargetAudienceManager.getPushId(null);
    }

    public boolean isExpRelease() {
        if (mExpConfigManager == null) return true;
        return mExpConfigManager.isExpRelease(mTriggerRecord);
    }

    public TDStrategyResult calculate(Facts facts) {
        if (mPushResultManger != null) {
            return mPushResultManger.calculate(facts);
        }
        return null;
    }

}
