/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.core.result;

import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Date;
import java.util.Iterator;
import java.util.TimeZone;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cn.thinkingdata.core.utils.TimeUtil;
import cn.thinkingdata.strategy.core.task.TaskFlow;
import cn.thinkingdata.strategy.rules.api.Facts;
import cn.thinkingdata.strategy.utils.AnalyticUtils;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.StrategyConstants;

/**
 * @author liulongbing
 * @since 2024/3/13
 */
public class PushResultManager {
    private static final String CONTENT_PATTERN = "\\$\\((.*?)\\)";
    private static final String OCCASION_PATTERN = "\\$\\[(.*?)\\]";
    private JSONObject contentJson;
    private JSONArray userParamsList;
    private final TaskFlow mFlow;

    public PushResultManager(TaskFlow flow, JSONObject taskJson) {
        this.mFlow = flow;
        try {
            JSONArray groupContentList = taskJson.optJSONArray("groupContentList");
            if (groupContentList != null && groupContentList.length() > 0) {
                JSONObject contentItem = groupContentList.optJSONObject(0);
                if (contentItem != null) {
                    contentJson = contentItem.optJSONObject("content");
                }
            }
            userParamsList = taskJson.optJSONArray("channelUserParamsList");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public TDStrategyResult calculate(Facts facts) {
        TDStrategyResult mStrategyResult = new TDStrategyResult(mFlow.appId, mFlow.taskId, mFlow.channelMsgType);
        mStrategyResult.content = buildContentMap(facts);
        mStrategyResult.pushId = mFlow.getPushId();
        mStrategyResult.userParams = buildUserParamsMap(facts);
        mStrategyResult.opsProperties = buildOpsProperties();
        return mStrategyResult;
    }

    private JSONObject buildOpsProperties() {
        JSONObject ops = new JSONObject();
        JSONObject opsProperties = new JSONObject();
        try {
            opsProperties.put("ops_project_id", mFlow.projectId);
            opsProperties.put("ops_task_id", mFlow.taskId);
            opsProperties.put("ops_exp_group_id", mFlow.mTriggerRecord.expGroupId);
            opsProperties.put("ops_trigger_time", TimeUtil.formatDate(mFlow.mTriggerRecord.triggerTime, StrategyConstants.TIME_PATTERN_1, AnalyticUtils.getAnalyticsTimeZone(mFlow.appId)));
            opsProperties.put("ops_task_exec_detail_id", UUID.randomUUID().toString());
            ops.put("#ops_receipt_properties", opsProperties);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return ops;
    }


    private JSONObject buildUserParamsMap(Facts facts) {
        JSONObject userParams = new JSONObject();
        if (userParamsList != null) {
            try {
                for (int i = 0; i < userParamsList.length(); i++) {
                    JSONObject jsonItem = userParamsList.optJSONObject(i);
                    if (jsonItem != null) {
                        String key = jsonItem.optString("key");
                        Object value = jsonItem.opt("value");
                        if (CommonUtils.isNull(value)) {
                            String defaultValue = jsonItem.optString("defaultValue");
                            String columnName = jsonItem.optString("columnName");
                            userParams.put(key, getUserParamsValue(facts, columnName, defaultValue));
                        } else {
                            userParams.put(key, value);
                        }
                    }
                }
            } catch (Exception ignore) {
            }
        }
        return userParams;
    }

    private JSONObject buildContentMap(Facts facts) {
        JSONObject contents = new JSONObject();
        if (contentJson == null) return contents;
        try {
            Iterator<String> keys = contentJson.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                Object value = contentJson.opt(key);
                if (value instanceof String) {
                    String v = getClientValue(( String ) value, facts);
                    v = getOccasionValue(v, facts);
                    if (v != null) {
                        contents.put(key, v);
                    }
                } else if (value instanceof JSONObject) {
                    JSONObject j = getOccasionJson(( JSONObject ) value, facts);
                    if (j != null) {
                        contents.put(key, j);
                    }
                } else if (value instanceof JSONArray) {
                    JSONArray newArray = new JSONArray();
                    JSONArray array = ( JSONArray ) value;
                    for (int i = 0; i < array.length(); i++) {
                        JSONObject j = getOccasionJson(array.getJSONObject(i), facts);
                        if (j != null) {
                            newArray.put(j);
                        }
                    }
                    contents.put(key, newArray);
                } else {
                    contents.put(key, value);
                }
            }
        } catch (Exception ignore) {
        }
        return contents;
    }

    private JSONObject getOccasionJson(JSONObject valueJson, Facts facts) {
        if (valueJson == null) return null;
        JSONObject newJson = new JSONObject();
        try {
            Iterator<String> vKeys = valueJson.keys();
            while (vKeys.hasNext()) {
                String k = vKeys.next();
                Object vv = valueJson.opt(k);
                if (vv instanceof String) {
                    String s = getOccasionValue(( String ) vv, facts);
                    if (s != null) {
                        newJson.put(k, s);
                    }
                } else {
                    newJson.put(k, vv);
                }
            }
        } catch (Exception ignore) {

        }
        return newJson;
    }

    private String getOccasionValue(String value, Facts facts) {
        Pattern pattern = Pattern.compile(OCCASION_PATTERN);
        Matcher matcher = pattern.matcher(value);
        StringBuffer result = new StringBuffer();
        while (matcher.find()) {
            String placeholderContent = matcher.group(1);
            String replacement = getOccasionColumnValue(facts, placeholderContent);
            matcher.appendReplacement(result, replacement);
        }
        matcher.appendTail(result);
        String str = result.toString();
        if (TextUtils.isEmpty(str.trim())) {
            return null;
        }
        return str;
    }

    private String getOccasionColumnValue(Facts facts, String placeContent) {
        if (placeContent == null) return " ";
        String[] splits = placeContent.split(":");
        if (splits.length < 3) return " ";
        String eventName = splits[1];
        if (!TextUtils.equals(eventName, facts.get(StrategyConstants.EVENT_NAME_KEY))) return "";
        String columnName = splits[2];
        if (columnName == null) return " ";
        Object value = null;
        if (columnName.contains(".")) {
            String[] columns = columnName.split("\\.");
            if (columns.length == 2) {
                Object j = facts.get(columns[0]);
                if (j instanceof JSONObject) {
                    value = (( JSONObject ) j).opt(columns[1]);
                }
            }
        } else {
            value = facts.get(columnName);
        }
        if (value == null) {
            if (splits.length == 4 && splits[3] != null) {
                value = splits[3];
            } else {
                value = " ";
            }
        }
        return value.toString();
    }


    private String getClientValue(String value, Facts facts) {
        Pattern pattern = Pattern.compile(CONTENT_PATTERN);
        Matcher matcher = pattern.matcher(value);
        StringBuffer result = new StringBuffer();
        while (matcher.find()) {
            String placeholderContent = matcher.group(1);
            String replacement = getContentColumnValue(facts, placeholderContent);
            matcher.appendReplacement(result, replacement);
        }
        matcher.appendTail(result);
        return result.toString();
    }

    private String getContentColumnValue(Facts facts, String placeContent) {
        if (placeContent == null) return "";
        String[] splits = placeContent.split(":");
        Object value = CommonUtils.getColumnValue(facts, splits);
        if (value == null) {
            if (splits.length == 3 && splits[2] != null) {
                value = splits[2];
            } else {
                value = new Object();
            }
        }
        if (value instanceof Date) {
            return TimeUtil.formatDate(( Date ) value, StrategyConstants.TIME_PATTERN, TimeZone.getDefault());
        }
        return value.toString();
    }

    private Object getUserParamsValue(Facts facts, String placeContent, String defaultValue) {
        if (placeContent == null) return defaultValue;
        String[] splits = placeContent.split(":");
        Object value = CommonUtils.getColumnValue(facts, splits);
        if (value == null) {
            value = defaultValue;
        }
        return value;
    }


}
