/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.core.period;

import static cn.thinkingdata.strategy.utils.StrategyConstants.*;

import android.util.Pair;

import org.json.JSONObject;

import cn.thinkingdata.strategy.core.task.TaskFlow;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.StrategyConstants;

/**
 * @author liulongbing
 * @since 2024/3/13
 */
public class PeriodLimitFactory {

    public static ITimeCycle createTriggerTimeCycle(JSONObject json, double zoneOffset) {
        ITimeCycle timeCycle = null;
        try {
            String periodStart = json.optString("periodStart");
            String periodEnd = json.optString("periodEnd");
            TaskPeriodInfo info = new TaskPeriodInfo();
            info.periodStart = periodStart;
            info.periodEnd = periodEnd;
            info.zoneOffset = zoneOffset;
            info.timePattern = StrategyConstants.TIME_PATTERN;
            String periodTimeSymbol = json.optString("periodTimeSymbol");
            PeriodScrollInfo pInfo = new PeriodScrollInfo();
            pInfo.timeCycleNumber = 1;
            switch (periodTimeSymbol) {
                case "TS01":
                    pInfo.timeCycleUnit = TimeCycleUnit.DAY;
                    String startTimeStr1 = json.optString("dayStartTime");
                    CommonUtils.parseStartTime(startTimeStr1, pInfo);
                    timeCycle = new DailyScrollTimeCycle(info, pInfo);
                    break;
                case "TS02":
                    timeCycle = new StartEndLimit(info);
                    break;
                case "TS03":
                    String startTimeStr2 = json.optString("dayStartTime");
                    pInfo.timeCycleUnit = TimeCycleUnit.WEEK;
                    pInfo.startNumber = json.optInt("startDay");
                    CommonUtils.parseStartTime(startTimeStr2, pInfo);
                    timeCycle = new WeekScrollTimeCycle(info, pInfo);
                    break;
                case "TS04":
                    pInfo.timeCycleUnit = TimeCycleUnit.MONTH;
                    pInfo.startNumber = json.optInt("startDay");
                    timeCycle = new MonthScrollTimeCycle(info, pInfo);
                    break;
                case "TS05":
                    timeCycle = new ClientStartTimeCycle(info);
                    break;
                default:
                    return null;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return timeCycle;
    }

    public static ITimeCycle createFrequencyLimit(TaskFlow flow, JSONObject limitRule) {
        ITimeCycle timeCycle = null;
        try {
            String ruleType = limitRule.optString("ruleType");
            if (flow.mTimeCycle == null) return null;
            Pair<Long, Long> pair = flow.mTimeCycle.getCurrentTimeCycle();
            TaskPeriodInfo info = new TaskPeriodInfo();
            info.periodStartStamp = pair.first;
            info.periodEndStamp = pair.second;
            switch (ruleType) {
                case FREQUENCY_TASK:
                    timeCycle = new StartEndLimit(info);
                    break;
                case FREQUENCY_TIME_CYCLE:
                    PeriodScrollInfo pInfo = new PeriodScrollInfo();
                    pInfo.timeCycleNumber = limitRule.optInt("timeCycleNumber");
                    pInfo.timeCycleUnit = getTimeCycleUnit(limitRule.optString("timeCycleUnit"));
                    timeCycle = new SlideTimeCycle(info, pInfo);
                    break;
                case FREQUENCY_CLIENT_START:
                    timeCycle = new ClientStartTimeCycle(info);
                    break;
                case FREQUENCY_TIME_CYCLE_ROLL:
                    PeriodScrollInfo scrollInfo = new PeriodScrollInfo();
                    scrollInfo.timeCycleNumber = limitRule.optInt("timeCycleNumber");
                    scrollInfo.timeCycleUnit = getTimeCycleUnit(limitRule.optString("timeCycleUnit"));
                    JSONObject timeCycleDef = limitRule.optJSONObject("timeCycleDef");
                    if (timeCycleDef != null) {
                        scrollInfo.startNumber = timeCycleDef.optInt("startDay");
                        String startTimeStr = timeCycleDef.optString("startTime");
                        CommonUtils.parseStartTime(startTimeStr, scrollInfo);
                    }
                    if (scrollInfo.timeCycleUnit == TimeCycleUnit.DAY) {
                        timeCycle = new DailyScrollTimeCycle(info, scrollInfo);
                    } else if (scrollInfo.timeCycleUnit == TimeCycleUnit.WEEK) {
                        timeCycle = new WeekScrollTimeCycle(info, scrollInfo);
                    } else if (scrollInfo.timeCycleUnit == TimeCycleUnit.MONTH) {
                        timeCycle = new MonthScrollTimeCycle(info, scrollInfo);
                    }
                    break;
                default:
                    break;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return timeCycle;
    }

    public static TimeCycleUnit getTimeCycleUnit(String unit) {
        switch (unit) {
            case "day":
            case "DAY":
                return TimeCycleUnit.DAY;
            case "week":
            case "WEEK":
                return TimeCycleUnit.WEEK;
            case "month":
            case "MONTH":
                return TimeCycleUnit.MONTH;
            case "hour":
            case "HOUR":
                return TimeCycleUnit.HOUR;
            case "minute":
            case "MINUTE":
                return TimeCycleUnit.MINUTE;
        }
        return null;
    }

}
