/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.core.period;

import android.util.Pair;

import java.util.Calendar;

import cn.thinkingdata.strategy.utils.AnalyticUtils;

/**
 * @author liulongbing
 * @since 2024/3/22
 */
public class DailyScrollTimeCycle extends AbstractTimeCycle {

    public DailyScrollTimeCycle(TaskPeriodInfo periodInfo, PeriodScrollInfo sInfo) {
        super(periodInfo, sInfo);
        if (scrollInfo == null) {
            scrollInfo = new PeriodScrollInfo();
            scrollInfo.timeCycleNumber = 1;
            scrollInfo.timeCycleUnit = TimeCycleUnit.DAY;
        } else {
            if (scrollInfo.timeCycleNumber < 1) {
                scrollInfo.timeCycleNumber = 1;
            }
        }
    }

    @Override
    public Pair<Long, Long> getCurrentTimeCycle() {
        long startOfDayTimestamp = getStartOfTimestamp();
        long startTimeStamp = startOfDayTimestamp + ( long ) getStartDay() * 24 * 3600 * 1000 + ( long ) scrollInfo.startHour * 3600 * 1000 + ( long ) scrollInfo.startMinute * 60 * 1000;
        long endTimeStamp = startTimeStamp + ( long ) getCycleNumber() * 24 * 3600 * 1000;
        long currentTimeStamp = AnalyticUtils.getCurrentTimeStamp();
        if (currentTimeStamp < startTimeStamp) {
            if (startTime <= startTimeStamp) {
                return new Pair<>(startTime, startTimeStamp);
            } else {
                //还未到任务开始时间
                return new Pair<>(0L, 0L);
            }
        }
        boolean isInCycle = currentTimeStamp <= endTimeStamp;
        while (!isInCycle) {
            startTimeStamp = endTimeStamp;
            int cycleNumber = getCycleNumber();
            endTimeStamp = startTimeStamp + ( long ) cycleNumber * 24 * 3600 * 1000;
            isInCycle = currentTimeStamp <= endTimeStamp;
        }
        if (startTimeStamp < startTime) {
            startTimeStamp = startTime;
        }
        if (endTimeStamp > endTime) {
            endTimeStamp = endTime;
        }
        return new Pair<>(startTimeStamp, endTimeStamp);
    }

    protected long getStartOfTimestamp() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(startTime);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTimeInMillis();
    }

    protected int getStartDay() {
        return 0;
    }

    protected int getCycleNumber() {
        return scrollInfo.timeCycleNumber;
    }

}
