/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.core.frequency;

import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import cn.thinkingdata.strategy.core.period.ITimeCycle;
import cn.thinkingdata.strategy.core.period.PeriodLimitFactory;
import cn.thinkingdata.strategy.core.task.TaskFlow;
import cn.thinkingdata.strategy.storage.db.FrequencyLimitInfo;
import cn.thinkingdata.strategy.storage.db.StrategyDataBaseHelper;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.TDStrategyLog;

/**
 * @author liulongbing
 * @since 2024/3/12
 */
public class FrequencyLimit extends AbstractTaskLimit {

    private FrequencyLimitInfo mFrequencyLimitInfo;

    public FrequencyLimit(TaskFlow mTaskFlow, JSONObject json) {
        super(mTaskFlow, json);
    }

    @Override
    void parseLimitInfo(JSONObject json) throws JSONException {
        if (json == null) return;
        String frequencyLimitStr = json.optString("frequencyLimits");
        JSONObject frequencyLimitJson = new JSONObject(frequencyLimitStr);
        enableLimit = frequencyLimitJson.optBoolean("enableFrequencyLimits");
        if (!enableLimit) return;
        JSONArray ruleList = frequencyLimitJson.optJSONArray("ruleList");
        if (null != ruleList && ruleList.length() > 0) {
            JSONObject limitRule = ruleList.optJSONObject(0);
            if (limitRule == null) return;
            int maxTimes = limitRule.optInt("maxTimes");
            ITimeCycle timeCycle = PeriodLimitFactory.createFrequencyLimit(mTaskFlow, limitRule);
            if (timeCycle == null || maxTimes <= 0) return;
            mLimits.add(new Pair<>(timeCycle, maxTimes));
        }
    }

    private FrequencyLimitInfo getFrequencyLimitInfo() {
        if (mFrequencyLimitInfo == null) {
            mFrequencyLimitInfo = new FrequencyLimitInfo();
            mFrequencyLimitInfo.appId = mTaskFlow.appId;
            mFrequencyLimitInfo.userId = mClientUserId;
            mFrequencyLimitInfo.taskId = mTaskFlow.taskId;
        }
        return mFrequencyLimitInfo;
    }

    @Override
    int getLimitCount(long start, long end) {
        FrequencyLimitInfo fInfo = getFrequencyLimitInfo();
        TDStrategyLog.i("Frequency Limit UserID: " + fInfo.userId + " Start : " + CommonUtils.getLogFormatTime(start) + " End : " + CommonUtils.getLogFormatTime(end));
        return StrategyDataBaseHelper.getHelper().getFrequencyLimitCount(fInfo, start, end);
    }

    @Override
    void addLimitStorage() {
        StrategyDataBaseHelper.getHelper().addFrequencyLimit(getFrequencyLimitInfo());
    }

}
