/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.core.exp;

import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONObject;

import cn.thinkingdata.core.utils.TimeUtil;
import cn.thinkingdata.strategy.core.trigger.entity.ClientTriggerRecord;
import cn.thinkingdata.strategy.utils.AnalyticUtils;
import cn.thinkingdata.strategy.utils.StrategyConstants;
import cn.thinkingdata.strategy.utils.TDStrategyLog;

/**
 * @author liulongbing
 * @since 2024/4/17
 */
public class ExpConfigManager {

    private boolean enableExp;
    private boolean expSampleOut;
    private int expType;
    private boolean expReleasePushStarted;
    private long expStartTimeStamp;
    private long expEndTimeStamp;
    private String expGroupId;
    private boolean isPush;
    private boolean expEntityIdExist;

    public ExpConfigManager(JSONObject taskJson, double zoneOffset) {
        JSONObject expConfigJson = taskJson.optJSONObject("expConfig");
        if (expConfigJson == null) return;
        enableExp = expConfigJson.optBoolean("enableExp");
        if (!enableExp) return;
        expSampleOut = expConfigJson.optBoolean("expSampleOut");
        expEntityIdExist = expConfigJson.optBoolean("expEntityIdExist");
        expType = expConfigJson.optInt("expType");
        expReleasePushStarted = expConfigJson.optBoolean("expReleasePushStarted");
        String expStartDate = expConfigJson.optString("expStartDate");
        String expEndDate = expConfigJson.optString("expEndDate");
        expStartTimeStamp = TimeUtil.getFormatDate(expStartDate, StrategyConstants.TIME_PATTERN, zoneOffset).getTime();
        expEndTimeStamp = TimeUtil.getFormatDate(expEndDate, StrategyConstants.TIME_PATTERN, zoneOffset).getTime();
        JSONArray groupContentList = taskJson.optJSONArray("groupContentList");
        if (groupContentList == null || groupContentList.length() < 0) return;
        JSONObject groupContent = groupContentList.optJSONObject(0);
        if (groupContent == null) return;
        expGroupId = groupContent.optString("expGroupId");
        isPush = groupContent.optBoolean("isPush");
    }

    private boolean isGapExp() {
        return !enableExp || expReleasePushStarted;
    }

    /**
     * 判断分流主体是否存在
     *
     * @return true 存在 false 不存在
     */
    private boolean isExpGroupIdExist(ClientTriggerRecord record) {
        boolean isNotExits = TextUtils.isEmpty(expGroupId) && !expEntityIdExist;
        if (isNotExits) {
            // 分流主体不存在时丢弃用户
            record.status = AnalyticUtils.WITHOUT_SPLIT_FLOW_ID;
            AnalyticUtils.trackTriggerEvent(record);
        }
        return !isNotExits;
    }

    /**
     * 实验条件是否通过 true为通过 false为不通过
     */
    public boolean isExpRelease(ClientTriggerRecord record) {
        record.expGroupId = expGroupId;
        //true 实验为开启 或者在放量中
        if (isGapExp()) return true;
        if (!isExpGroupIdExist(record)) return false;
        long currentTimeStamp = AnalyticUtils.getCurrentTimeStamp();
        if (expType == 1) {
            return currentTimeStamp >= expStartTimeStamp && currentTimeStamp <= expEndTimeStamp;
        } else if (expType == 2) {
            if (currentTimeStamp < expStartTimeStamp) {
                TDStrategyLog.i("Exp type: " + expType + " is less than start time");
                return false;
            }
            if (currentTimeStamp > expEndTimeStamp) {
                //赛马实验等待放量丢失用户
                record.status = AnalyticUtils.RACE_SKIP_PUSH;
                AnalyticUtils.trackTriggerEvent(record);
                return false;
            }
            return true;
        }
        return false;
    }

    public boolean isExpSampleOut() {
        if (isGapExp()) return false;
        return expSampleOut;
    }

    public boolean isPush() {
        if (isGapExp()) return true;
        return isPush;
    }


}
