/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.remoteconfig.user;

import android.annotation.SuppressLint;
import android.content.Context;
import android.text.TextUtils;

import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author liulongbing
 * @since 2024/3/7
 */
public class ClientUserManager {

    private final ExecutorService mPool;

    @SuppressLint("StaticFieldLeak")
    private volatile static ClientUserManager instance = null;

    private final Context mContext;

    private ClientUserManager(Context context) {
        this.mContext = context;
        mPool = new ThreadPoolExecutor(0, Integer.MAX_VALUE,
                10L, TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(), new ThreadFactory() {
            @Override
            public Thread newThread(Runnable r) {
                return new Thread(r, "TDRemoteConfig.ClientIDExecuteThread");
            }
        });
    }

    public static ClientUserManager getInstance(Context context) {
        if (instance == null) {
            synchronized (ClientUserManager.class) {
                if (instance == null) {
                    instance = new ClientUserManager(context);
                }
            }
        }
        return instance;
    }

    public String getClientUserIdSync(String appId, String accountId, String distinctId) {
        if (TextUtils.isEmpty(accountId)) {
            ClientUser user = ClientUserIdDataBaseHelper.getHelper(mContext).findUserByDistinctId(appId, distinctId);
            if (user == null) {
                String clientUserId = generateClientUserId();
                createNewUser(appId, "", distinctId, clientUserId);
                return clientUserId;
            } else {
                return user.clientUserId;
            }
        } else {
            ClientUser user = ClientUserIdDataBaseHelper.getHelper(mContext).findUserByAccountId(appId, accountId);
            ClientUser userDis = ClientUserIdDataBaseHelper.getHelper(mContext).findUserByDistinctId(appId, distinctId);
            if (user == null) {
                if (userDis == null) {
                    //生成新的
                    String clientUserId = generateClientUserId();
                    createNewUser(appId, accountId, distinctId, clientUserId);
                    return clientUserId;
                } else {
                    if (TextUtils.isEmpty(userDis.accountId)) {
                        //更新accountId
                        ClientUserIdDataBaseHelper.getHelper(mContext).updateAccountId(appId, accountId, distinctId);
                        return userDis.clientUserId;
                    } else {
                        //生成新的
                        String clientUserId = generateClientUserId();
                        createNewUser(appId, accountId, "", clientUserId);
                        return clientUserId;
                    }
                }
            } else {
                if (userDis == null) {
                    if (TextUtils.isEmpty(user.distinctId)) {
                        ClientUserIdDataBaseHelper.getHelper(mContext).updateDistinctId(appId, accountId, distinctId);
                    } else {
                        createNewUser(appId, user.accountId, distinctId, user.clientUserId);
                    }
                }
                return user.clientUserId;
            }
        }
    }

    public void getClientUserId(String appId, String accountId, String distinctId, OnGetClientUserIdCallBack callBack) {
        mPool.execute(new Runnable() {
            @Override
            public void run() {
                String clientUserId = getClientUserIdSync(appId, accountId, distinctId);
                if (callBack != null) {
                    callBack.onClientUserIdSuccess(clientUserId);
                }
            }
        });
    }

    public Future<String> getClientUserId(String appId, String accountId, String distinctId) {
        return mPool.submit(new Callable<String>() {
            @Override
            public String call() throws Exception {
                return getClientUserIdSync(appId, accountId, distinctId);
            }
        });
    }

    private void createNewUser(String appId, String accountId, String distinctId, String clientUserId) {
        ClientUser newUser = new ClientUser();
        newUser.appId = appId;
        newUser.accountId = accountId;
        newUser.distinctId = distinctId;
        newUser.clientUserId = clientUserId;
        ClientUserIdDataBaseHelper.getHelper(mContext).insertNewUser(newUser);
    }

    private String generateClientUserId() {
        return UUID.randomUUID().toString();
    }

    public interface OnGetClientUserIdCallBack {
        void onClientUserIdSuccess(String clientUserId);
    }
}
