/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.remoteconfig.task;

import android.app.Application;
import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;

import java.util.ArrayList;
import java.util.List;

import cn.thinkingdata.remoteconfig.TDRemoteConfigSettings;
import cn.thinkingdata.remoteconfig.core.TDRemoteInstance;
import cn.thinkingdata.remoteconfig.core.TDTemplateSetting;
import cn.thinkingdata.remoteconfig.receiver.RemoteConfigLifecycleCallBack;
import cn.thinkingdata.remoteconfig.receiver.TDRemoteConfigObservable;
import cn.thinkingdata.remoteconfig.utils.RemoteConfigUtil;

/**
 * @author liulongbing
 * @since 2024/5/11
 */
public class RemoteTimerTask {

    private static RemoteTimerTask instance = null;

    private static final String THREAD_TIMER_WORKER = "remote.sdk.timerWorker";
    private static final int MSG_TIMER_TASK_NORMAL = 1;
    private static final int MSG_TIMER_TASK_DEBUG = 2;
    private static final int MSG_FETCH_CONTROL_DELAY = 3;

    private static final int TASK_INTERVAL_NORMAL = 30000;
    private static final int TASK_INTERVAL_DEBUG = 5000;
    private static final int FETCH_DELAY_INTERVAL = 60000;

    private final Handler mHandler;
    private final OnTimerTaskListener mTimerTaskListener;

    public final List<TimerTaskInfo> normalTasks;
    public final List<TimerTaskInfo> debugTasks;

    private boolean hasRegisterActivityLifeCycle = false;

    public static RemoteTimerTask getInstance() {
        if (instance == null) {
            instance = new RemoteTimerTask();
        }
        return instance;
    }

    public RemoteTimerTask() {
        this.mTimerTaskListener = new TaskTimerListener();
        normalTasks = new ArrayList<>();
        debugTasks = new ArrayList<>();
        final HandlerThread workerThread =
                new HandlerThread(THREAD_TIMER_WORKER,
                        Thread.MIN_PRIORITY);
        workerThread.start();
        mHandler = new TimerHandler(workerThread.getLooper());
    }

    public void registerActivityLifeCycle(Context context) {
        if (!this.hasRegisterActivityLifeCycle) {
            Application app = ( Application ) context.getApplicationContext();
            app.registerActivityLifecycleCallbacks(new RemoteConfigLifecycleCallBack());
            this.hasRegisterActivityLifeCycle = true;
        }
    }

    public void addNewConfigs(TDRemoteConfigSettings config) {
        TimerTaskInfo info = new TimerTaskInfo(config.appId, config.templateCode);
        if (config.mode == TDRemoteConfigSettings.TDRemoteConfigMode.DEBUG) {
            info.isDebug = true;
            debugTasks.add(info);
            startDebugTimerTask();
        } else {
            info.isDebug = false;
            normalTasks.add(info);
            startNormalTimerTask();
        }
    }

    public void startNormalTimerTask() {
        if (!mHandler.hasMessages(MSG_TIMER_TASK_NORMAL)) {
            mHandler.sendEmptyMessageDelayed(MSG_TIMER_TASK_NORMAL, TASK_INTERVAL_NORMAL);
        }
    }

    public void startDebugTimerTask() {
        if (!mHandler.hasMessages(MSG_TIMER_TASK_DEBUG)) {
            mHandler.sendEmptyMessageDelayed(MSG_TIMER_TASK_DEBUG, TASK_INTERVAL_DEBUG);
        }
    }

    public void sendFetchControlDelayTask(TimerTaskInfo info) {
        Message msg = Message.obtain();
        msg.what = MSG_FETCH_CONTROL_DELAY;
        msg.obj = info;
        mHandler.sendMessageDelayed(msg, FETCH_DELAY_INTERVAL);
    }

    public void removeDelayTemplateMessage() {
        mHandler.removeMessages(MSG_FETCH_CONTROL_DELAY);
    }

    private class TimerHandler extends Handler {
        public TimerHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            switch (msg.what) {
                case MSG_TIMER_TASK_NORMAL:
                    if (mTimerTaskListener != null) {
                        mTimerTaskListener.onTimerTaskNormal();
                    }
                    startNormalTimerTask();
                    break;
                case MSG_TIMER_TASK_DEBUG:
                    if (mTimerTaskListener != null) {
                        mTimerTaskListener.onTimerTaskDebug();
                    }
                    startDebugTimerTask();
                    break;
                case MSG_FETCH_CONTROL_DELAY:
                    TimerTaskInfo info = ( TimerTaskInfo ) msg.obj;
                    if (mTimerTaskListener != null) {
                        mTimerTaskListener.onControlDelay(info);
                    }
                    break;
                default:
                    break;
            }
        }
    }

    public void removeSystemErrorTask(String appId, String tempCode) {
        TimerTaskInfo info = new TimerTaskInfo(appId, tempCode);
        normalTasks.remove(info);
        debugTasks.remove(info);
    }

    public void debugTaskToNormal(TimerTaskInfo info) {
        debugTasks.remove(info);
        if (debugTasks.size() == 0) {
            //如果没有debug的任务 就不需要一直循环发消息了
            mHandler.removeMessages(MSG_TIMER_TASK_DEBUG);
        }
        if (!normalTasks.contains(info)) {
            info.isDebug = false;
            normalTasks.add(info);
            startNormalTimerTask();
        }
    }

    public interface OnTimerTaskListener {

        void onTimerTaskNormal();

        void onTimerTaskDebug();

        void onControlDelay(TimerTaskInfo info);

    }

    public void handlePageCycleUpdate() {
        RemoteTaskManager.getInstance().addTask(new Runnable() {
            @Override
            public void run() {
                List<TimerTaskInfo> list = RemoteConfigUtil.copyNewTimerTasks(normalTasks);
                for (TimerTaskInfo timerTaskInfo : list) {
                    TDRemoteInstance instance = TDRemoteInstance.getInstance(timerTaskInfo.appId);
                    if (instance == null) continue;
                    TDTemplateSetting templateSetting = instance.getTemplateSetting(timerTaskInfo.tempCode);
                    if (templateSetting == null) continue;
                    if (templateSetting.isFetchExpired()) {
                        instance.fetchWithTempCode(timerTaskInfo.tempCode, 1, true);
                    }
                }
            }
        });
    }

    private void handleTimerTasks(int type) {
        RemoteTaskManager.getInstance().addTask(new Runnable() {
            @Override
            public void run() {
                List<TimerTaskInfo> list;
                //深拷贝一份 如果遇到降级 会修改任务列表 导致闪退
                if (type == MSG_TIMER_TASK_NORMAL) {
                    list = RemoteConfigUtil.copyNewTimerTasks(normalTasks);
                } else {
                    list = RemoteConfigUtil.copyNewTimerTasks(debugTasks);
                }
                for (TimerTaskInfo timerTaskInfo : list) {
                    TDRemoteInstance instance = TDRemoteInstance.getInstance(timerTaskInfo.appId);
                    if (instance == null) continue;
                    TDTemplateSetting templateSetting = instance.getTemplateSetting(timerTaskInfo.tempCode);
                    if (templateSetting == null) continue;
                    boolean isExpired = templateSetting.isExpired();
                    if (isExpired) {
                        TDRemoteConfigObservable.getInstance().notifyOnConfigExpired(timerTaskInfo.appId, timerTaskInfo.tempCode, templateSetting.getClientUserId());
                    }
                    if (isExpired || timerTaskInfo.isDebug) {
                        instance.fetchWithTempCode(timerTaskInfo.tempCode, 1, false);
                    }
                }
            }
        });
    }

    public class TaskTimerListener implements RemoteTimerTask.OnTimerTaskListener {
        @Override
        public void onTimerTaskNormal() {
            handleTimerTasks(MSG_TIMER_TASK_NORMAL);
        }

        @Override
        public void onTimerTaskDebug() {
            handleTimerTasks(MSG_TIMER_TASK_DEBUG);
        }

        @Override
        public void onControlDelay(TimerTaskInfo info) {
            RemoteTaskManager.getInstance().addTask(new Runnable() {
                @Override
                public void run() {
                    if (info == null) return;
                    TDRemoteInstance instance = TDRemoteInstance.getInstance(info.appId);
                    if (instance == null) return;
                    instance.fetchWithTempCode(info.tempCode, 1, true);
                }
            });

        }
    }

}
