/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.remoteconfig.core;

import android.content.Context;
import android.text.TextUtils;
import android.util.Pair;

import org.json.JSONException;
import org.json.JSONObject;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.thinkingdata.remoteconfig.TDRemoteConfig;
import cn.thinkingdata.remoteconfig.TDRemoteConfigSettings;
import cn.thinkingdata.remoteconfig.model.TDNetModel;
import cn.thinkingdata.remoteconfig.receiver.TDRemoteConfigObservable;
import cn.thinkingdata.remoteconfig.sp.LocalStorageType;
import cn.thinkingdata.remoteconfig.sp.RemoteConfigStoragePlugin;
import cn.thinkingdata.remoteconfig.task.RemoteTimerTask;
import cn.thinkingdata.remoteconfig.task.TimerTaskInfo;
import cn.thinkingdata.remoteconfig.utils.AnalyticUtil;
import cn.thinkingdata.remoteconfig.utils.JsonUtil;
import cn.thinkingdata.remoteconfig.utils.TDRemoteConfigConstants;
import cn.thinkingdata.remoteconfig.utils.LogUtil;

/**
 * @author liulongbing
 * @since 2024/5/9
 */
public class TDTemplateSetting {

    private static boolean hasSendDebugEvent = false;

    private final TDRemoteConfigSettings mConfigSettings;
    private String serverUrl;

    private final Object mStoragePluginObj = new Object();
    private volatile RemoteConfigStoragePlugin mStoragePlugin;

    private final Map<String, String> customParams = new HashMap<>();

    private JSONObject defaultParams;

    public long expiration_time;
    private final Object applyConfigsObj = new JSONObject();
    private JSONObject applyConfigs;

    public String activationStrategy;
    public String mConfigMd5;
    private int requestControlCount = 0;
    private long lastFetchTime;
    public long lastFetchStamp = 0;
    private final Context mContext;
    private final TDAppSetting mAppSetting;
    private final List<TDRemoteConfig.OnConfigFetchListener> mFetchListener;

    public TDTemplateSetting(Context context, TDRemoteConfigSettings settings, TDAppSetting appSetting) {
        this.mContext = context;
        this.mConfigSettings = settings;
        this.mAppSetting = appSetting;
        this.mFetchListener = new ArrayList<>();
    }

    public String getTempCode() {
        if (TextUtils.equals(mConfigSettings.templateCode, TDRemoteConfigConstants.TEMPLATE_DEFAULT)) {
            return "";
        }
        return mConfigSettings.templateCode;
    }

    public boolean isDebug() {
        return mConfigSettings.mode == TDRemoteConfigSettings.TDRemoteConfigMode.DEBUG;
    }

    public void initTemplateInfoAsync() {
        parseServerUrl();
        mergeCustomParams(mConfigSettings.getCustomFetchParams());
        checkTemplateStoragePluginSafe();
        this.lastFetchStamp = mStoragePlugin.get(LocalStorageType.LAST_FETCH_TIME);
        synchronized (applyConfigsObj) {
            loadDefaultAndApplyParams();
        }
    }

    public String getServerUrl() {
        return this.serverUrl;
    }

    private void checkTemplateStoragePluginSafe() {
        if (mStoragePlugin == null) {
            synchronized (mStoragePluginObj) {
                if (mStoragePlugin == null) {
                    mStoragePlugin = new RemoteConfigStoragePlugin(mContext, mConfigSettings.appId, mConfigSettings.templateCode);
                }
            }
        }
    }

    public void parseServerUrl() {
        try {
            URL url = new URL(mConfigSettings.serverUrl);
            this.serverUrl = url.getProtocol()
                    + "://" + url.getHost()
                    + (url.getPort() > 0 ? ":" + url.getPort() : "");
            if (mConfigSettings.mode == TDRemoteConfigSettings.TDRemoteConfigMode.DEBUG) {
                this.serverUrl = this.serverUrl + "/v1/remote-config/debug";
            } else {
                this.serverUrl = this.serverUrl + "/v1/remote-config/get";
            }
        } catch (MalformedURLException e) {
            e.printStackTrace();
        }
    }

    public void mergeCustomParams(JSONObject source) {
        if (null == source) return;
        try {
            JsonUtil.mergeJsonToMap(source, this.customParams);
            LogUtil.i1("custom fetch params result is %s", this.customParams);
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public void removeCustomParams(String key) {
        if (null == key) return;
        this.customParams.remove(key);
    }

    public Map<String, String> getCustomParams() {
        return this.customParams;
    }


    private void loadDefaultAndApplyParams() {
        if (defaultParams == null) {
            defaultParams = new JSONObject();
            String defaultConfig = mStoragePlugin.get(LocalStorageType.DEFAULT_CONFIG);
            try {
                JSONObject defaultJson = new JSONObject(defaultConfig);
                JSONObject configJson = defaultJson.optJSONObject("config");
                JSONObject typeJson = defaultJson.optJSONObject("type");
                JsonUtil.parseJsonConfig(configJson, typeJson);
                if (configJson != null) {
                    JsonUtil.mergeJSONObject(configJson, defaultParams);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        if (applyConfigs == null) {
            applyConfigs = new JSONObject();
            String lastFetch = mStoragePlugin.get(LocalStorageType.LAST_FETCH_CONFIG);
            try {
                JSONObject lastFetchJson = new JSONObject(lastFetch);
                JSONObject configJson = lastFetchJson.optJSONObject("config");
                JSONObject typeJson = lastFetchJson.optJSONObject("type");
                expiration_time = lastFetchJson.optLong("expirationPeriod");
                JsonUtil.parseJsonConfig(configJson, typeJson);
                if (null != configJson) {
                    JsonUtil.mergeJSONObject(configJson, applyConfigs);
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public void addConfigFetchListener(TDRemoteConfig.OnConfigFetchListener listener) {
        if (null != listener) {
            synchronized (mFetchListener) {
                mFetchListener.add(listener);
            }
        }
    }

    public void mergeDefaultParams(JSONObject source) {
        checkTemplateStoragePluginSafe();
        try {
            synchronized (applyConfigsObj) {
                loadDefaultAndApplyParams();
                JsonUtil.mergeJSONObject(source, defaultParams);
                LogUtil.i1("set default params success , final result is %s", this.defaultParams.toString(4));
                JSONObject newDefault = JsonUtil.parseConfigToJson(defaultParams);
                mStoragePlugin.save(LocalStorageType.DEFAULT_CONFIG, newDefault.toString());
            }
            mAppSetting.updateLastModifyFile();
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public void clearDefaultParams() {
        checkTemplateStoragePluginSafe();
        try {
            synchronized (applyConfigsObj) {
                loadDefaultAndApplyParams();
                defaultParams = new JSONObject();
                LogUtil.i("clear default params success");
                mStoragePlugin.save(LocalStorageType.DEFAULT_CONFIG, "{}");
            }
            mAppSetting.updateLastModifyFile();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void updateRemoteConfigs(TDNetModel model) {
        if (null != model.data) {
            updateLastFetchTime();
            this.lastFetchStamp = model.timeStamp;
            mStoragePlugin.save(LocalStorageType.LAST_FETCH_TIME, model.timeStamp);
            JSONObject configJson = model.data.optJSONObject("config");
            JSONObject typeJson = model.data.optJSONObject("type");
            JSONObject templateInfoJson = model.data.optJSONObject("templateInfo");
            long expiration_period = 0L;
            if (templateInfoJson != null) {
                activationStrategy = templateInfoJson.optString("activationStrategy");
                expiration_period = templateInfoJson.optLong("expirationPeriod");
            }
            if (expiration_period <= 0) {
                expiration_period = TDRemoteConfigConstants.CONFIG_EXPIRED_TIME;
            }
            expiration_time = System.currentTimeMillis() / 1000 + expiration_period;
            //重启生效
            JSONObject newJson = new JSONObject();
            try {
                newJson.put("config", configJson);
                newJson.put("type", typeJson);
                newJson.put("expirationPeriod", expiration_time);
                mStoragePlugin.save(LocalStorageType.LAST_FETCH_CONFIG, newJson.toString());
                mAppSetting.updateLastModifyFile();
            } catch (JSONException e) {
                e.printStackTrace();
            }
            if (isDebug() && TextUtils.equals(mConfigSettings.templateCode, TDRemoteConfigConstants.TEMPLATE_DEFAULT)) {
                String cmd5 = "";
                if (templateInfoJson != null) {
                    cmd5 = templateInfoJson.optString("md5");
                }
                if (!TextUtils.equals(mConfigMd5, cmd5)) {
                    AnalyticUtil.trackClientDebugEvent(mConfigSettings.appId, configJson);
                    mConfigMd5 = cmd5;
                }
            }
            if (TextUtils.equals(activationStrategy, TDRemoteConfigConstants.ACTIVATION_STRATEGY_IMMEDIATELY)) {
                //立刻生效
                JsonUtil.parseJsonConfig(configJson, typeJson);
                synchronized (applyConfigsObj) {
                    this.applyConfigs = new JSONObject();
                    try {
                        if (null != configJson) {
                            JsonUtil.mergeJSONObject(configJson, applyConfigs);
                        }
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }
                TDRemoteConfigObservable.getInstance().notifyOnConfigApplySuccess(mConfigSettings.appId, mConfigSettings.templateCode, getClientUserId());
                JSONObject statusJson = new JSONObject();
                if (configJson != null) {
                    try {
                        statusJson.put("strategy_status_map", configJson.optJSONObject("#strategy_status_map"));
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }
                synchronized (mFetchListener) {
                    for (TDRemoteConfig.OnConfigFetchListener onConfigFetchListener : this.mFetchListener) {
                        onConfigFetchListener.onFetchSuccess(statusJson);
                    }
                }
            }
        }
    }

    public String getClientUserId() {
        return mAppSetting.getClientUserId();
    }

    public Pair<JSONObject, JSONObject> getAll() {
        checkTemplateStoragePluginSafe();
        JSONObject first = new JSONObject();
        JSONObject second = new JSONObject();
        Pair<JSONObject, JSONObject> pair = new Pair<>(first, second);
        try {
            synchronized (applyConfigsObj) {
                loadDefaultAndApplyParams();
                if (isStrategyNoExpired()) {
                    JsonUtil.mergeJSONObject(applyConfigs, first);
                }
                JsonUtil.mergeJSONObject(defaultParams, second);
            }
        } catch (Exception ignore) {
        }
        return pair;
    }

    private boolean isStrategyNoExpired() {
        if (TextUtils.equals(activationStrategy, TDRemoteConfigConstants.ACTIVATION_STRATEGY_NEXT)) {
            return true;
        }
        return !isExpired();
    }

    public boolean isExpired() {
        boolean isExpired = System.currentTimeMillis() / 1000 > this.expiration_time;
        if (isExpired) {
            LogUtil.i("remote config is expired");
        }
        return this.expiration_time == 0L || isExpired;
    }

    public boolean isFetchExpired() {
        return AnalyticUtil.getCurrentTimeStamp() - this.lastFetchStamp > 6 * 3600 * 1000;
    }

    public void clearOldConfig() {
        synchronized (applyConfigsObj) {
            this.applyConfigs = new JSONObject();
        }
        mStoragePlugin.save(LocalStorageType.LAST_FETCH_CONFIG, "{}");
        mAppSetting.updateLastModifyFile();
    }

    public void addRequestCount() {
        if (isDebug()) return;
        requestControlCount++;
    }

    public boolean isRequestControl() {
        if (isDebug()) return false;
        long currentTimeMinute = System.currentTimeMillis() / 1000 / 60;
        if (currentTimeMinute == this.lastFetchTime) {
            boolean isControl = requestControlCount > 3;
            if (isControl) {
                LogUtil.i("fetch config too frequently , try again 60s later");
                RemoteTimerTask.getInstance().sendFetchControlDelayTask(new TimerTaskInfo(mConfigSettings.appId, mConfigSettings.templateCode));
            }
            return isControl;
        }
        return false;
    }

    private void updateLastFetchTime() {
        if (isDebug()) return;
        long currentTimeMinute = System.currentTimeMillis() / 1000 / 60;
        if (currentTimeMinute != this.lastFetchTime) {
            this.requestControlCount = 1;
        }
        this.lastFetchTime = currentTimeMinute;
    }

    /**
     * 降级策略
     */
    public void downGradeMode() {
        if (mConfigSettings.mode == TDRemoteConfigSettings.TDRemoteConfigMode.DEBUG) {
            mConfigSettings.mode = TDRemoteConfigSettings.TDRemoteConfigMode.NORMAL;
            //降级之后重新解析url
            parseServerUrl();
            RemoteTimerTask.getInstance().debugTaskToNormal(new TimerTaskInfo(mConfigSettings.appId, mConfigSettings.templateCode));
        }
    }

    public void uploadDebugEvent(int code) {
        if (!isDebug()) return;
        if (hasSendDebugEvent) return;
        hasSendDebugEvent = true;
        AnalyticUtil.trackDebugFailEvent(mConfigSettings.appId, code);
    }

}
