/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.remoteconfig.user;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.text.TextUtils;

import java.io.File;

import cn.thinkingdata.remoteconfig.utils.AnalyticUtil;
import cn.thinkingdata.remoteconfig.utils.TDRemoteConfigConstants;
import cn.thinkingdata.remoteconfig.utils.LogUtil;

/**
 * @author liulongbing
 * @since 2024/3/6
 */
public class ClientUserIdDataBaseHelper extends SQLiteOpenHelper {

    private static final String DB_NAME = "td_remote_config.db";
    private static final int DB_VERSION = 1;
    private static final String TABLE_NAME = "t_users";
    public static final String KEY_APP_ID = "app_id";
    public static final String KEY_ACCOUNT_ID = "account_id";
    public static final String KEY_DISTINCT_ID = "distinct_id";
    public static final String KEY_CLIENT_USER_ID = "client_uid";
    public static final String KEY_UPDATE_TIME = "update_time";
    private static final String CREATE_USER_TABLE = String.format("CREATE TABLE %s (id INTEGER PRIMARY KEY AUTOINCREMENT, %s TEXT NOT NULL,%s TEXT NULL,%s TEXT NULL, %s TEXT NULL, %s INTEGER NULL)", TABLE_NAME, KEY_APP_ID, KEY_ACCOUNT_ID, KEY_DISTINCT_ID, KEY_CLIENT_USER_ID, KEY_UPDATE_TIME);

    private static File dbFile;
    private static ClientUserIdDataBaseHelper mHelper;

    public static ClientUserIdDataBaseHelper getHelper(Context context) {
        if (mHelper == null || !isDatabaseExists(dbFile)) {
            mHelper = new ClientUserIdDataBaseHelper(context);
        }
        return mHelper;
    }

    public ClientUserIdDataBaseHelper(Context context) {
        super(context, DB_NAME, null, DB_VERSION);
        dbFile = context.getDatabasePath(DB_NAME);
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        db.execSQL(CREATE_USER_TABLE);
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {

    }

    public void insertNewUser(ClientUser user) {
        try {
            ContentValues values = new ContentValues();
            values.put(KEY_APP_ID, user.appId);
            if (!TextUtils.isEmpty(user.accountId)) {
                values.put(KEY_ACCOUNT_ID, user.accountId);
            }
            if (!TextUtils.isEmpty(user.distinctId)) {
                values.put(KEY_DISTINCT_ID, user.distinctId);
            }
            if (!TextUtils.isEmpty(user.clientUserId)) {
                values.put(KEY_CLIENT_USER_ID, user.clientUserId);
            }
            values.put(KEY_UPDATE_TIME, AnalyticUtil.getCurrentTimeStamp());
            SQLiteDatabase db = getWritableDatabase();
            db.insert(TABLE_NAME, null, values);
        } catch (Exception e) {
            LogUtil.e(e.getMessage());
        }
    }

    public ClientUser findUserByAccountId(String appId, String accountId) {
        ClientUser user = null;
        Cursor cursor = null;
        try {
            SQLiteDatabase db = getReadableDatabase();
            cursor = db.query(TABLE_NAME, null, KEY_APP_ID + "=? AND " + KEY_ACCOUNT_ID + "=?", new String[]{appId, accountId}, null, null, null);
            user = ClientUser.parseCursor(cursor);
        } catch (Exception e) {
            LogUtil.e(e.getMessage());
        } finally {
            if (null != cursor) {
                cursor.close();
            }
        }
        return user;
    }

    public ClientUser findUserByDistinctId(String appId, String distinctId) {
        ClientUser user = null;
        Cursor cursor = null;
        try {
            SQLiteDatabase db = getReadableDatabase();
            cursor = db.query(TABLE_NAME, null, KEY_APP_ID + "=? AND " + KEY_DISTINCT_ID + "=?", new String[]{appId, distinctId}, null, null, null);
            user = ClientUser.parseCursor(cursor);
        } catch (Exception e) {
            LogUtil.e(e.getMessage());
        } finally {
            if (null != cursor) {
                cursor.close();
            }
        }
        return user;
    }

    public void updateAccountId(String appId, String accountId, String distinctId) {
        try {
            SQLiteDatabase db = getWritableDatabase();
            ContentValues values = new ContentValues();
            values.put(KEY_ACCOUNT_ID, accountId);
            db.update(TABLE_NAME, values, KEY_APP_ID + "=? AND " + KEY_DISTINCT_ID + "=?", new String[]{appId, distinctId});
        } catch (Exception e) {
            LogUtil.e(e.getMessage());
        }
    }

    public void updateDistinctId(String appId, String accountId, String distinctId) {
        try {
            SQLiteDatabase db = getWritableDatabase();
            ContentValues values = new ContentValues();
            values.put(KEY_DISTINCT_ID, distinctId);
            db.update(TABLE_NAME, values, KEY_APP_ID + "=? AND " + KEY_ACCOUNT_ID + "=?", new String[]{appId, accountId});
        } catch (Exception e) {
            LogUtil.e(e.getMessage());
        }
    }

    private static boolean isDatabaseExists(File file) {
        if (file == null) return true;
        return file.exists();
    }

    public void deleteAccountByLimit() {
        try {
            SQLiteDatabase dbQuery = getReadableDatabase();
            Cursor cursor = dbQuery.rawQuery("SELECT COUNT(*) FROM " + TABLE_NAME, null);
            cursor.moveToFirst();
            long total = cursor.getInt(0);
            cursor.close();
            if (total > TDRemoteConfigConstants.MAX_ACCOUNT_LIMIT) {
                SQLiteDatabase db = getWritableDatabase();
                long deleteCount = total - TDRemoteConfigConstants.DELETE_ACCOUNT_LIMIT;
                String subQuery = "(SELECT id FROM " + TABLE_NAME + " ORDER BY " + KEY_UPDATE_TIME + " ASC LIMIT " + deleteCount + ")";
                db.delete(TABLE_NAME, "id IN " + subQuery, null);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


}
