/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.remoteconfig.core;

import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * @author liulongbing
 * @since 2024/9/25
 */
public class TDObject {

    private Object configValues;

    private JSONObject defaultValues;

    private final List<Object> keyList;

    public TDObject(Pair<JSONObject, JSONObject> configPair) {
        keyList = new ArrayList<>();
        if (configPair != null) {
            this.configValues = configPair.first;
            this.defaultValues = configPair.second;
        }
    }

    public TDObject(Object values) {
        keyList = new ArrayList<>();
        this.configValues = values;
        this.defaultValues = null;
    }

    public TDObject get(String key, Object defaultValue) {
        if (this.configValues == null) return this;
        keyList.add(key);
        if (this.configValues instanceof JSONObject) {
            this.configValues = (( JSONObject ) this.configValues).opt(key);
        }
        if (this.configValues == null && keyList.size() <= 2) {
            this.configValues = getDefaultValue();
            this.defaultValues = null;
        }
        if (this.configValues == null) {
            this.configValues = defaultValue;
        }
        return this;
    }

    private Object getDefaultValue() {
        if (this.defaultValues == null) return null;
        Object defaultValue = this.defaultValues;
        for (int i = 0; i < this.keyList.size(); i++) {
            Object k = this.keyList.get(i);
            if (k instanceof String) {
                String key = ( String ) k;
                if (defaultValue instanceof JSONObject) {
                    defaultValue = (( JSONObject ) defaultValue).opt(key);
                } else {
                    defaultValue = null;
                }
            } else if (k instanceof Integer) {
                int index = ( int ) k;
                if (defaultValue instanceof JSONArray) {
                    defaultValue = (( JSONArray ) defaultValue).opt(index);
                } else {
                    defaultValue = null;
                }
            }
            if (defaultValue == null) {
                break;
            }
        }
        return defaultValue;
    }


    public TDObject get(String key) {
        return get(key, null);
    }

    public TDObject get(int index, Object defaultValue) {
        if (this.configValues == null) return this;
        if (keyList.size() < 2) {
            keyList.add(index);
        }
        if (this.configValues instanceof JSONArray) {
            this.configValues = (( JSONArray ) this.configValues).opt(index);
        }
        if (this.configValues == null && keyList.size() <= 2) {
            this.configValues = getDefaultValue();
            this.defaultValues = null;
        }
        if (this.configValues == null) {
            this.configValues = defaultValue;
        }
        return this;
    }

    public TDObject get(int index) {
        return get(index, null);
    }

    public TDObject arrayWithFilter(IFilter filter) {
        if (this.configValues == null || filter == null) return this;
        if (this.configValues instanceof JSONArray) {
            JSONArray newJsonArray = new JSONArray();
            JSONArray valueArray = ( JSONArray ) this.configValues;
            for (int i = 0; i < valueArray.length(); i++) {
                Object obj = valueArray.opt(i);
                TDObject tdObject = new TDObject(obj);
                if (filter.filter(tdObject)) {
                    newJsonArray.put(obj);
                }
            }
            this.configValues = newJsonArray;
        } else {
            this.configValues = null;
        }
        return this;
    }

    public String stringValue() {
        if (this.configValues == null) return null;
        if (this.configValues instanceof String) {
            return ( String ) this.configValues;
        } else {
            return this.configValues.toString();
        }
    }

    public Long longValue() {
        try {
            if (this.configValues instanceof Integer) {
                return Long.valueOf(( Integer ) this.configValues);
            } else if (this.configValues instanceof Long) {
                return ( Long ) this.configValues;
            } else if (this.configValues instanceof String) {
                return Long.valueOf(this.configValues.toString());
            }
        } catch (Exception ignore) {
        }
        return null;
    }

    public Double doubleValue() {
        try {
            if (this.configValues instanceof Integer) {
                return (( Integer ) this.configValues).doubleValue();
            } else if (this.configValues instanceof Long) {
                return (( Long ) this.configValues).doubleValue();
            } else if (this.configValues instanceof Double) {
                return ( Double ) this.configValues;
            } else if (this.configValues instanceof String) {
                return Double.valueOf(this.configValues.toString());
            }
        } catch (Exception ignore) {
        }
        return null;
    }

    public Boolean booleanValue() {
        try {
            if (this.configValues instanceof Boolean) {
                return ( Boolean ) this.configValues;
            } else if (this.configValues instanceof String) {
                return Boolean.parseBoolean(this.configValues.toString());
            }
        } catch (Exception ignore) {

        }
        return null;
    }

    public JSONObject objectValue() {
        try {
            if (this.configValues instanceof JSONObject) {
                return ( JSONObject ) this.configValues;
            } else if (this.configValues instanceof String) {
                return new JSONObject(this.configValues.toString());
            }
        } catch (Exception ignore) {
        }
        return null;
    }

    public JSONArray arrayValue() {
        try {
            if (this.configValues instanceof JSONArray) {
                return ( JSONArray ) this.configValues;
            } else if (this.configValues instanceof String) {
                return new JSONArray(this.configValues.toString());
            }
        } catch (Exception ignore) {
        }
        return null;
    }

    public String configValue() {
        if (this.configValues == null) {
            return "";
        }
        return this.configValues.toString();
    }

    public interface IFilter {
        boolean filter(TDObject obj);
    }

}
