/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.remoteconfig.core;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;

import org.json.JSONObject;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import cn.thinkingdata.core.router.provider.IAnalyticsProvider;
import cn.thinkingdata.remoteconfig.TDRemoteConfigSettings;
import cn.thinkingdata.remoteconfig.model.TDNetModel;
import cn.thinkingdata.remoteconfig.sp.AppStoragePlugin;
import cn.thinkingdata.remoteconfig.sp.LocalStorageType;
import cn.thinkingdata.remoteconfig.sp.RemoteConfigStoragePlugin;
import cn.thinkingdata.remoteconfig.sysconfig.SystemConfigManager;
import cn.thinkingdata.remoteconfig.task.RemoteTaskManager;
import cn.thinkingdata.remoteconfig.user.ClientUserIdDataBaseHelper;
import cn.thinkingdata.remoteconfig.user.ClientUserManager;
import cn.thinkingdata.remoteconfig.utils.AnalyticUtil;
import cn.thinkingdata.remoteconfig.utils.RemoteConfigUtil;
import cn.thinkingdata.remoteconfig.utils.TDRemoteConfigConstants;
import cn.thinkingdata.remoteconfig.utils.LogUtil;

/**
 * @author liulongbing
 * @since 2024/5/9
 */
public class TDAppSetting {

    private AppStoragePlugin mAppStoragePlugin;
    private final Map<String, TDTemplateSetting> templateConfigSettings = new HashMap<>();
    private final String appId;
    private final Context mContext;
    private String currentClientUserId;
    private boolean hasInitApp = false;

    public TDAppSetting(Context context, TDRemoteConfigSettings settings) {
        this.appId = settings.appId;
        this.mContext = context;
        addTemplateCode(settings);
    }

    public void initAppInfoAsync(String tempCode) {
        if (!hasInitApp) {
            mAppStoragePlugin = new AppStoragePlugin(mContext, appId);
            Future<String> userIdFuture = ClientUserManager.getInstance(mContext).getClientUserId(this.appId, getAccountId(), getDistinctId());
            try {
                currentClientUserId = userIdFuture.get(3, TimeUnit.SECONDS);
            } catch (Exception ignore) {
            }
            try {
                File sharedPrefsDir = new File(mContext.getApplicationInfo().dataDir, "shared_prefs");
                File spFile = new File(sharedPrefsDir, RemoteConfigStoragePlugin.PREFERENCE_NAME + this.appId + ".xml");
                String lastModify = mAppStoragePlugin.get(LocalStorageType.LAST_MODIFY_FILE);
                if (spFile.exists() && !TextUtils.isEmpty(lastModify)) {
                    if (!TextUtils.equals(lastModify, RemoteConfigUtil.MD5(spFile.lastModified() + ""))) {
                        //发送风控事件
                        AnalyticUtil.trackRiskEvent(this.appId, "2");
                    }
                }
            } catch (Exception ignore) {
            }
        }
        ClientUserIdDataBaseHelper.getHelper(mContext).deleteAccountByLimit();
        TDTemplateSetting templateSetting = getTemplateSetting(tempCode);
        if (templateSetting != null) {
            templateSetting.initTemplateInfoAsync();
        }
        hasInitApp = true;
    }

    public void updateLastModifyFile() {
        RemoteTaskManager.getInstance().addTask(new Runnable() {
            @Override
            public void run() {
                try {
                    Thread.sleep(200);
                    File sharedPrefsDir = new File(mContext.getApplicationInfo().dataDir, "shared_prefs");
                    File spFile = new File(sharedPrefsDir, RemoteConfigStoragePlugin.PREFERENCE_NAME + appId + ".xml");
                    if (spFile.exists()) {
                        mAppStoragePlugin.save(LocalStorageType.LAST_MODIFY_FILE, RemoteConfigUtil.MD5(spFile.lastModified() + ""));
                    }
                } catch (Exception ignore) {
                }
            }
        });
    }

    public String getClientUserId() {
        return currentClientUserId;
    }

    public boolean addTemplateCode(TDRemoteConfigSettings settings) {
        synchronized (templateConfigSettings) {
            if (templateConfigSettings.containsKey(settings.templateCode)) return false;
            TDTemplateSetting tempSetting = new TDTemplateSetting(mContext, settings, this);
            templateConfigSettings.put(settings.templateCode, tempSetting);
            return true;
        }
    }

    public TDTemplateSetting getTemplateSetting(String tempCode) {
        if (SystemConfigManager.getInstance().isRCCStop(this.appId)) return null;
        synchronized (templateConfigSettings) {
            if (TextUtils.isEmpty(tempCode)) {
                return templateConfigSettings.get(TDRemoteConfigConstants.TEMPLATE_DEFAULT);
            }
            return templateConfigSettings.get(tempCode);
        }
    }

    public Map<String, TDTemplateSetting> getTemplateConfigSettings() {
        synchronized (templateConfigSettings) {
            return templateConfigSettings;
        }
    }

    public String getAppId() {
        return this.appId;
    }

    public String getServerUrl(String tempCode) {
        TDTemplateSetting templateSetting = getTemplateSetting(tempCode);
        if (templateSetting != null) {
            return templateSetting.getServerUrl();
        }
        return "";
    }

    public String getAccountId() {
        IAnalyticsProvider provider = AnalyticUtil.getAnalyticProvider();
        if (provider != null) {
            return provider.getLoginId(this.appId);
        }
        return "";
    }

    public String getDistinctId() {
        IAnalyticsProvider provider = AnalyticUtil.getAnalyticProvider();
        if (provider != null) {
            return provider.getDistinctId(this.appId);
        }
        return "";
    }

    public void updateAppConfigs(TDNetModel model) {
        if (null != model.data) {
            String userId = model.data.optString("localUserId");
            if (!TextUtils.isEmpty(userId)) {
                mAppStoragePlugin.save(LocalStorageType.LOCAL_USER_ID, userId);
                JSONObject templateInfo = model.data.optJSONObject("templateInfo");
                long userIdPeriod = 0L;
                if (templateInfo != null) {
                    userIdPeriod = templateInfo.optLong("userIdPeriod");
                }
                if (userIdPeriod <= 0L) {
                    userIdPeriod = TDRemoteConfigConstants.USER_ID_EXPIRED_TIME;
                }
                long expiredTime = System.currentTimeMillis() / 1000 + userIdPeriod;
                mAppStoragePlugin.save(LocalStorageType.USER_ID_EXPIRED, expiredTime);
            }
        }
    }

    public String getUserId() {
        long expiredTime = mAppStoragePlugin.get(LocalStorageType.USER_ID_EXPIRED);
        if (expiredTime == -1L) {
            return "";
        }
        if (System.currentTimeMillis() / 1000 > expiredTime) {
            clearUserId();
            LogUtil.i("local userId has expired and has been cleared");
            return "";
        }
        return mAppStoragePlugin.get(LocalStorageType.LOCAL_USER_ID);
    }

    public void clearUserId() {
        mAppStoragePlugin.save(LocalStorageType.LOCAL_USER_ID, "");
        mAppStoragePlugin.save(LocalStorageType.USER_ID_EXPIRED, -1L);
    }

    public boolean isFirstChangeAccount() {
        return TextUtils.isEmpty(mAppStoragePlugin.get(LocalStorageType.LAST_FETCH_CLIENT_USER));
    }

    public boolean isChangeAccount(String accountId, String clientUserId) {
        String lastClientUserId = mAppStoragePlugin.get(LocalStorageType.LAST_FETCH_CLIENT_USER);
        boolean isClientUserIdChange = !TextUtils.equals(lastClientUserId, clientUserId);
        if (isClientUserIdChange) {
            mAppStoragePlugin.save(LocalStorageType.LAST_FETCH_CLIENT_USER, clientUserId);
        }
        String lastAccountId = mAppStoragePlugin.get(LocalStorageType.LAST_ACCOUNT_ID);
        boolean isAccountIdChange = !TextUtils.equals(lastAccountId, accountId);
        if (isAccountIdChange) {
            mAppStoragePlugin.save(LocalStorageType.LAST_ACCOUNT_ID, accountId);
        }
        boolean isChange = isClientUserIdChange || isAccountIdChange;
        currentClientUserId = clientUserId;
        return isChange;
    }

}
