/*
 * Copyright (C) 2023 ThinkingData
 */
package cn.thinkingdata.remoteconfig.utils;

import android.content.Context;
import android.text.TextUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import cn.thinkingdata.core.network.TDNetResponse;
import cn.thinkingdata.core.preset.TDPresetModel;
import cn.thinkingdata.core.router.provider.IAnalyticsProvider;
import cn.thinkingdata.core.utils.MessageDigestUtils;
import cn.thinkingdata.core.utils.ZipUtils;
import cn.thinkingdata.remoteconfig.TDRemoteConfigSettings;
import cn.thinkingdata.remoteconfig.core.TDTemplateSetting;
import cn.thinkingdata.remoteconfig.model.TDNetModel;
import cn.thinkingdata.remoteconfig.task.TimerTaskInfo;
import cn.thinkingdata.remoteconfig.user.TDUserInfo;

/**
 * @author liulongbing
 * @create 2023/12/7
 * @since
 */
public class RemoteConfigUtil {

    public static String calculateSHA256(String timeStamp, String data) {
        return MessageDigestUtils.calculateSHA256(timeStamp + data + "thinkingdata-remote-config");
    }

    private static TDRemoteConfigSettings copyNewSettings(TDRemoteConfigSettings old) {
        TDRemoteConfigSettings settings = new TDRemoteConfigSettings();
        settings.appId = old.appId;
        settings.serverUrl = old.serverUrl;
        settings.templateCode = old.templateCode;
        settings.mode = old.mode;
        if (old.getCustomFetchParams() != null) {
            try {
                JSONObject params = new JSONObject();
                Iterator<String> iterator = old.getCustomFetchParams().keys();
                while (iterator.hasNext()) {
                    String key = iterator.next();
                    params.put(key, old.getCustomFetchParams().opt(key));
                }
                settings.setCustomFetchParams(params);
            } catch (Exception ignore) {
            }
        }
        return settings;
    }

    public static List<TimerTaskInfo> copyNewTimerTasks(List<TimerTaskInfo> old) {
        List<TimerTaskInfo> newList = null;
        if (old != null) {
            newList = new ArrayList<>();
            for (TimerTaskInfo info : old) {
                TimerTaskInfo newInfo = new TimerTaskInfo(info.appId, info.tempCode);
                newInfo.isDebug = info.isDebug;
                newList.add(newInfo);
            }
        }
        return newList;
    }

    public static TDRemoteConfigSettings buildSettings(TDRemoteConfigSettings settings) {
        //深拷贝一份出来
        TDRemoteConfigSettings newSettings = copyNewSettings(settings);
        if (TextUtils.isEmpty(newSettings.appId) || TextUtils.isEmpty(newSettings.serverUrl)) {
            LogUtil.i("appId or serverUrl can not be empty");
            return null;
        }
        newSettings.appId = newSettings.appId.replace(" ", "");
        if (TextUtils.isEmpty(newSettings.templateCode)) {
            newSettings.templateCode = TDRemoteConfigConstants.TEMPLATE_DEFAULT;
        }
        return newSettings;
    }

    public static JSONObject obtainPostParams(Context context, String appId, TDTemplateSetting templateSetting, TDUserInfo userInfo) {
        JSONObject postJson = new JSONObject();
        if (null == templateSetting) return postJson;
        try {
            postJson.put("appid", appId);
            if (userInfo.userId != -1L) {
                postJson.put("local_user_id", userInfo.userId);
            }
            postJson.put("template_code", templateSetting.getTempCode());
            postJson.put("last_fetch_time", templateSetting.lastFetchStamp + "");
            JSONObject request_params = new JSONObject();
            Map<String, Object> presetProperties = TDPresetModel.getInstance(context).getPresetProperties();
            request_params.put("#carrier", presetProperties.get("#carrier"));
            request_params.put("#os", presetProperties.get("#os"));
            request_params.put("#bundle_id", presetProperties.get("#bundle_id"));
            request_params.put("#device_model", presetProperties.get("#device_model"));
            request_params.put("#system_language", presetProperties.get("#system_language"));
            request_params.put("#simulator", presetProperties.get("#simulator"));
            request_params.put("#manufacturer", presetProperties.get("#manufacturer"));
            request_params.put("#os_version", presetProperties.get("#os_version"));
            request_params.put("#app_version", presetProperties.get("#app_version"));
            request_params.put("#network_type", TDPresetModel.getInstance(context).getCurrentNetworkType());
            request_params.put("#device_type", presetProperties.get("#device_type"));
            request_params.put("#device_id", TDPresetModel.getInstance(context).getDeviceId());
            IAnalyticsProvider provider = AnalyticUtil.getAnalyticProvider();
            if (provider != null) {
                Map<String, Object> analyticsProperties = provider.getAnalyticsProperties(appId);
                request_params.put("#zone_offset", analyticsProperties.get("#zone_offset"));
                request_params.put("#install_time", analyticsProperties.get("#install_time"));
            }
            if (!TextUtils.isEmpty(userInfo.accountId)) {
                request_params.put("#account_id", userInfo.accountId);
            }
            request_params.put("#distinct_id", userInfo.distinctId);
            request_params.put("#client_user_id", userInfo.clientUserId);
            Map<String, String> customParams = templateSetting.getCustomParams();
            for (String s : customParams.keySet()) {
                request_params.put(s, customParams.get(s));
            }
            postJson.put("request_params", request_params);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return postJson;
    }

    public static TDNetModel toNetModel(TDNetResponse response, String appId) {
        if (response == null) return null;
        TDNetModel model = new TDNetModel();
        if (response.statusCode == 200) {
            try {
                JSONObject dataJson = new JSONObject(response.responseData);
                model.code = dataJson.optInt("code");
                model.msg = dataJson.optString("msg");
                if (model.code == 0) {
                    String data = dataJson.optString("data");
                    String timestamp = "";
                    String sign = "";
                    if (response.responseHeaders != null) {
                        List<String> timestampList = response.responseHeaders.get("timestamp");
                        if (timestampList != null && timestampList.size() > 0) {
                            timestamp = timestampList.get(0);
                        }
                        List<String> signList = response.responseHeaders.get("sign");
                        if (signList != null && signList.size() > 0) {
                            sign = signList.get(0);
                        }
                    }
                    if (TextUtils.isEmpty(timestamp) || TextUtils.isEmpty(sign)) {
                        model.code = TDRemoteConfigConstants.SIGN_NOT_FIND_ERROR_CODE;
                        model.msg = TDRemoteConfigConstants.SIGN_NOT_FIND_ERROR_MSG;
                        return model;
                    }
                    model.timeStamp = Long.parseLong(timestamp);
                    String datsSign = calculateSHA256(timestamp, data);
                    if (!TextUtils.equals(sign, datsSign)) {
                        //签名验证失败
                        model.code = TDRemoteConfigConstants.SIGN_ILLEGAL_ERROR_CODE;
                        model.msg = TDRemoteConfigConstants.SIGN_ILLEGAL_ERROR_MSG;
                        AnalyticUtil.trackRiskEvent(appId,"1");
                        return model;
                    }
                    String str = ZipUtils.unGzip(data);
                    model.data = new JSONObject(str);
                }
            } catch (Exception e) {
                model.code = TDRemoteConfigConstants.DATA_PARSE_ERROR_CODE;
                model.msg = e.getMessage();
            }
        } else {
            model.code = response.statusCode;
            model.msg = response.msg;
        }
        return model;
    }

    public static String MD5(String str) {
        return MessageDigestUtils.calculateMD5(str + "td-remote-config");
    }

}
