/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.remoteconfig.sysconfig;

import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import cn.thinkingdata.core.network.Request;
import cn.thinkingdata.core.network.TDNetResponse;
import cn.thinkingdata.core.network.TEHttpCallback;
import cn.thinkingdata.core.network.TEHttpClient;
import cn.thinkingdata.remoteconfig.utils.SpUtils;
import cn.thinkingdata.remoteconfig.utils.LogUtil;
import cn.thinkingdata.remoteconfigsdk.BuildConfig;

/**
 * @author liulongbing
 * @since 2024/5/14
 */
public class SystemConfigManager {

    private volatile static SystemConfigManager instance;

    private final Object configLock = new Object();
    private final List<SystemConfigModel> systemConfigs = new ArrayList<>();

    private SystemConfigManager() {

    }

    public static SystemConfigManager getInstance() {
        if (instance == null) {
            synchronized (SystemConfigManager.class) {
                instance = new SystemConfigManager();
            }
        }
        return instance;
    }

    public void loadLocalConfig(String appId, String serverUrl) {
        if (isFindAppIdConfig(appId)) return;
        String localConfig = SpUtils.getInstance().getString(SpUtils.KEY_SYSTEM_CONFIG + appId, "{}");
        parseSystemConfig(appId, localConfig);
        loadRemoteSystemConfig(appId, serverUrl);
    }

    public boolean isFindAppIdConfig(String appId) {
        synchronized (configLock) {
            for (SystemConfigModel model : systemConfigs) {
                if (model != null && TextUtils.equals(model.appId, appId)) {
                    return true;
                }
            }
        }
        return false;
    }

    private void parseSystemConfig(String appId, String str) {
        try {
            JSONObject json = new JSONObject(str);
            Iterator<String> keys = json.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                JSONObject valueJson = json.optJSONObject(key);
                if (valueJson == null) continue;
                SystemConfigModel model = new SystemConfigModel();
                model.appId = appId;
                model.plat = key;
                model.disabled = valueJson.optBoolean("disabled");
                JSONArray disabledVersionList = valueJson.optJSONArray("disabled_version_list");
                if (disabledVersionList != null) {
                    model.disabledVersionList = new ArrayList<>();
                    for (int i = 0; i < disabledVersionList.length(); i++) {
                        model.disabledVersionList.add(disabledVersionList.optString(i));
                    }
                }
                synchronized (configLock) {
                    systemConfigs.add(model);
                }

            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public void loadRemoteSystemConfig(String appId, String serverUrl) {
        if (serverUrl == null) return;
        try {
            URL url = new URL(serverUrl);
            String str = url.getProtocol()
                    + "://" + url.getHost()
                    + (url.getPort() > 0 ? ":" + url.getPort() : "");
            TEHttpClient client = new TEHttpClient.Builder()
                    .build();
            Request request = new Request.Builder()
                    .url(str + "/v1/remote-config/config?appid=" + appId)
                    .get().build();
            client.newCall(request).enqueue(new TEHttpCallback() {
                @Override
                public void onFailure(int i, String s) {

                }

                @Override
                public void onSuccess(TDNetResponse response) {
                    if (null != response) {
                        String dataStr = response.responseData;
                        LogUtil.i("Get system config success : " + dataStr);
                        SpUtils.getInstance().putString(SpUtils.KEY_SYSTEM_CONFIG + appId, dataStr);
                        parseSystemConfig(appId, dataStr);
                    }
                }
            });
        } catch (Exception ignore) {
        }
    }

    public boolean isRCCStop(String appId) {
        return isStop(appId, "rcc_client", BuildConfig.SDK_VERSION_NAME);
    }

    public boolean isStrategyStop(String appId, String version) {
        return isRCCStop(appId) || isStop(appId, "strategy_client", version);
    }

    private boolean isStop(String appId, String plat, String version) {
        synchronized (configLock) {
            for (SystemConfigModel model : systemConfigs) {
                if (TextUtils.equals(appId, model.appId) && TextUtils.equals(plat, model.plat)) {
                    if (model.disabled) return true;
                    return model.disabledVersionList != null && model.disabledVersionList.contains(version);
                }
            }
            return false;
        }
    }


}
