/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.remoteconfig.core;

import android.content.Context;
import android.text.TextUtils;
import android.util.Pair;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.thinkingdata.core.exception.TDHttpException;
import cn.thinkingdata.core.network.Request;
import cn.thinkingdata.core.network.TDNetResponse;
import cn.thinkingdata.core.network.TEHttpClient;
import cn.thinkingdata.core.preset.TDPresetModel;
import cn.thinkingdata.core.receiver.TDAnalyticsObservable;
import cn.thinkingdata.core.receiver.TDNetWorkObservable;
import cn.thinkingdata.core.utils.NetWorkUtils;
import cn.thinkingdata.remoteconfig.TDRemoteConfig;
import cn.thinkingdata.remoteconfig.TDRemoteConfigSettings;
import cn.thinkingdata.remoteconfig.adapter.RCNetWorkReceiverAdapter;
import cn.thinkingdata.remoteconfig.model.TDNetModel;
import cn.thinkingdata.remoteconfig.receiver.TDRemoteConfigObservable;
import cn.thinkingdata.remoteconfig.receiver.TDRemoteConfigObserver;
import cn.thinkingdata.remoteconfig.sysconfig.SystemConfigManager;
import cn.thinkingdata.remoteconfig.task.RemoteTaskManager;
import cn.thinkingdata.remoteconfig.task.RemoteTimerTask;
import cn.thinkingdata.remoteconfig.user.ClientUserManager;
import cn.thinkingdata.remoteconfig.user.TDUserInfo;
import cn.thinkingdata.remoteconfig.utils.AnalyticUtil;
import cn.thinkingdata.remoteconfig.utils.JsonUtil;
import cn.thinkingdata.remoteconfig.utils.RemoteConfigUtil;
import cn.thinkingdata.remoteconfig.utils.SpUtils;
import cn.thinkingdata.remoteconfig.utils.LogUtil;
import cn.thinkingdata.remoteconfigsdk.BuildConfig;

/**
 * @author liulongbing
 * @since 2024/5/9
 */
public class TDRemoteInstance {

    private static final Map<String, TDRemoteInstance> sInstances = new HashMap<>();
    public static String defaultAppId;//默认的AppID，最先初始化的

    private boolean isRegisterNetWorkReceiver = false;

    final Context mContext;
    private final TDAppSetting mAppSetting;

    private final List<String> tempNetStatusList;

    public TDRemoteInstance(Context context, TDRemoteConfigSettings mConfigSettings) {
        this.mContext = context.getApplicationContext();
        this.mAppSetting = new TDAppSetting(mContext, mConfigSettings);
        tempNetStatusList = new ArrayList<>();
    }

    static {
        TDAnalyticsObservable.getInstance().registerObserver(new TDRemoteConfigObserver());
    }

    public static TDRemoteInstance getInstance(String appId) {
        synchronized (sInstances) {
            if (TextUtils.isEmpty(appId)) {
                return sInstances.get(defaultAppId);
            }
            return sInstances.get(appId);
        }
    }

    public static void init(Context context, TDRemoteConfigSettings mConfigSettings) {
        synchronized (sInstances) {
            TDRemoteConfigSettings settings = RemoteConfigUtil.buildSettings(mConfigSettings);
            if (settings == null || context == null) return;
            TDRemoteInstance instance = sInstances.get(settings.appId);
            if (instance == null) {
                instance = new TDRemoteInstance(context, settings);
                if (TextUtils.isEmpty(defaultAppId)) {
                    defaultAppId = settings.appId;
                }
                sInstances.put(settings.appId, instance);
                //异步初始化耗时任务
                instance.initAsync(settings);
            } else {
                if (instance.addNewTemplateCode(settings)) {
                    instance.initAsync(settings);
                }
            }
        }
    }

    private void initAsync(TDRemoteConfigSettings configSettings) {
        RemoteTaskManager.getInstance().addTask(new Runnable() {
            @Override
            public void run() {
                SpUtils.getInstance().init(mContext);
                SystemConfigManager.getInstance().loadLocalConfig(configSettings.appId, configSettings.serverUrl);
                if (SystemConfigManager.getInstance().isRCCStop(configSettings.appId)) return;
                mAppSetting.initAppInfoAsync(configSettings.templateCode);
                RemoteTimerTask.getInstance().registerActivityLifeCycle(mContext);
                RemoteTimerTask.getInstance().addNewConfigs(configSettings);
                registerNetWorkObserver();
                //初始化请求数据
                fetchWithTempCode(configSettings.templateCode, 3, false);
                LogUtil.i(String.format("TDRemoteConfig SDK initialize success with appId = %s , templateCode = %s , version = %s", configSettings.appId, configSettings.templateCode, BuildConfig.SDK_VERSION_NAME));
            }
        });
    }

    public void fetchWithTempCode(String code, int retryCount, boolean isNeedControl) {
        if (SystemConfigManager.getInstance().isRCCStop(mAppSetting.getAppId())) return;
        TDUserInfo userInfo = new TDUserInfo(mAppSetting.getAccountId(), mAppSetting.getDistinctId(), getClientUserId(), mAppSetting.getUserId());
        JSONObject params = RemoteConfigUtil.obtainPostParams(mContext, mAppSetting.getAppId(), mAppSetting.getTemplateSetting(code), userInfo);
        fetchSync(code, retryCount, params, isNeedControl);
    }

    private boolean addNewTemplateCode(TDRemoteConfigSettings settings) {
        return this.mAppSetting.addTemplateCode(settings);
    }

    public TDTemplateSetting getTemplateSetting(String tempCode) {
        return mAppSetting.getTemplateSetting(tempCode);
    }

    public void fetchSync(String tempCode, int retryCount, JSONObject postParams, boolean isNeedControl) {
        if (!NetWorkUtils.isNetWorkAvailable(mContext)) {
            tempNetStatusList.add(tempCode);
            return;
        }
        TDTemplateSetting templateSetting = mAppSetting.getTemplateSetting(tempCode);
        if (templateSetting == null || postParams == null) return;
        templateSetting.addRequestCount();
        if (isNeedControl && templateSetting.isRequestControl()) return;
        try {
            TEHttpClient client = new TEHttpClient.Builder()
                    .readTimeout(30000)
                    .connectTimeout(30000)
                    .build();
            String postStr = postParams.toString();
            long timeStamp = AnalyticUtil.getCurrentTimeStamp();
            String sign = RemoteConfigUtil.calculateSHA256(timeStamp + "", postStr);
            LogUtil.i(String.format("fetch config start with appId = %s，template = %s，params = %s", mAppSetting.getAppId(), tempCode, postParams.toString(4)));
            Request request = new Request.Builder()
                    .url(mAppSetting.getServerUrl(tempCode))
                    .addHeader("sign", sign)
                    .addHeader("Content-Type", "application/json")
                    .addHeader("Connection", "close")
                    .addHeader("device_id", TDPresetModel.getInstance(mContext).getDeviceId())
                    .addHeader("template_code", templateSetting.getTempCode())
                    .addHeader("timestamp", timeStamp + "")
                    .addHeader("appid", mAppSetting.getAppId())
                    .post(postStr).build();
            TDNetResponse response = client.newCall(request).execute();
            TDNetModel model = RemoteConfigUtil.toNetModel(response, mAppSetting.getAppId());
            if (model.code == 0) {
                LogUtil.i(String.format("get remote config success = %s", model.data.toString(4)));
                mAppSetting.updateAppConfigs(model);
                templateSetting.updateRemoteConfigs(model);
            } else {
                LogUtil.e(String.format("get remote config fail with reason %s", model.msg));
                if (model.code < 0 && model.code > -100) {
                    //错误最终态
                    templateSetting.uploadDebugEvent(model.code);
                    RemoteTimerTask.getInstance().removeSystemErrorTask(mAppSetting.getAppId(), tempCode);
                } else if ((model.code <= -100 && model.code >= -200) || model.code == -201) {
                    retryCount--;
                    if (retryCount > 0) {
                        LogUtil.i("retry fetch after 3 seconds");
                        Thread.sleep(3000);
                        fetchSync(tempCode, retryCount, postParams, isNeedControl);
                    }
                } else if (model.code == -202 || model.code == -203) {
                    //debug模式降级
                    templateSetting.downGradeMode();
                } else if ((model.code >= 500 && model.code < 600) || model.code == TDHttpException.ERROR_CONNECT_TIME_OUT || model.code == TDHttpException.ERROR_EXCEPTION) {
                    //服务器错误 500-600  超时 网络异常 不需要移除 等待30轮训重试
                } else {
                    RemoteTimerTask.getInstance().removeSystemErrorTask(mAppSetting.getAppId(), tempCode);
                }
            }
        } catch (Exception e) {
            LogUtil.e("get config fail with reason ", e);
        }
    }

    private String getClientUserId() {
        return mAppSetting.getClientUserId();
    }

    public void setCustomParams(JSONObject params, String tempCode) {
        TDTemplateSetting templateSetting = mAppSetting.getTemplateSetting(tempCode);
        if (null != templateSetting) {
            RemoteTaskManager.getInstance().addTask(new Runnable() {
                @Override
                public void run() {
                    templateSetting.mergeCustomParams(params);
                }
            });
        }
    }

    public void removeCustomFetchParams(String key, String tempCode) {
        TDTemplateSetting templateSetting = mAppSetting.getTemplateSetting(tempCode);
        if (null != templateSetting) {
            RemoteTaskManager.getInstance().addTask(new Runnable() {
                @Override
                public void run() {
                    templateSetting.removeCustomParams(key);
                }
            });
        }
    }

    public void applyDefaultParams(JSONObject params, String tempCode) {
        TDTemplateSetting templateSetting = mAppSetting.getTemplateSetting(tempCode);
        if (null != templateSetting) {
            templateSetting.mergeDefaultParams(params);
        }
    }

    public void addConfigFetchListener(TDRemoteConfig.OnConfigFetchListener listener, String tempCode) {
        TDTemplateSetting templateSetting = mAppSetting.getTemplateSetting(tempCode);
        if (null != templateSetting) {
            templateSetting.addConfigFetchListener(listener);
        }
    }

    public void applyDefaultParams(String filePath, String tempCode) {
        if (filePath == null) return;
        TDTemplateSetting templateSetting = mAppSetting.getTemplateSetting(tempCode);
        if (null != templateSetting) {
            RemoteTaskManager.getInstance().addTask(new Runnable() {
                @Override
                public void run() {
                    try {
                        if (filePath.contains("/")) {
                            templateSetting.mergeDefaultParams(new JSONObject(JsonUtil.getJsonFromPath(filePath)));
                        } else {
                            templateSetting.mergeDefaultParams(new JSONObject(JsonUtil.getJsonFromAssets(filePath, mContext)));
                        }
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }
            });
        }
    }

    public void clearDefaultParams(String tempCode) {
        TDTemplateSetting templateSetting = mAppSetting.getTemplateSetting(tempCode);
        if (null != templateSetting) {
            templateSetting.clearDefaultParams();
        }
    }

    public Pair<JSONObject, JSONObject> getAll(String tempCode) {
        TDTemplateSetting templateSetting = mAppSetting.getTemplateSetting(tempCode);
        if (null != templateSetting) {
            return templateSetting.getAll();
        }
        return new Pair<>(new JSONObject(), new JSONObject());
    }

    private void registerNetWorkObserver() {
        if (isRegisterNetWorkReceiver) return;
        isRegisterNetWorkReceiver = true;
        TDNetWorkObservable.getInstance(mContext).addNetWorkObserver(new RCNetWorkReceiverAdapter() {
            @Override
            public void onNetWorkChanged() {
                if (NetWorkUtils.isNetWorkAvailable(mContext)) {
                    tryFetch();
                }
            }
        });
    }

    private void tryFetch() {
        RemoteTaskManager.getInstance().addTask(new Runnable() {
            @Override
            public void run() {
                for (String temp : tempNetStatusList) {
                    LogUtil.i("network is available , fetch remote config");
                    fetchWithTempCode(temp, 1, false);
                }
                tempNetStatusList.clear();
            }
        });
    }

    public void doChangeAccount(String accountId, String distinctId) {
        if (SystemConfigManager.getInstance().isRCCStop(mAppSetting.getAppId())) return;
        ClientUserManager.getInstance(mContext).getClientUserId(mAppSetting.getAppId(), accountId, distinctId, new ClientUserManager.OnGetClientUserIdCallBack() {
            @Override
            public void onClientUserIdSuccess(String clientUserId) {
                RemoteTaskManager.getInstance().addTask(new Runnable() {
                    @Override
                    public void run() {
                        LogUtil.i(String.format("do change account with appId = %s , accountId = %s , distinctId = %s", mAppSetting.getAppId(), accountId, distinctId));
                        mAppSetting.clearUserId();
                        boolean isFirstChangeAccount = mAppSetting.isFirstChangeAccount();
                        boolean isChangeAccount = mAppSetting.isChangeAccount(accountId, clientUserId);
                        if (!isChangeAccount) return;
                        LogUtil.i("change account success");
                        for (String tempCode : mAppSetting.getTemplateConfigSettings().keySet()) {
                            TDTemplateSetting templateSetting = mAppSetting.getTemplateSetting(tempCode);
                            if (templateSetting == null) continue;
                            if (!isFirstChangeAccount) {
                                templateSetting.clearOldConfig();
                            }
                            RemoteTimerTask.getInstance().removeDelayTemplateMessage();
                            TDRemoteConfigObservable.getInstance().notifyOnChangeAccount(mAppSetting.getAppId(), tempCode, clientUserId);
                            if (templateSetting.isRequestControl()) return;
                            TDUserInfo userInfo = new TDUserInfo(accountId, distinctId, clientUserId, mAppSetting.getUserId());
                            JSONObject params = RemoteConfigUtil.obtainPostParams(mContext, mAppSetting.getAppId(), mAppSetting.getTemplateSetting(tempCode), userInfo);
                            fetchSync(tempCode, 1, params, true);
                        }
                    }
                });
            }
        });
    }

}
