package com.therouter.plugin;

import com.android.build.api.transform.*;
import com.android.build.gradle.internal.pipeline.TransformManager;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;

import org.apache.commons.io.FileUtils;
import org.gradle.api.Project

/**
 * Created by ZhangTao on 18/2/24.
 */

public class TheRouterTransform extends Transform {

    private Project mProject;
    private boolean isLibrary;

    private static final String CACHE_THEROUTER_JAR = "therouter_jar";
    private static final String CACHE_ALL_FILE = "all_class_file";
    private static final String CACHE_FLOW_TASK_DEPEND = "flow_task_depend";
    private static final String CACHE_ROUTERMAP = "router_map";

    public TheRouterTransform(Project p, boolean isLibrary) {
        this.mProject = p;
        this.isLibrary = isLibrary;
    }

    @Override
    public String getName() {
        return "TheRouter";
    }

    @Override
    public Set<QualifiedContent.ContentType> getInputTypes() {
        return TransformManager.CONTENT_CLASS;
    }

    @Override
    public Set<QualifiedContent.Scope> getScopes() {
        HashSet<QualifiedContent.Scope> set = new HashSet<>();
        // 从1.2.3版本开始，只遍历源码，遍历jar包是为了把源码的改动通过字节码写入TheRouter注入类
        if (isLibrary) {
            set.add(QualifiedContent.Scope.PROJECT);
        } else {
            set.add(QualifiedContent.Scope.PROJECT);
            set.add(QualifiedContent.Scope.SUB_PROJECTS);
            set.add(QualifiedContent.Scope.EXTERNAL_LIBRARIES);
        }
        return set;
    }

    @Override
    public boolean isIncremental() {
        return Boolean.parseBoolean(Utils.getLocalProperty(mProject, TheRouterPlugin.INCREMENTAL));
    }

    @Override
    public void transform(Context context, Collection<TransformInput> inputs,
                          Collection<TransformInput> referencedInputs,
                          TransformOutputProvider outputProvider,
                          boolean isIncremental) {
        try {
            theRouterTransform(isIncremental, inputs, outputProvider);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void theRouterTransform(boolean isIncremental, Collection<TransformInput> inputs, TransformOutputProvider outputProvider) throws Exception {
        System.out.println("TheRouter编译插件：" + LogUI.C_BLACK_GREEN.getValue() + "cn.therouter:" + BuildConfig.NAME
                + ":" + BuildConfig.VERSION + LogUI.E_NORMAL.getValue());
        System.out.println("当前编译 JDK Version 为::" + System.getProperty("java.version"));
        System.out.println("本次是增量编译::" + isIncremental);
        System.out.println("CHECK_ROUTE_MAP::" + Utils.getLocalProperty(mProject, TheRouterPlugin.CHECK_ROUTE_MAP));
        System.out.println("CHECK_FLOW_UNKNOW_DEPEND::" + Utils.getLocalProperty(mProject, TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND));

        System.out.println("---------TheRouter handle " + mProject.getName() + " Module-------------------------");
        if (!isIncremental) {
            outputProvider.deleteAll();
        }
        File therouterJar = null;
        for (TransformInput input : inputs) {
            // 从1.2.3版本开始，只遍历源码，遍历jar包是为了把源码的改动通过字节码写入TheRouter注入类
            if (!isLibrary) {
                // 遍历jar包
                for (JarInput jarInput : input.getJarInputs()) {
                    String jarName = jarInput.getName().toLowerCase();
                    File dest = outputProvider.getContentLocation(jarName, jarInput.getContentTypes(),
                            jarInput.getScopes(), Format.JAR);
                    if (jarName.contains("cn.therouter:router:")) {
                        therouterJar = dest;
                        FileUtils.write(Utils.createCacheFile(mProject, CACHE_THEROUTER_JAR), dest.getAbsolutePath(), "UTF-8", false);
                    }
                    if (dest.exists()) {
                        FileUtils.forceDelete(dest);
                    }
                    FileUtils.copyFile(jarInput.getFile(), dest);
                }
            }
            // 从1.2.3版本开始，只遍历源码
            for (DirectoryInput directoryInput : input.getDirectoryInputs()) {
                File dir = directoryInput.getFile();
                File dest = outputProvider.getContentLocation(directoryInput.getName(),
                        directoryInput.getContentTypes(), directoryInput.getScopes(), Format.DIRECTORY);
                FileUtils.forceMkdir(dest);
                String srcDirPath = dir.getAbsolutePath();
                String destDirPath = dest.getAbsolutePath();
                if (isIncremental) {
                    Map<File, Status> fileStatusMap = directoryInput.getChangedFiles();
                    for (Map.Entry<File, Status> changedFile : fileStatusMap.entrySet()) {
                        Status status = changedFile.getValue();
                        File inputFile = changedFile.getKey();
                        String destFilePath = inputFile.getAbsolutePath().replace(srcDirPath, destDirPath);
                        File destFile = new File(destFilePath);
                        switch (status) {
                            case REMOVED:
                                if (destFile.exists()) {
                                    FileUtils.forceDelete(destFile);
                                }
                                break;
                            case ADDED:
                            case CHANGED:
                                if (destFile.exists()) {
                                    FileUtils.forceDelete(destFile);
                                }
                                FileUtils.copyFile(inputFile, destFile);
                                SourceInfo sourceInfo = TheRouterInjects.tagClass(mProject, destFile);
                                // 路由表
                                String routeMapString = Utils.set2String(sourceInfo.routeMapStringFromSource);
                                if (!routeMapString.isEmpty()) {
                                    FileUtils.write(Utils.createCacheFile(mProject, CACHE_ROUTERMAP), routeMapString, "UTF-8", true);
                                }
                                // FlowTask依赖，用于循环检测
                                String flowTaskDependString = Utils.set2String(sourceInfo.serviceProviderFromSource);
                                if (!flowTaskDependString.isEmpty()) {
                                    FileUtils.write(Utils.createCacheFile(mProject, CACHE_FLOW_TASK_DEPEND), flowTaskDependString, "UTF-8", true);
                                }
                                break;
                            default:
                                break;
                        }
                    }
                } else {
                    if (dest.exists()) {
                        FileUtils.forceDelete(dest);
                    }
                    FileUtils.copyDirectory(dir, dest);
                    SourceInfo sourceInfo = TheRouterInjects.tagClass(mProject, dest);

                    // 当前工程的全部类，用于路由表校验
                    String all = Utils.set2String(sourceInfo.allSourceClass);
                    if (!all.isEmpty()) {
                        FileUtils.write(Utils.createCacheFile(mProject, CACHE_ALL_FILE), all, "UTF-8", false);
                    }
                    // 路由表
                    String routeMapString = Utils.set2String(sourceInfo.routeMapStringFromSource);
                    if (!routeMapString.isEmpty()) {
                        FileUtils.write(Utils.createCacheFile(mProject, CACHE_ROUTERMAP), routeMapString, "UTF-8", false);
                    }
                    // FlowTask依赖，用于循环检测
                    String flowTaskDependString = Utils.set2String(sourceInfo.serviceProviderFromSource);
                    if (!flowTaskDependString.isEmpty()) {
                        FileUtils.write(Utils.createCacheFile(mProject, CACHE_FLOW_TASK_DEPEND), flowTaskDependString, "UTF-8", false);
                    }
                }
            }
        }

        if (!isLibrary) {
            Map<String, String> serviceProvideMap = new HashMap<>();
            Set<String> autowiredSet = new HashSet<>();
            Set<String> routeSet = new HashSet<>();
            for (Project it : mProject.getRootProject().getSubprojects()) {
                // serviceProvideMap
                List<String> serviceprovidercache = FileUtils.readLines(
                        Utils.createCacheFile(it, TheRouterInjects.THEROUTER_SERVICE_PROVIDER_CLASS), "UTF-8");
                for (String line : serviceprovidercache) {
                    serviceProvideMap.put(line.trim(), BuildConfig.VERSION);
                }
                // autowiredSet
                List<String> autowiredFileContent = FileUtils.readLines(
                        Utils.createCacheFile(it, TheRouterInjects.THEROUTER_AUTOWIRED_CLASS), "UTF-8");
                for (String line : autowiredFileContent) {
                    autowiredSet.add(line.trim());
                }
                // routeSet
                List<String> routerClassFileContent = FileUtils.readLines(
                        Utils.createCacheFile(it, TheRouterInjects.THEROUTER_AUTOWIRED_CLASS), "UTF-8");
                for (String line : routerClassFileContent) {
                    routeSet.add(line.trim());
                    println "=========" + line
                }
            }
            if (therouterJar != null) {
                TheRouterInjects.injectClassCode(therouterJar, serviceProvideMap, autowiredSet, routeSet, isIncremental);
            } else {
                throw new RuntimeException("未找到therouter依赖");
            }
        }

        check(isIncremental);

        System.out.println("---------TheRouter handle " + mProject.getName() + " Module finish-------------------------");
    }

    private void check(boolean isIncremental) throws Exception {
        final Set<String> allClass = new HashSet<>();
        final Set<String> routeMapStringSet = new HashSet<>();
        final Map<String, String> flowTaskDependMap = new HashMap<>();
        if (!isLibrary) {
            for (Project it : mProject.getRootProject().getSubprojects()) {
                // 记录全部类
                List<String> strings = FileUtils.readLines(Utils.createCacheFile(it, CACHE_ALL_FILE), "UTF-8");
                for (String line : strings) {
                    allClass.add(line.trim());
                }
                // 记录路由表
                List<String> strings3 = FileUtils.readLines(Utils.createCacheFile(it, CACHE_ROUTERMAP), "UTF-8");
                for (String line : strings3) {
                    routeMapStringSet.add(line.trim());
                }
                // 记录FlowTask依赖
                List<String> strings4 = FileUtils.readLines(Utils.createCacheFile(it, CACHE_FLOW_TASK_DEPEND), "UTF-8");
                Gson gson = new GsonBuilder().setPrettyPrinting().create();
                for (String line : strings4) {
                    HashMap<String, String> map = gson.fromJson(line.trim(), HashMap.class);
                    flowTaskDependMap.putAll(map);
                }
            }

            Set<RouteItem> pageSet = new HashSet<>();
            Gson gson = new GsonBuilder().setPrettyPrinting().create();
            for (String it : routeMapStringSet) {
                pageSet.addAll(gson.fromJson(it, new TypeToken<List<RouteItem>>() {
                }.getType()));
            }
            // 让第三方Activity也支持路由，第三方页面的路由表可以在assets中添加
            File assetRouteMap = new File(mProject.getProjectDir(), "src/main/assets/therouter/routeMap.json");
            if (assetRouteMap.exists()) {
                if (TheRouterPlugin.DELETE.equalsIgnoreCase(Utils.getLocalProperty(mProject, TheRouterPlugin.CHECK_ROUTE_MAP))) {
                    System.out.println("---------TheRouter delete route map------------------------------------------");
                    assetRouteMap.delete();
                    assetRouteMap.createNewFile();
                } else {
                    String assetString = FileUtils.readFileToString(assetRouteMap);
                    System.out.println("---------TheRouter get route map from: /assets/therouter/routeMap.json-------");
                    try {
                        List<RouteItem> assetsList = gson.fromJson(assetString, new TypeToken<List<RouteItem>>() {
                        }.getType());
                        for (RouteItem item : assetsList) {
                            if (!pageSet.contains(item)) {
                                pageSet.add(item);
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            } else {
                System.out.println("---------TheRouter route map does not exist: /assets/therouter/routeMap.json-------");
                assetRouteMap.getParentFile().mkdirs();
                assetRouteMap.createNewFile();
            }

            Map<String, List<RouteItem>> result = new HashMap<>();
            // 检查url合法性
            for (RouteItem routeItem : pageSet) {
                String url = routeItem.path;
                if (url.contains("?")) {
                    URI uri = new URI(routeItem.path);
                    Map<String, Object> map = uri.getProperties();
                    for (String key : map.keySet()) {
                        routeItem.params.put(key, map.get(key).toString());
                    }
                    url = url.substring(0, url.indexOf('?'));
                }
                List<RouteItem> routeList = result.get(url);
                if (routeList == null) {
                    routeList = new ArrayList<>();
                    routeList.add(routeItem);
                    result.put(url, routeList);
                }
            }
            // 检查路由表合法性
            for (List<RouteItem> list : result.values()) {
                String className = null;
                for (RouteItem routeItem : list) {
                    if (className == null) {
                        className = routeItem.className;
                    } else if (!className.equals(routeItem.className)) {
                        throw new RuntimeException("Multiple Activity to single Url: $className and " + routeItem.className);
                    }
                    if (!Utils.getLocalProperty(mProject, TheRouterPlugin.CHECK_ROUTE_MAP).isEmpty()) {
                        // 只在全量编译时检验
                        if (!isIncremental) {
                            boolean classNotFound = true;
                            for (String item : allClass) {
                                if (item.contains(routeItem.className)) {
                                    classNotFound = false;
                                    break;
                                }
                            }
                            if (classNotFound) {
                                if (TheRouterPlugin.ERROR.equalsIgnoreCase(Utils.getLocalProperty(mProject, TheRouterPlugin.CHECK_ROUTE_MAP))) {
                                    throw new ClassNotFoundException(routeItem.className + " in /assets/therouter/routeMap.json");
                                } else if (TheRouterPlugin.WARNING.equalsIgnoreCase(Utils.getLocalProperty(mProject, TheRouterPlugin.CHECK_ROUTE_MAP))) {
                                    System.out.println("" + LogUI.C_WARN.getValue() + "[" + routeItem.className + " in /assets/therouter/routeMap.json]" + LogUI.E_NORMAL.getValue());
                                }
                            }
                        }
                    }
                }
            }

            List<RouteItem> pageList = new ArrayList<>(pageSet);
            Collections.sort(pageList);
            String json = gson.toJson(pageList);
            FileUtils.write(assetRouteMap, json, false);
            System.out.println("---------TheRouter create new route map--------------");


            Map<String, Set<String>> flowTaskDepend = new HashMap<>();
            for (String taskName : flowTaskDependMap.keySet()) {
                Set<String> value = flowTaskDepend.get(taskName);
                if (value == null) {
                    value = new HashSet<>();
                }
                String dependsOn = flowTaskDependMap.get(taskName);
                if (!dependsOn.isBlank()) {
                    for (String depend : dependsOn.split(",")) {
                        if (!depend.isBlank()) {
                            value.add(depend.trim());
                        }
                    }
                }
                flowTaskDependMap.put(taskName, value);
            }

            if (!Utils.getLocalProperty(mProject, TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND).isEmpty()) {
                flowTaskDependMap.values().each {
                    taskName ->
                        flowTaskDependMap[taskName].each {
                            if (!flowTaskDependMap.containsKey(it)) {
                                if (TheRouterPlugin.ERROR.equalsIgnoreCase(Utils.getLocalProperty(mProject, TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND))) {
                                    throw new RuntimeException("\n\n==========================================" +
                                            "\nTheRouter:: FlowTask::   " +
                                            "\nCan not found Task: [" + it + "] from " + taskName + " dependsOn" +
                                            "\n==========================================\n\n");
                                } else if (TheRouterPlugin.ERROR.equalsIgnoreCase(Utils.getLocalProperty(mProject, TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND))) {
                                    System.out.println();
                                    System.out.println(LogUI.C_WARN.getValue() + "==========================================" + LogUI.E_NORMAL.getValue());
                                    System.out.println(LogUI.C_WARN.getValue() + "TheRouter:: FlowTask::   " + LogUI.E_NORMAL.getValue());
                                    System.out.println(LogUI.C_WARN.getValue() + "Can not found Task: [" + it + "] from " + taskName + " dependsOn" + LogUI.E_NORMAL.getValue());
                                    System.out.println(LogUI.C_WARN.getValue() + "==========================================" + LogUI.E_NORMAL.getValue());
                                    System.out.println();
                                }
                            }
                        }
                }
            }

            for (String it : flowTaskDependMap.keySet()) {
                Utils.fillTodoList(flowTaskDependMap, it);
            }

            if (Boolean.valueOf(Utils.getLocalProperty(mProject, TheRouterPlugin.SHOW_FLOW_DEPEND))) {
                for (String it : flowTaskDependMap.keySet()) {
                    Utils.fillNode(Utils.createNode(flowTaskDependMap, it), null);
                }

                System.out.println();
                System.out.println(LogUI.C_WARN.getValue() + "TheRouter:: FlowTask::dependency   " + LogUI.E_NORMAL.getValue());
                System.out.println(LogUI.C_WARN.getValue() + "==========================================" + LogUI.E_NORMAL.getValue());
                ArrayList<String> dependStackList = new ArrayList<>(Utils.dependStack);
                Collections.sort(dependStackList);
                for (String it : dependStackList) {
                    System.out.println(LogUI.C_WARN.getValue() + "[Root --> " + it + "]" + LogUI.E_NORMAL.getValue());
                }
                System.out.println(LogUI.C_WARN.getValue() + "==========================================" + LogUI.E_NORMAL.getValue());
                System.out.println();
            }

            System.out.println("---------TheRouter check flow task map--------------");

        }
    }
}
