package com.therouter.plugin;

import com.android.build.api.transform.*;
import com.android.build.gradle.internal.pipeline.TransformManager;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;

import org.apache.commons.io.FileUtils;
import org.gradle.api.Project;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import groovy.lang.PropertyValue;
import kotlin.io.FilesKt;

/**
 * Created by ZhangTao on 18/2/24.
 */

public class TheRouterTransform extends Transform {

    private final Map<String, String> buildProperties = new HashMap<>();

    private Project mProject;
    private boolean isLibrary;

    private static final String FILE_THEROUTER_JAR = "therouter_jar";
    private static final String FILE_ALL_CLASS = "all_class";
    private static final String FILE_DELETE_CLASS = "delete_class";

    public TheRouterTransform(Project p, boolean isLibrary) {
        this.mProject = p;
        this.isLibrary = isLibrary;
    }

    @Override
    public String getName() {
        return "TheRouter";
    }

    @Override
    public Set<QualifiedContent.ContentType> getInputTypes() {
        return TransformManager.CONTENT_CLASS;
    }

    @Override
    public Set<QualifiedContent.Scope> getScopes() {
        HashSet<QualifiedContent.Scope> set = new HashSet<>();
        set.add(QualifiedContent.Scope.PROJECT);
        return set;
    }

    @Override
    public boolean isIncremental() {
        return Boolean.valueOf(getLocalProperty(TheRouterPlugin.INCREMENTAL));
    }

    @Override
    public void transform(Context context, Collection<TransformInput> inputs,
                          Collection<TransformInput> referencedInputs,
                          TransformOutputProvider outputProvider,
                          boolean isIncremental)
            throws IOException, TransformException, InterruptedException {
        theRouterTransform(isIncremental, inputs, outputProvider);
    }

    private void theRouterTransform(boolean isIncremental, Collection<TransformInput> inputs, TransformOutputProvider outputProvider) throws IOException, URISyntaxException {
        System.out.println("TheRouter编译插件：" + LogUI.C_BLACK_GREEN.getValue() + "cn.therouter:" + BuildConfig.NAME
                + ":" + BuildConfig.VERSION + LogUI.E_NORMAL.getValue());
        System.out.println("当前编译 JDK Version 为::" + System.getProperty("java.version"));
        System.out.println("本次是增量编译::" + isIncremental);
        System.out.println("CHECK_ROUTE_MAP::" + getLocalProperty(TheRouterPlugin.CHECK_ROUTE_MAP));
        System.out.println("CHECK_FLOW_UNKNOW_DEPEND::" + getLocalProperty(TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND));

        long startFirst = System.currentTimeMillis();
        long start = System.currentTimeMillis();
        final Set<String> allClass = new HashSet<>();
        final Set<String> deletedClass = new HashSet<>();
        final Set<String> routeMapStringSet = new HashSet<>();
        final Map<String, String> flowTaskMap = new HashMap<>();
        System.out.println("---------TheRouter handle " + mProject.getName() + " module start-------------------------");
        if (!isIncremental) {
            outputProvider.deleteAll();
        }
        File therouterFile = null;
        for (TransformInput input : inputs) {
            // 从1.2.3版本开始，只遍历源码，遍历jar包是为了把源码的改动通过字节码写入TheRouter注入类
            if (!isLibrary) {
                // 遍历jar包
                for (JarInput jarInput : input.getJarInputs()) {
                    String jarName = jarInput.getName().toLowerCase();
                    File dest = outputProvider.getContentLocation(jarName, jarInput.getContentTypes(),
                            jarInput.getScopes(), Format.JAR);
                    if (TheRouterInjects.tagJar(jarInput.getFile())) {
                        System.out.println("=========" + dest.getAbsolutePath());
                        therouterFile = dest;
                        FileUtils.write(createCacheFile(mProject, FILE_THEROUTER_JAR), dest.getAbsolutePath(), "UTF-8", false);
                    }
                }
            }
            // 从1.2.3版本开始，只遍历源码
            for (DirectoryInput directoryInput : input.getDirectoryInputs()) {
                File dir = directoryInput.getFile();
                File dest = outputProvider.getContentLocation(directoryInput.getName(),
                        directoryInput.getContentTypes(), directoryInput.getScopes(), Format.DIRECTORY);
                FileUtils.forceMkdir(dest);
                String srcDirPath = dir.getAbsolutePath();
                String destDirPath = dest.getAbsolutePath();
                if (isIncremental) {
                    Map<File, Status> fileStatusMap = directoryInput.getChangedFiles();
                    for (Map.Entry<File, Status> changedFile : fileStatusMap.entrySet()) {
                        Status status = changedFile.getValue();
                        File inputFile = changedFile.getKey();
                        String destFilePath = inputFile.getAbsolutePath().replace(srcDirPath, destDirPath);
                        File destFile = new File(destFilePath);
                        switch (status) {
                            case REMOVED:
                                deletedClass.add(inputFile.getAbsolutePath().replaceAll("/", "."));
                                if (destFile.exists()) {
                                    destFile.delete();
                                }
                                break;
                            case ADDED:
                            case CHANGED:
                                SourceInfo sourceInfo = TheRouterInjects.tagClass(inputFile.getAbsolutePath());
                                routeMapStringSet.addAll(sourceInfo.routeMapStringFromSource);
                                flowTaskMap.putAll(sourceInfo.flowTaskMapFromSource);
                                FileUtils.copyFile(inputFile, destFile);
                                break;
                            default:
                                break;
                        }
                    }
                } else {
                    SourceInfo sourceInfo = TheRouterInjects.tagClass(directoryInput.getFile().getAbsolutePath());
                    allClass.addAll(sourceInfo.allSourceClass);
                    routeMapStringSet.addAll(sourceInfo.routeMapStringFromSource);
                    flowTaskMap.putAll(sourceInfo.flowTaskMapFromSource);
                    if (dest.exists()) {
                        FileUtils.forceDelete(dest);
                    }
                    FileUtils.copyDirectory(directoryInput.getFile(), dest);
                }
            }
        }

        if (isLibrary) {
            FileUtils.write(createCacheFile(mProject, FILE_ALL_CLASS), set2String(allClass), "UTF-8", false);
            FileUtils.write(createCacheFile(mProject, FILE_DELETE_CLASS), set2String(deletedClass), "UTF-8", false);
        } else {
            for (Project it : mProject.getRootProject().getSubprojects()) {
                List<String> strings = FileUtils.readLines(createCacheFile(it, FILE_ALL_CLASS), "UTF-8");
                for (String line : strings) {
                    allClass.add(line.trim());
                }
                List<String> strings2 = FileUtils.readLines(createCacheFile(it, FILE_DELETE_CLASS), "UTF-8");
                for (String line : strings2) {
                    deletedClass.add(line.trim());
                }
            }

            if (therouterFile != null) {
                TheRouterInjects.injectClassCode(therouterFile, isIncremental);
            } else {
                throw new RuntimeException("未找到therouter依赖");
            }

            long time = System.currentTimeMillis() - start;
            System.out.println("---------TheRouter ASM, spend：${time}ms----------------------");
            start = System.currentTimeMillis();

            Set<RouteItem> pageSet = new HashSet<>();
            Gson gson = new GsonBuilder().setPrettyPrinting().create();
            for (String it : routeMapStringSet) {
                pageSet.addAll(gson.fromJson(it, new TypeToken<List<RouteItem>>() {
                }.getType()));
            }
            // 让第三方Activity也支持路由，第三方页面的路由表可以在assets中添加
            File assetRouteMap = new File(mProject.getProjectDir(), "src/main/assets/therouter/routeMap.json");
            if (assetRouteMap.exists()) {
                if (TheRouterPlugin.DELETE.equalsIgnoreCase(getLocalProperty(TheRouterPlugin.CHECK_ROUTE_MAP))) {
                    System.out.println("---------TheRouter delete route map------------------------------------------");
                    assetRouteMap.delete();
                    assetRouteMap.createNewFile();
                } else {
                    String assetString = FileUtils.readFileToString(assetRouteMap);
                    System.out.println("---------TheRouter get route map from: /assets/therouter/routeMap.json-------")
                    try {
                        List<RouteItem> assetsList = gson.fromJson(assetString, new TypeToken<List<RouteItem>>() {
                        }.getType());
                        for (RouteItem item : assetsList) {
                            if (!pageSet.contains(item)) {
                                pageSet.add(item);
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            } else {
                System.out.println("---------TheRouter route map does not exist: /assets/therouter/routeMap.json-------");
                assetRouteMap.getParentFile().mkdirs();
                assetRouteMap.createNewFile();
            }

            Map<String, List<RouteItem>> result = new HashMap<>();
            // 检查url合法性
            for (RouteItem routeItem : pageSet) {
                String url = routeItem.path;
                if (url.contains("?")) {
                    URI uri = new URI(routeItem.path);
                    Map<String, Object> map = uri.getProperties();
                    for (String key : map.keySet()) {
                        routeItem.params.put(key, map.get(key).toString());
                    }
                    url = url.substring(0, url.indexOf('?'));
                }
                List<RouteItem> routeList = result.get(url);
                if (routeList == null) {
                    routeList = new ArrayList<>();
                    routeList.add(routeItem);
                    result.put(url, routeList);
                }
            }
            // 检查路由表合法性
            result.values().each {
                String className = null
                it.each {
                    routeItem ->
                        if (className == null) {
                            className = routeItem.className
                        } else if (className != routeItem.className) {
                            throw new RuntimeException("Multiple Activity to single Url: $className and ${routeItem.className}")
                        }
                        if (!getLocalProperty(TheRouterPlugin.CHECK_ROUTE_MAP).isEmpty()) {
                            // 只在全量编译时检验
                            if (!isIncremental) {
                                boolean classNotFound = true
                                for (String item : allClass) {
                                    if (item.contains(routeItem.className)) {
                                        classNotFound = false
                                        break
                                    }
                                }
                                if (classNotFound) {
                                    if (TheRouterPlugin.ERROR.equalsIgnoreCase(getLocalProperty(TheRouterPlugin.CHECK_ROUTE_MAP))) {
                                        throw new ClassNotFoundException(routeItem.className + " in /assets/therouter/routeMap.json")
                                    } else if (TheRouterPlugin.WARNING.equalsIgnoreCase(getLocalProperty(TheRouterPlugin.CHECK_ROUTE_MAP))) {
                                        System.out.println("${LogUI.C_WARN.value}[${routeItem.className} in /assets/therouter/routeMap.json]${LogUI.E_NORMAL.value}")
                                    }
                                }
                            }
                        }
                }
            }

            List<RouteItem> pageList;
            if (isIncremental && !deletedClass.isEmpty()) {
                pageList = new ArrayList<>(pageSet.size());
                for (RouteItem item : pageSet) {
                    boolean needDelete = false;
                    for (String deletedClassItem : deletedClass) {
                        if (deletedClassItem.contains(item.className)) {
                            needDelete = true;
                            break;
                        }
                    }
                    if (!needDelete) {
                        pageList.add(item);
                    }
                }
            } else {
                pageList = new ArrayList<>(pageSet);
            }
            Collections.sort(pageList);
            String json = gson.toJson(pageList);
            FileUtils.write(assetRouteMap, json, false);
            time = System.currentTimeMillis() - start;
            System.out.println("---------TheRouter create new route map, spend:" + time + "ms--------------");
            start = System.currentTimeMillis();


            Map<String, Set<String>> flowTaskDependMap = new HashMap<>();
            flowTaskMap.keySet().each {
                Set<String> value = flowTaskDependMap.get(it);
                if (value == null) {
                    value = new HashSet<>();
                }
                String dependsOn = flowTaskMap.get(it);
                if (!dependsOn.isBlank()) {
                    dependsOn.split(",").each {
                        depend ->
                            if (!depend.isBlank()) {
                                value.add(depend.trim());
                            }
                    }
                }
                flowTaskDependMap.put(it, value)
            }

            if (!getLocalProperty(TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND).isEmpty()) {
                flowTaskDependMap.values().each {
                    taskName ->
                        flowTaskDependMap[taskName].each {
                            if (!flowTaskDependMap.containsKey(it)) {
                                if (TheRouterPlugin.ERROR.equalsIgnoreCase(getLocalProperty(TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND))) {
                                    throw new RuntimeException("\n\n==========================================" +
                                            "\nTheRouter:: FlowTask::   " +
                                            "\nCan not found Task: [$it] from $taskName dependsOn" +
                                            "\n==========================================\n\n")
                                } else if (TheRouterPlugin.ERROR.equalsIgnoreCase(getLocalProperty(TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND))) {
                                    System.out.println();
                                    System.out.println("${LogUI.C_WARN.value}" + "==========================================" + "${LogUI.E_NORMAL.value}")
                                    System.out.println("${LogUI.C_WARN.value}" + "TheRouter:: FlowTask::   " + "${LogUI.E_NORMAL.value}")
                                    System.out.println("${LogUI.C_WARN.value}" + "Can not found Task: [$it] from $taskName dependsOn" + "${LogUI.E_NORMAL.value}")
                                    System.out.println("${LogUI.C_WARN.value}" + "==========================================" + "${LogUI.E_NORMAL.value}")
                                    System.out.println();
                                }
                            }
                        }
                }
            }

            flowTaskDependMap.keySet().each {
                fillTodoList(flowTaskDependMap, it)
            }

            if (Boolean.valueOf(getLocalProperty(TheRouterPlugin.SHOW_FLOW_DEPEND))) {
                flowTaskDependMap.keySet().each {
                    fillNode(createNode(flowTaskDependMap, it), null)
                }

                System.out.println()
                System.out.println("${LogUI.C_WARN.value}" + "TheRouter:: FlowTask::dependency   " + "${LogUI.E_NORMAL.value}")
                System.out.println("${LogUI.C_WARN.value}" + "==========================================" + "${LogUI.E_NORMAL.value}")
                dependStack.sort().each {
                    System.out.println("${LogUI.C_WARN.value}" + "[Root --> $it]" + "${LogUI.E_NORMAL.value}")
                }
                System.out.println("${LogUI.C_WARN.value}" + "==========================================" + "${LogUI.E_NORMAL.value}")
                System.out.println();
            }

            time = System.currentTimeMillis() - start;
            System.out.println("---------TheRouter check flow task map, spend:" + time + "ms--------------");

            time = System.currentTimeMillis() - startFirst;
            System.out.println("---------TheRouter handle " + mProject.getName() + " module finish, spend:" + time + "ms-------------------------");
        }
    }

    private final List<String> loopDependStack = new ArrayList<>();

    private void fillTodoList(Map<String, Set<String>> map, String root) {
        Set<String> dependsSet = map.get(root);
        if (dependsSet != null && !dependsSet.isEmpty()) {
            if (loopDependStack.contains(root)) {
                throw new RuntimeException("\n\n==========================================" +
                        "\nTheRouter:: FlowTask::   " +
                        "\nCyclic dependency: [" + getLog(loopDependStack, root) + "]" +
                        "\n==========================================\n\n");
            }
            loopDependStack.add(root);
            for (String depend : dependsSet) {
                fillTodoList(map, depend);
            }
            loopDependStack.remove(root);
        }
    }

    Set<String> dependStack = new HashSet<>();

    private void fillNode(Node node, String root) {
        if (node.getChildren() == null || node.getChildren().isEmpty()) {
            if (root == null) {
                dependStack.add(node.getName());
            } else {
                dependStack.add(node.getName() + " --> " + root);
            }
        } else {
            for (Node it : node.getChildren()) {
                if (root == null) {
                    fillNode(it, node.getName());
                } else {
                    fillNode(it, node.getName() + " --> " + root);
                }
            }
        }
    }

    private Node createNode(Map<String, Set<String>> map, String root) {
        final Node node = new Node(root);
        Set<Node> childrenNode = new HashSet<>();
        Set<String> dependsSet = map.get(root);
        if (dependsSet != null && !dependsSet.isEmpty()) {
            for (String depend : dependsSet) {
                childrenNode.add(createNode(map, depend));
            }
        }
        node.setChildren(childrenNode);
        return node;
    }

    private String set2String(Set<String> set) {
        StringBuilder stringBuilder = new StringBuilder();
        for (String item : set) {
            stringBuilder.append(item.trim()).append("\n");
        }
        return stringBuilder.toString();
    }

    private File createCacheFile(Project project, String name) {
        File folder = new File(project.getBuildDir().getAbsolutePath() + "/generated/therouter");
        folder.mkdirs();
        File cache = new File(folder, name);
        if (cache.exists()) {
            cache.delete();
        }
        try {
            cache.createNewFile();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return cache;
    }

    private String getLog(List<String> list, String root) {
        if (list == null || list.isEmpty()) {
            return "";
        }
        StringBuilder stringBuilder = new StringBuilder();
        for (String task : list) {
            stringBuilder.append(task).append("-->");
        }
        if (root != null) {
            stringBuilder.append(root);
        }
        return stringBuilder.toString();
    }

    String getLocalProperty(String key) {
        try {
            if (!buildProperties.containsKey(key)) {
                initProperties();
            }
            String value = buildProperties.get(key);
            return value == null ? "" : value;
        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
    }

    void initProperties() {
        File gradlePropertiesFile;
        try {
            gradlePropertiesFile = new File(mProject.getRootDir(), "gradle.properties");
            if (!gradlePropertiesFile.exists()) {
                gradlePropertiesFile = new File("../gradle.properties");
            }
        } catch (Exception e) {
            gradlePropertiesFile = new File("../gradle.properties");
        }
        Properties gradleProperties = new Properties();
        try {
            gradleProperties.load(new FileInputStream(gradlePropertiesFile));
        } catch (Exception e) {
            e.printStackTrace();
        }
        buildProperties.put(TheRouterPlugin.CHECK_ROUTE_MAP, gradleProperties.getProperty(TheRouterPlugin.CHECK_ROUTE_MAP));
        buildProperties.put(TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND, gradleProperties.getProperty(TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND));
        buildProperties.put(TheRouterPlugin.SHOW_FLOW_DEPEND, gradleProperties.getProperty(TheRouterPlugin.SHOW_FLOW_DEPEND));
        buildProperties.put(TheRouterPlugin.INCREMENTAL, gradleProperties.getProperty(TheRouterPlugin.INCREMENTAL));

        File localPropertiesFile;
        try {
            localPropertiesFile = new File(mProject.getRootDir(), "local.properties");
            if (!localPropertiesFile.exists()) {
                localPropertiesFile = new File("../local.properties");
            }
        } catch (Exception e) {
            localPropertiesFile = new File("../local.properties");
        }
        Properties localProperties = new Properties();
        try {
            localProperties.load(new FileInputStream(localPropertiesFile));
        } catch (Exception e) {
            e.printStackTrace();
        }
        String v = localProperties.getProperty(TheRouterPlugin.CHECK_ROUTE_MAP);
        if (v != null && !v.isEmpty()) {
            buildProperties.put(TheRouterPlugin.CHECK_ROUTE_MAP, v);
        }
        v = localProperties.getProperty(TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND);
        if (v != null && !v.isEmpty()) {
            buildProperties.put(TheRouterPlugin.CHECK_FLOW_UNKNOW_DEPEND, v);
        }
        v = localProperties.getProperty(TheRouterPlugin.SHOW_FLOW_DEPEND);
        if (v != null && !v.isEmpty()) {
            buildProperties.put(TheRouterPlugin.SHOW_FLOW_DEPEND, v);
        }
        v = localProperties.getProperty(TheRouterPlugin.INCREMENTAL);
        if (v != null && !v.isEmpty()) {
            buildProperties.put(TheRouterPlugin.INCREMENTAL, v);
        }
    }
}
