package cn.openx.boot.framework.booster.service;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.openx.boot.framework.booster.constant.DBConstant;
import cn.openx.boot.framework.booster.domian.enums.CommEnum;
import cn.openx.boot.framework.booster.entity.BaseEntity;
import cn.openx.boot.framework.booster.util.dto.DtoUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.common.Mapper;
import tk.mybatis.mapper.entity.Example;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

/**
 * @author zhanls 2021/11/4 00:26
 * @version 1.0
 * @Description - 契合表设计规范的基础CURD类，可以满足大部分基本CURD需求，所有ServiceImpl需要实现此类
 **/
@Slf4j
public class BaseCurdService<T> {

    @Autowired
    public Mapper<T> mapper;

    /**
     * 功能描述: 获取当前用户
     * @return: java.lang.String
     * @version: 1.0 2021/11/4 22:39 zhanls
     */
    public String getCurrentUser() {
        // TODO 获取当前用户名称，否则 noLoginUser
        return "noLoginUser";
    }

    /**
     * ID 生成策略
     *
     * @return: ID
     * @version: 1.0 2021/11/4 22:33 zhanls
     */
    public String generateId() {
        return IdUtil.simpleUUID();
    }

    /**
     * 通用新增服务
     *
     * @param t 待保存的业务类
     * @return -1 标识失败 否则标识成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseInsert(T t) {
        if (t instanceof BaseEntity) {
            //允许自己生成id
            if (StrUtil.isBlank(((BaseEntity) t).getId())) {
                ((BaseEntity) t).setId(generateId());
            }
            setBaseData((BaseEntity) t);
            return mapper.insertSelective(t);
        }
        return -1;
    }


    /**
     * 通用新增服务,支持自定义ID
     *
     * @param t  待保存的业务类
     * @param id 自定义主键
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseInsert(T t, String id) {
        if (t instanceof BaseEntity) {
            if (StrUtil.isNotEmpty(id)) {
                ((BaseEntity) t).setId(id);
            }
            return baseInsert(t);
        }
        return -1;
    }

    /**
     * 新增或者保存baseEntity通用服务
     *
     * @param t    业务类
     * @param uuId 业务主键
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     */
    public Integer insertOrUpdate(T t, String uuId) {
        if (t instanceof BaseEntity) {
            String id = ((BaseEntity) t).getId();
            if (StrUtil.isNotBlank(id)) {
                return baseUpdateByPK(t);
            } else {
                return baseInsert(t, uuId);
            }
        }
        return -1;
    }

    /**
     * 通用删除服务：物理删除
     *
     * @param  t 物理删除
     * @return
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
//    public Integer deleteByPrimaryKey(T t) {
//        return  mapper.deleteByPrimaryKey(t);
//    }

    /**
     * 通用删除服务：根据业务类主键进行删除
     *
     * @param t  业务类
     * @param id 业务主键
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseDeleteByPK(String id, T t) {
        if (StrUtil.isNotBlank(id)) {
            ((BaseEntity) t).setId(id);
        }
        return baseDeleteByPK(t);
    }

    /**
     * 通用删除服务：根据业务类主键进行删除
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseDeleteByPK(T t) {
        if (t instanceof BaseEntity) {
            setDeleteDataInfo((BaseEntity) t);
            return updateById(t);
        }
        return -1;
    }


    /**
     * 通用删除服务:按传入的字段和值，删除对应的数据
     *
     * @param t     业务类
     * @param field 按这个字段删除数据
     * @param value 字段的值
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseDeleteByfields(T t, String field, String value) {
        if (t instanceof BaseEntity) {
            setDeleteDataInfo((BaseEntity) t);
            Example example = new Example(t.getClass(), true, true);
            example.createCriteria().andEqualTo(field, value);
            return mapper.updateByExampleSelective(t, example);
        }
        return -1;
    }

    /**
     * 通用更新服务:按传入业务类的主键对业务类进行更新
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseUpdateByPK(T t) {
        if (t instanceof BaseEntity) {
            return updateById(t);
        }
        return -1;
    }

    /**
     * 通用更新服务:按传入业务类的主键对业务类进行更新（null值会被更新）
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseUpdateAllByPK(T t) {
        if (t instanceof BaseEntity) {
            setModifyData((BaseEntity) t);
            Example example = new Example(t.getClass(), true, true);
            example.createCriteria().andEqualTo(DBConstant.PROPERTIES_ID, ((BaseEntity) t).getId());
            return mapper.updateByExample(t, example);
        }
        return -1;
    }

    /**
     * 私有：mapper更新新服务:按传入业务类的主键对业务类进行更新
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    private Integer updateById(T t) {
        setModifyData((BaseEntity) t);
        Example example = new Example(t.getClass(), true, true);
        example.createCriteria().andEqualTo(DBConstant.PROPERTIES_ID, ((BaseEntity) t).getId());
        return mapper.updateByExampleSelective(t, example);
    }

    /**
     * 通用失效服务:按传入业务类的主键对业务类进行失效操作
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseDisableByPK(T t) {
        if (t instanceof BaseEntity) {
            ((BaseEntity) t).setIsValid(DBConstant.VALUE_INVALID);
            return updateById(t);
        }
        return -1;
    }

    /**
     * 通用失效服务:按传入业务类的主键对业务类进行失效操作
     *
     * @param t  业务类
     * @param id 业务类主键
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseDisableByPK(String id, T t) {
        if (StrUtil.isNotBlank(id)) {
            ((BaseEntity) t).setId(id);
        }
        return baseDisableByPK(t);
    }

    /**
     * 通用生效服务:按传入业务类的主键对业务类进行生效操作
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseEnableByPK(T t) {
        if (t instanceof BaseEntity) {
            ((BaseEntity) t).setIsValid(DBConstant.VALUE_VALID);
            return updateById(t);
        }
        return -1;
    }

    /**
     * 通用生效服务:按传入业务类的主键对业务类进行生效操作
     *
     * @param t  业务类
     * @param id 业务主键
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public Integer baseEnableByPK(String id, T t) {
        if (StrUtil.isNotBlank(id)) {
            ((BaseEntity) t).setId(id);
        }
        return baseEnableByPK(t);
    }

    /**
     * 通用查询单体服务:按传入业务类的主键对业务类进行(未删除)单体业务查询操作
     * 注：根据实体中的属性进行查询，只能有一个返回值，有多个结果是抛出异常，查询条件使用等号
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public T baseSelectByPK(T t) {
        ((BaseEntity) t).setIsDelete(DBConstant.VALUE_NOT_DELTED);
        return baseSelectByPKAndNotCareOthers(t);
    }

    /**
     * 通用查询单体服务:按传入业务类的主键对业务类进行(不care是否有效、删除)单体业务查询操作
     * 注：根据实体中的属性进行查询，只能有一个返回值，有多个结果是抛出异常，查询条件使用等号
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public T baseSelectByPKAndNotCareOthers(T t) {
        if (t instanceof BaseEntity) {
            return mapper.selectOne(t);
        }
        return null;
    }


    /**
     * 通用查询列表服务:传入的对象的非空属性为条件查询列表，按创建日期倒序排序
     * 注：允许返回被逻辑删除的数据
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    protected List<T> baseSelectListByObjectAllowDelete(T t) {
        T changeT = DtoUtils.setNullValue(t);
        List<T> list = new ArrayList<>();
        if (t instanceof BaseEntity) {
            Example example = getExample(changeT);
            example.setOrderByClause(DBConstant.FIELD_DATE_CREATE + " " + DBConstant.ORDER_DESC);
            list = mapper.selectByExample(example);
        }
        return list;
    }

    /**
     * 通用查询列表服务:传入的对象的非空属性为条件查询列表，按创建日期倒序排序
     * 注：不允许返回被逻辑删除的数据
     *
     * @param t 业务类
     * @return -1 标识失败,反之成功
     * @version: 1.0 2021/11/4 22:33 zhanls
     **/
    public List<T> baseSelectListByObject(T t) {
        ((BaseEntity) t).setIsDelete(DBConstant.VALUE_NOT_DELTED);
        return baseSelectListByObjectAllowDelete(t);
    }


    /**
     * 以传入的对象的非空属性为条件查询列表，按创建日期正序或到倒序排序
     *
     * @param t             业务类
     * @param orderByClause 升序或者降序
     * @return list 业务列表
     * @version: 1.0 2021/11/4 22:33 zhanls
     */
    public List<T> baseSelectListByObjectOrderByClause(T t, String orderByClause) {
        List<T> list = new ArrayList<>();
        if (t instanceof BaseEntity) {
            ((BaseEntity) t).setIsDelete(DBConstant.VALUE_NOT_DELTED);
            Example example = getExample(t);
            example.setOrderByClause(orderByClause);
            list = mapper.selectByExample(example);
        }
        return list;
    }


    /**
     * 通用代码，获取t内非空属性
     *
     * @param t 业务类
     * @return example 查询对象
     * @version: 1.0 2021/11/4 22:33 zhanls
     */
    private Example getExample(T t) {
        BaseEntity baseEntity = (BaseEntity) t;
        String searchType = baseEntity.getSearchType();
        Example example = new Example(t.getClass(), true, true);
        Example.Criteria criteria = example.createCriteria();
        Field[] allFields = FieldUtils.getAllFields(t.getClass());

        // 获取t内非空属性
        for (Field field : allFields) {
            field.setAccessible(true);
            try {
                if (ObjectUtil.isNotNull(field.get(t)) && (!"serialVersionUID".equals(field.getName())) && (!"searchType".equals(field.getName()))) {
                    Object fieldObj = field.get(t);
                    if (StrUtil.isBlank(searchType) || !(fieldObj instanceof String)) {
                        criteria.andEqualTo(field.getName(), fieldObj);
                        continue;
                    }
                    //动态获取模糊匹配的匹配模式
                    criteria.andLike(field.getName(), getSearchStr(fieldObj, searchType));
                }
            } catch (IllegalAccessException e) {
                log.error("通用代码，获取t内非空属性，发生异常{}", e);
            }
        }
        return example;
    }

    /**
     * 通用代码，查询单个对象
     *
     * @param t 业务类
     * @return t 业务类
     * @version: 1.0 2021/11/18 23:10 zhanls
     */
    public T baseSelectOneByObject(T t) {
        if (t instanceof BaseEntity) {
            ((BaseEntity) t).setIsDelete(DBConstant.VALUE_NOT_DELTED);
            return mapper.selectOne(t);
        } else {
            /*非法的实例类型是否抛出异常后续考虑*/
            return null;
        }

    }

    /**
     * 私有：统一设置创建人、创建时间
     * @param t 业务类
     */
    private void setBaseData(BaseEntity t) {
        t.setCreateOpr(getCurrentUser());
        t.setDateCreate(DateUtil.date());
        t.setIsDelete(CommEnum.NO.getCode());
        t.setIsValid(CommEnum.YES.getCode());
        setModifyData(t);
    }

    /**
     * 私有：统一设置修改人、修改时间
     * @param t 业务类
     */
    private void setModifyData(BaseEntity t) {
        t.setModifyOpr(getCurrentUser());
        t.setDateModify(DateUtil.date());
    }

    /**
     * 私有：统一设置为已失效、已删除状态
     * @param t 业务类
     **/
    private void setDeleteData(BaseEntity t) {
        t.setIsValid(DBConstant.VALUE_INVALID);
        t.setIsDelete(DBConstant.VALUE_DELETED);
    }


    /**
     * 统一设置为已失效、已删除状态并更新修改人等信息
     * @param t 业务类
     **/
    private void setDeleteDataInfo(BaseEntity t) {
        setDeleteData(t);
        setModifyData(t);
    }

    /**
     * 私有：动态获取模糊匹配的匹配模式
     *
     * @param
     * @return
     **/
    private String getSearchStr(Object fieldObj, String searchType) {
        String fieldStr = String.valueOf(fieldObj);
        //模糊查询机制处理
        switch (searchType) {
            case DBConstant.LIKE:
                fieldStr = DBConstant.LIKE_SYMBOL.concat(fieldStr).concat(DBConstant.LIKE_SYMBOL);
                break;
            case DBConstant.LEFT_LIKE:
                fieldStr = DBConstant.LIKE_SYMBOL.concat(fieldStr);
                break;
            case DBConstant.RIGHT_LIKE:
                fieldStr = fieldStr.concat(DBConstant.LIKE_SYMBOL);
                break;
            default:
                fieldStr = fieldStr.concat(DBConstant.LIKE_SYMBOL);
                log.error("未能识别的searcheType：{}", searchType);
                break;
        }
        return fieldStr;
    }

}
