package cn.openx.boot.framework.booster.web.result;

import cn.hutool.core.util.NumberUtil;
import cn.openx.boot.framework.booster.web.code.BaseCode;
import cn.openx.boot.framework.booster.web.exception.ApiException;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;

import java.beans.ConstructorProperties;

/**
 * @author zhanls 2021/11/3 23:54
 * @version 1.0
 * @Description - API结果类，所有WEB请求统一使用此类作为返回值
 **/
@EqualsAndHashCode(callSuper = true)
@ToString(callSuper = true)
@NoArgsConstructor
@Slf4j
public class ApiResult<E> extends ApiCoreResult<E> {

    public static <E> ApiResult<E> ok() {
        return ok(null);
    }

    public static <E> ApiResult<E> ok(E data) {
        return build(BaseCode.SUCCESS, data);
    }

    public static <E> ApiResult<E> err(BaseCode code) {
        return err(code, code.getMessage());
    }

    public static <E> ApiResult<E> err(BaseCode code, String message) {
        return err(code.getCode(), message);
    }

    public static <E> ApiResult<E> err(String message) {
        return err(BaseCode.SERVER_ERROR.getCode(), message);
    }

    public static <E> ApiResult<E> err(String message, String data) {
        /**兼容之前  code message 模式 */
        if (NumberUtil.isNumber(message)) {
            BaseCode baseCode = BaseCode.parse(Integer.valueOf(message));
            if(!BaseCode.SERVER_ERROR.getCode().equals(baseCode.getCode())) {
                return (ApiResult<E>) build(baseCode.getCode(), baseCode.getMessage(), data);
            }else{
                return (ApiResult<E>) build(message, data, data);
            }
        } else {
            return (ApiResult<E>) build(BaseCode.SERVER_ERROR.getCode(), message, data);
        }
    }

    public static <E> ApiResult<E> build(BaseCode code, E data) {
        return build(code.getCode(), code.getMessage(), data);
    }

    public static <E> ApiResult<E> build(String code, String message, E data) {
        return new ApiResult<>(code, message, data);
    }

    public static <E> ApiResult<E> err(String message, E data) {
        return build(BaseCode.SERVER_ERROR.getCode(), message, data);
    }

    public static <E> ApiResult<E> err(BaseCode code, E data) {
        return build(code.getCode(), code.getMessage(), data);
    }

    public static <E> ApiResult<E> build(BaseCode code) {
        return build(code, null);
    }

    public static <E> ApiResult<E> with(Throwable e) {
        if (e != null && e instanceof ApiException) {
            return err(((ApiException) e).getCode(), e.getMessage());
        }
        if (e != null) {
            return err(BaseCode.SERVER_ERROR, e.getMessage());
        }
        return err(BaseCode.SERVER_ERROR);
    }

    @ConstructorProperties({"code", "message", "data"})
    public ApiResult(String code, String message, E data) {
        super(code, message, data);
        /**err 记录日志*/
        if (!BaseCode.SUCCESS.getCode().equals(code)) {
            log.error("message : {} ", message);
            log.error("data: {} ", data);
        }
    }

}
