package cn.openx.boot.framework.booster.util.date;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import static cn.openx.boot.framework.booster.constant.Constants.PATTERN_DATE;
import static cn.openx.boot.framework.booster.constant.Constants.PATTERN_DATE_TIME;


/**
 * @date: 2018/7/3
 * @description 日期工具类
 */
public class DateUtils {

    @SuppressWarnings("unused")
    private static final int[] days = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
    public static final String day = "d";
    public static final String week = "w";
    public static final String month = "m";
    public static final String year = "y";
    private static final ThreadLocal<DateFormat> DATETIME_FORMAT = new ThreadLocal<DateFormat>(){
        @Override
        protected DateFormat initialValue() {
            return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        }
    };
    private static final ThreadLocal<DateFormat> DATE_FORMAT = new ThreadLocal<DateFormat>(){
        @Override
        protected DateFormat initialValue() {
            return new SimpleDateFormat("yyyy-MM-dd");
        }
    };

    private static final ThreadLocal<DateFormat> ACCULATE_TIME_FORMAT = new ThreadLocal<DateFormat>(){
        @Override
        protected DateFormat initialValue() {
            return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss:SSS");
        }
    };
    private DateUtils() {
    }

    /**
     * 得到当前日期的整数形式,yyyyMMdd。
     * @return 当前日期的整数形式
     */
    public static int getDate() {
        Calendar cal = Calendar.getInstance();
        return cal.get(Calendar.YEAR) * 10000 + (cal.get(Calendar.MONTH) + 1)
                * 100 + cal.get(Calendar.DAY_OF_MONTH);
    }

    /**
     * 返回时间 格式:时/分/秒 hhnnss
     *
     * @return 时间的整数形式
     */
    public static int getTime() {
        Calendar cal = Calendar.getInstance();
        int hour = cal.get(Calendar.HOUR_OF_DAY);
        int minute = cal.get(Calendar.MINUTE);
        int second = cal.get(Calendar.SECOND);
        return hour * 10000 + minute * 100 + second;
    }

    /**
     * 得到当前时间的java.util.Date类型的日期
     *
     * @return 日期
     */
    public static Date getNowTime() {
        return Calendar.getInstance().getTime();
    }

    /**
     * 得到当前时间的java.util.Date类型的日期,精确到日
     *
     * @return 日期对象
     */
    public static Date getNowDate() {
        Calendar now = Calendar.getInstance();
        return new GregorianCalendar(now.get(Calendar.YEAR),
                now.get(Calendar.MONTH), now.get(Calendar.DAY_OF_MONTH))
                .getTime();
    }


    /**
     * 在当前日期增加指定月数
     * 得到当前日期之后某月的日期,如当前日期2008年10月1日,2月后为2008年12月1日
     *
     * @param monthCount
     *            月份数
     * @return 日期
     */
    public static Calendar addMonth(int monthCount) {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.MONTH, cal.get(Calendar.MONDAY) + monthCount);
        return cal;
    }
    /**
     * 指定增加某种日期类型(如年、月、日、时、分、秒)的数目的,当前时间之后的某个时间
     * @param type
     *            时间类型,如年、月、日、时、分、秒
     *            eg:Calendar.MONTH Calendar.YEAR Calendar.DAY_OF_MONTH
     *            Calendar.HOUR Calendar.MINUTE Calendar.SECOND
     * @param amount
     *            时间数目
     * @return 日期
     */
    public static Calendar add(int type, int amount) {
        Calendar cal = Calendar.getInstance();
        cal.set(type, cal.get(type) + amount);
        return cal;
    }
    /**
     * 在指定日期增加指定月数
     * 得到当前日期之后某月的日期,如当前日期2008年10月1日,2月后为2008年12月1日
     *
     * @param monthCount
     *            月份数
     * @return 日期
     */
    public static Calendar addMonth(Date date, int monthCount) {
        Calendar cal = convertDateToCal(date);
        cal.set(Calendar.SECOND, cal.get(Calendar.MONDAY) + monthCount);
        return cal;
    }

    /**
     * 生成当前日期指定天数后的日期
     *
     * @param days
     *            指定天数
     * @return 数字形式的日期
     */
    public static Date addDayByNow(int days) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, days);
        return (Date)calendar.getTime();
    }

    /**
     * 根据传入日期生成指定天数后的日期
     *
     * @param date
     *            传入日期
     * @param days
     *            指定天数
     * @return 日期
     */
    public static Date addDay(Date date, int days) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DATE, days);
        return (Date) cal.getTime();
    }
    /**
     * 根据期限间隔,生成给定日期指定天数后的日期
     *
     * @param date
     *            给定日期
     * @param term
     *            (d,D,m,M,y,Y) 期限间隔
     * @return 指定天数后的日期
     */
    public static Date addDay(Date date, String term) {
        return addDay(date, countDay(date, term));
    }
    /**
     * 计算两日之间的间隔天数
     *
     * @param start
     *            开始日期
     * @param end
     *            结束日期
     * @return 间隔天数
     */
    public static int getDays(Date start, Date end) {
        Calendar startCal = Calendar.getInstance();
        Calendar endCal = Calendar.getInstance();
        startCal.setTime(start);
        endCal.setTime(end);
        return (int) ((endCal.getTime().getTime() - startCal.getTime()
                .getTime()) / (1000 * 60 * 60 * 24));
    }
    /**
     * 格式化输出
     *
     * @param cal
     *            日期对象
     * @param sformat
     *            输出格式
     * @return 字符串格式日期
     */
    public static String formatCalToStr(Calendar cal, String sformat) {
        DateFormat dateFormat=null;
        try {
            dateFormat = new SimpleDateFormat(sformat);
        } catch (IllegalArgumentException e) {
            dateFormat = DATETIME_FORMAT.get();
        }
        return dateFormat.format(cal == null ? Calendar.getInstance().getTime()
                : cal.getTime());
    }

    /**
     * 转换日期 Date -> 20080101
     *
     * @param date
     *            日期
     * @param format
     *            日期的字符串格式如yyyyMMdd
     * @return 日期的字符串形式如yyyyMMdd
     */
    public static String formatDateToStr(Date date, String format) {
        if (date == null){
            return null;
        }
        DateFormat sdf = format == null?
                DATETIME_FORMAT.get() :"yyyy-MM-dd".equals(format)?
                DATE_FORMAT.get():new SimpleDateFormat(format);
        if ("CNS".equals(format)) {
            sdf = DATE_FORMAT.get();
            String[] str = sdf.format(date).split("-");
            return str[0] + "年" + str[1] + "月" + str[2] + "日";
        }
        return sdf.format(date);
    }

    /**
     * 转换日期 20080101 -> Date
     *
     * @param dateStr
     *            日期字符串
     * @param format
     *            日期的字符串格式如yyyyMMdd
     * @return Date 日期
     * @throws ParseException
     *             日期解析异常
     */
    //static DateFormat testFormat=new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    public static Date formatStrToDate(String dateStr, String format) {
        if (dateStr == null || dateStr.length() == 0){
            return null;
        }

        DateFormat sdf = format == null?
                DATETIME_FORMAT.get() :"yyyy-MM-dd".equals(format)?
                DATE_FORMAT.get():new SimpleDateFormat(format);
        try {
            return sdf.parse(dateStr);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取精确到毫秒的时间
     *
     * @param date
     * @return
     */
    public static String getAcculateTime(Date date) {
        if (date == null){
            return "";
        }

        return ACCULATE_TIME_FORMAT.get().format(date);
    }

    /**
     * 简化日期型，去掉时间显示
     *
     * @param date
     * @return
     */
    public static String getSimpleDate(Date date) {
        if (date == null){
            return "";
        }

//		DateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
//		return sdf.format(date);
        return DATE_FORMAT.get().format(date);
    }

    /**
     * 简化日期型，去掉时间显示
     *
     * @param date
     * @return
     */
    public static String getSimpleDate(String date) {
        return getSimpleDate(formatStrToDate(date, "yyyy-MM-dd"));
    }

    /**
     * 将java.util.Date类型日期转换为java.util.Calendar类型日期
     *
     * @param date
     *            java.util.Date 类型日期
     * @return java.util.Calendar 类型日期
     */
    public static Calendar convertDateToCal(Date date) {
        if (date == null) {
            date = new Date();
        }
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c;
    }

    /**
     * 数字格式的时间转换为字符串表示,如120000 -> 12:00:00
     *
     * @param time
     *            数字格式的时间
     * @return 字符串格式的时间
     */
    public static String getIntTimeToStr(int time) {
        int hour = time / 10000;
        int min = (time - hour * 10000) / 100;
        int sec = time - hour * 10000 - min * 100;
        String hours = hour + "";
        String mins = min + "";
        String secs = sec + "";
        if (0 <= hour && hour <= 9){
            hours = "0" + hour;
        }

        if (0 <= min && min <= 9){
            mins = "0" + min;
        }

        if (0 <= sec && sec <= 9){
            secs = "0" + sec;
        }

        return hours + ":" + mins + ":" + secs;
    }

    /**
     * 根据条件生成若干年、若干月、若干日之后的某日的整数形式日期:
     *
     * @param years
     *            若干年
     * @param months
     *            若干月
     * @param days
     *            若干日
     * @return 整数形式日期 格式:20180101
     */
    public int addValidDate(int years, int months, int days) {
        Calendar now = new GregorianCalendar();
        Calendar validDate = new GregorianCalendar(now.get(Calendar.YEAR),
                now.get(Calendar.MONTH) + 1, now.get(Calendar.DAY_OF_MONTH));

        String year = String.valueOf(validDate.get(Calendar.YEAR) + years);
        String month = String.valueOf(validDate.get(Calendar.MONTH) + months);
        String day = String
                .valueOf(validDate.get(Calendar.DAY_OF_MONTH) + days);
        if (month.length() < 2){
            month = "0" + month;
        }

        if (day.length() < 2){
            day = "0" + day;
        }

        String nowDate = year + month + day;
        return Integer.parseInt(nowDate);
    }

    /**
     * 获得指定日期的期限间隔的实际天数,比如给定2007年2月1日,期限1月,返回28;给定2007年3月1日,期限1月,返回31
     *
     * @param startDate
     *            指定日期
     * @param term
     *            期限间隔
     * @return 实际天数
     */
    public static int countDay(Date startDate, String term) {
        if (term == null || term.trim().length() == 0){
            return 0;
        }

        String str = term.trim().toLowerCase();
        int result = 0;
        char cha = str.charAt(str.length() - 1);
        if (47 < cha && cha < 58) {
            str = str + day;
        }
        if (str.length() == 1) {
            str = "0" + str;
        }
        try {
            result = Integer.parseInt(str.substring(0, str.length() - 1));
        } catch (Exception e) {
            return 0;
        }
        Calendar cal = Calendar.getInstance();
        cal.setTime(startDate);
        int days = 0;
        if (str.endsWith(day)) {
            days = result;
        } else if (str.endsWith(week)) {
            days = result * 7;
        } else if (str.endsWith(month)) {
            cal.add(Calendar.MONTH, result);
            days=getDays(startDate, cal.getTime());
        } else if (str.endsWith(year)) {
            cal.add(Calendar.YEAR, result);
            days=getDays(startDate, cal.getTime());
        } else {
            days = 0;
        }
        return days;
    }

    /**
     * 获得指定日期(字符串格式yyyy-MM-dd)的期限间隔的实际天数,
     * 比如给定2007年2月1日,期限1月,返回28;给定2007年3月1日,期限1月,返回31
     *
     * @param startDate
     *            指定日期yyyy-MM-dd
     * @param term
     *            期限间隔
     * @return 实际天数
     */
    public static int countDay(String startDate, String term) throws Exception {
//		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
//		Date date = sdf.parse(startDate);
        Date date=DATE_FORMAT.get().parse(startDate);
        return countDay(date, term);
    }

    /**
     * 按yyyyMMdd G:i:s格式比较两个日期的大小
     *
     * @param date1
     *            比较的第一个日期
     * @param date2
     *            比较的第二个日期
     * @return 两个日期相等返回0,date1大于date2返回1,date1小于date2返回-1.
     */
    public static int compareTime(Date date1, Date date2) {
        long intDate1 = date1.getTime();
        long intDate2 = date2.getTime();
        return intDate1 >= intDate2 ? (intDate1 == intDate2 ? 0 : 1) : -1;
    }

    /**
     * 按yyyyMMdd G:i:s格式比较两个日期的大小
     *
     * @param date1
     *            比较的第一个日期
     * @param date2
     *            比较的第二个日期
     * @return 两个日期相等返回0,date1大于date2返回1,date1小于date2返回-1.
     */
    public static int compareTime(String date1, String date2)
            throws ParseException {
        long intDate1 = formatStrToDate(date1, null).getTime();
        long intDate2 = formatStrToDate(date2, null).getTime();
        return intDate1 >= intDate2 ? (intDate1 == intDate2 ? 0 : 1) : -1;
    }

    /**
     * 返回两个日期相差年数
     *
     * @param start
     * @param end
     * @return int
     */
    public static int yearsOf(Date start, Date end) {
        Calendar c1 = Calendar.getInstance();
        c1.setTime(start);
        Calendar c2 = Calendar.getInstance();
        c2.setTime(end);
        return c2.get(Calendar.YEAR) - c1.get(Calendar.YEAR) + 1;
    }

    /**
     * 获取当月的第一天
     *
     * @param date
     * @return
     */
    public static Date getFirstDayOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        return cal.getTime();
    }

    /**
     * 获取当月的最后一天 add by liumeng 2016-05-26
     *
     * @param date
     * @return
     */
    public static Date getLastDayOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DATE, 1);
        cal.roll(Calendar.DATE, -1);
        return cal.getTime();
    }

    /**
     * 获取指定日期中文形式的星期几
     *
     * @param date
     * @param format
     * @return
     */
    public static String getChineseWeek(String date, String format) {
        String[] day = new String[] { "日", "一", "二", "三", "四", "五", "六" };
        return "星期"+ day[getWeek(date,format)];
    }

    /**
     * 获取指定日期的星期几
     *
     * @param date
     * @param format
     * @return
     */
    public static Integer getWeek(String date, String format) {
        format = format == null ? "yyyy-MM-dd HH:mm:ss" : format;
        Calendar cal = convertDateToCal(formatStrToDate(date, format));
        return cal.get(Calendar.DAY_OF_WEEK) - 1;
    }

    /**
     * 功能:判断是否为日期格式
     *
     * @param date
     * @return
     */
    public static boolean isDate(String date) {
        try {
            DATE_FORMAT.get().parse(date);
            return true;
        } catch (ParseException e) {
            return false;
        }
    }

    /**
     * 获取昨天的日期
     * @return
     */
    public static Date getYesterday() {
        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, -1);
        return formatStrToDate(formatDateToStr(cal.getTime(), null), null);
    }

    /**
     * 获取当前系统时间
     * @return
     */
    public static String getCurrentDateTime(){
        LocalDateTime localDateTime=LocalDateTime.now();
        DateTimeFormatter dateTimeFormatter=DateTimeFormatter.ofPattern(PATTERN_DATE_TIME);
        return localDateTime.format(dateTimeFormatter);
    }

    /**
     * 获取当前系统日期
     * @return
     */
    public static String getCurrentDate(){
        LocalDate localDate=LocalDate.now();
        DateTimeFormatter dateTimeFormatter=DateTimeFormatter.ofPattern(PATTERN_DATE);
        return localDate.format(dateTimeFormatter);
    }

}
