package cn.tenmg.hibernate.sqltool.service;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import cn.tenmg.sqltool.data.Page;

/**
 * 数据库访问服务
 * 
 * @author 赵伟均 wjzhao@aliyun.com
 *
 * @param <P> 实体类型
 */
public interface Service<P> {

	/**
	 * 保存对象。
	 * 
	 * @param p 要保存的对象
	 * @return 返回保存的对象
	 */
	P save(P p);

	/**
	 * 根据主键获取对象。
	 * 
	 * @param id 主键
	 * @return 返回获取的对象
	 */
	P get(Serializable id);

	/**
	 * 使用指定DSQL和指定参数（分别列出参数名和参数值）获取对象。
	 * 
	 * @param dsql   指定DSQL
	 * @param params 指定参数（分别列出参数名和参数值）
	 * @return 返回查询到的对象
	 */
	Object[] get(String dsql, Object... params);

	/**
	 * 使用指定DSQL和指定参数获取对象。
	 * 
	 * @param dsql   指定DSQL
	 * @param params 指定参数
	 * @return 返回查询到的对象
	 */
	Object[] get(String dsql, Map<String, Object> params);

	/**
	 * 使用指定类，指定DSQL和指定参数（分别列出参数名和参数值）获取对象。该方法将根据DSQL中的别名将查询结果映射为指定类的对象，
	 * 需要保证DSQL中的别名和指定类的属性名保持一致。
	 * 
	 * @param <T>    实体类
	 * @param type   指定类
	 * 
	 * @param dsql   指定DSQL
	 * @param params 指定参数（分别列出参数名和参数值）
	 * @return 返回查询到的对象
	 */
	<T> T get(Class<T> type, String dsql, Object... params);

	/**
	 * 使用指定类，指定DSQL和指定参数获取对象。该方法将根据DSQL中的别名将查询结果映射为指定类的对象， 需要保证DSQL中的别名和指定类的属性名保持一致。
	 * 
	 * @param <T>    实体类
	 * @param type   指定类
	 * 
	 * @param dsql   指定DSQL
	 * @param params 指定参数
	 * @return 返回查询到的对象
	 */
	<T> T get(Class<T> type, String dsql, Map<String, Object> params);

	/**
	 * 使用指定DSQL和指定参数（分别列出参数名和参数值）查询对象。
	 * 
	 * @param dsql   指定DSQL
	 * @param params 指定参数（分别列出参数名和参数值）
	 * @return 返回查询到的对象
	 */
	List<Object[]> query(String dsql, Object... params);

	/**
	 * 使用指定DSQL和指定参数查询对象。
	 * 
	 * @param dsql   指定DSQL
	 * @param params 指定参数
	 * @return 返回查询到的对象
	 */
	List<Object[]> query(String dsql, Map<String, Object> params);

	/**
	 * 使用指定类，指定DSQL和指定参数（分别列出参数名和参数值）查询对象。该方法将根据DSQL中的别名将查询结果全部映射为指定类的对象，并返回这个列表，
	 * 需要保证DSQL中的别名和指定类的属性名保持一致。
	 * 
	 * @param <T>    实体类
	 * @param type   指定类
	 * @param dsql   指定DSQL
	 * @param params 指定参数（分别列出参数名和参数值）
	 * @return 返回查询到的对象
	 */
	<T> List<T> query(Class<?> type, String dsql, Object... params);

	/**
	 * 使用指定类，指定DSQL和指定参数查询对象。该方法将根据DSQL中的别名将查询结果全部映射为指定类的对象，并返回这个列表，
	 * 需要保证DSQL中的别名和指定类的属性名保持一致。
	 * 
	 * @param <T>    实体类
	 * @param type   指定类
	 * @param dsql   指定DSQL
	 * @param params 指定参数
	 * @return 返回查询到的对象
	 */
	<T> List<T> query(Class<?> type, String dsql, Map<String, Object> params);

	/**
	 * 使用指定DSQL和指定参数（分别列出参数名和参数值）查询第一列。
	 * 
	 * @param <T>    基本类型
	 * @param dsql   指定DSQL
	 * @param params 指定参数（分别列出参数名和参数值）
	 * @return 返回查询到的第一列对象
	 */
	<T> List<T> queryFirstCol(String dsql, Object... params);

	/**
	 * 使用指定类，指定DSQL和指定参数查询对象。该方法将根据DSQL中的别名将查询结果全部映射为指定类的对象，并返回这个列表，
	 * 需要保证DSQL中的别名和指定类的属性名保持一致。
	 * 
	 * @param <T>    基本类型
	 * @param dsql   指定DSQL
	 * @param params 指定参数
	 * @return 返回查询到的对象
	 */
	<T> List<T> queryFirstCol(String dsql, Map<String, Object> params);

	/**
	 * 使用指定DSQL和指定参数（分别列出参数名和参数值）查询唯一对象，如count查询等。
	 * 
	 * @param <T>    基本类型
	 * @param dsql   指定DSQL
	 * @param params 指定参数（分别列出参数名和参数值）
	 * @return 返回查询到的对象
	 */
	<T> T queryUnique(String dsql, Object... params);

	/**
	 * 使用指定DSQL和指定参数查询唯一对象，如count查询等。
	 * 
	 * @param <T>    基本类型
	 * @param dsql   指定DSQL
	 * @param params 指定参数
	 * @return 返回查询到的对象
	 */
	<T> T queryUnique(String dsql, Map<String, Object> params);

	/**
	 * 使用指定类，指定DSQL和指定参数（分别列出参数名和参数值）查询唯一对象，如count查询等。该方法将根据SQL中的别名将查询结果映射为指定类的对象 ，
	 * 需要保证DSQL中的别名和指定类的属性名保持一致。
	 * 
	 * @param <T>    基本类型
	 * @param type   指定类
	 * @param dsql   指定DSQL
	 * @param params 指定参数（分别列出参数名和参数值）
	 * @return 返回查询到的对象
	 */
	<T> T queryUnique(Class<T> type, String dsql, Object... params);

	/**
	 * 使用指定类，指定DSQL和指定参数查询唯一对象，如count查询等。该方法将根据SQL中的别名将查询结果映射为指定类的对象 ，
	 * 需要保证DSQL中的别名和指定类的属性名保持一致。
	 * 
	 * @param <T>    基本类型
	 * @param type   指定类
	 * @param dsql   指定DSQL
	 * @param params 指定参数
	 * @return 返回查询到的对象
	 */
	<T> T queryUnique(Class<T> type, String dsql, Map<String, Object> params);

	/**
	 * 使用指定DSQL，指定页码，指定页容量和指定参数（分别列出参数名和参数值）分页查询对象
	 * 
	 * @param dsql        指定DSQL
	 * @param currentPage 指定页码
	 * @param pageSize    指定页容量
	 * @param params      指定参数（分别列出参数名和参数值）
	 * @return 返回查询到的对象并封装为Page对象
	 */
	Page<Object[]> page(String dsql, long currentPage, int pageSize, Object... params);

	/**
	 * 使用指定DSQL，指定页码，指定页容量和指定参数分页查询对象
	 * 
	 * @param dsql        指定DSQL
	 * @param currentPage 指定页码
	 * @param pageSize    指定页容量
	 * @param params      指定参数
	 * @return 返回查询到的对象并封装为Page对象
	 */
	Page<Object[]> page(String dsql, long currentPage, int pageSize, Map<String, Object> params);

	/**
	 * 使用指定类，指定DSQL，指定页码，指定页容量和指定参数（分别列出参数名和参数值）分页查询对象。 该方法将根据DSQL中的别名将对象映射为指定类的对象，
	 * 需要保证SQL中的别名和对象属性名保持一致。
	 * 
	 * @param <T>         实体类
	 * @param type        指定类
	 * @param dsql        指定DSQL
	 * @param currentPage 指定页码
	 * @param pageSize    指定页容量
	 * @param params      指定参数（分别列出参数名和参数值）
	 * @return 返回查询到的对象并封装为Page对象
	 */
	<T extends Serializable> Page<T> page(Class<T> type, String dsql, long currentPage, int pageSize, Object... params);

	/**
	 * 使用指定类，指定DSQL，指定页码，指定页容量和指定参数分页查询对象。 该方法将根据DSQL中的别名将对象映射为指定类的对象，
	 * 需要保证SQL中的别名和对象属性名保持一致。
	 * 
	 * @param <T>         实体类
	 * @param type        指定类
	 * @param dsql        指定DSQL
	 * @param currentPage 指定页码
	 * @param pageSize    指定页容量
	 * @param params      指定参数
	 * @return 返回查询到的对象并封装为Page对象
	 */
	<T extends Serializable> Page<T> page(Class<T> type, String dsql, long currentPage, int pageSize,
			Map<String, Object> params);

	/**
	 * 更新对象。
	 * 
	 * @param p 要更新的对象
	 * @return 返回更新后的对象
	 */
	P update(P p);

	/**
	 * 保存或更新对象。
	 * 
	 * @param p 要保存的对象
	 * @return 返回保存后的对象
	 */
	P saveOrUpdate(P p);

	/**
	 * 删除对象。
	 * 
	 * @param p 要删除的对象
	 * @return 返回删除的对象
	 */
	boolean delete(P p);

	/**
	 * 根据主键删除对象。
	 * 
	 * @param id 主键
	 * @return 返回删除的对象
	 */
	int delete(Serializable id);

	/**
	 * 批量插入对象。
	 *
	 * @param list 要插入的对象列表
	 * @param size 批量刷新记录数
	 * @return 成功返回true，否则false
	 */
	boolean insertBatch(List<P> list, int size);

	/**
	 * 批量更新对象。
	 *
	 * @param list 要更新的对象列表
	 * @param size 批量刷新记录数
	 * @return 成功返回true，否则false
	 */
	boolean updateBatch(List<P> list, int size);

	/**
	 * 批量保存/更新对象。
	 *
	 * @param list 要保存/更新的对象列表
	 * @param size 批量刷新记录数
	 * @return 成功返回true，否则false
	 */
	boolean saveOrUpdateBatch(List<P> list, int size);

	/**
	 * 执行指定DSQL和指定参数（分别列出参数名和参数值）并返回受影响行数。
	 * 
	 * @param dsql   指定DSQL
	 * @param params 指定参数（分别列出参数名和参数值）
	 * @return 返回受影响行数
	 */
	int excecute(String dsql, Object... params);

	/**
	 * 执行指定DSQL和指定参数并返回受影响行数。
	 * 
	 * @param dsql   指定DSQL
	 * @param params 指定参数
	 * @return 返回受影响行数
	 */
	int excecute(String dsql, Map<String, Object> params);

}
