package cn.tenmg.flink.jobs.launcher;

import java.io.File;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;

import org.apache.flink.api.common.JobID;
import org.apache.flink.api.common.JobStatus;
import org.apache.flink.client.deployment.StandaloneClusterId;
import org.apache.flink.client.program.PackagedProgram;
import org.apache.flink.client.program.PackagedProgram.Builder;
import org.apache.flink.client.program.PackagedProgramUtils;
import org.apache.flink.client.program.rest.RestClusterClient;
import org.apache.flink.configuration.ConfigOption;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.JobManagerOptions;
import org.apache.flink.configuration.JobManagerOptions.SchedulerType;
import org.apache.flink.configuration.MemorySize;
import org.apache.flink.configuration.RestOptions;
import org.apache.flink.configuration.SchedulerExecutionMode;
import org.apache.flink.runtime.jobgraph.JobGraph;
import org.apache.flink.runtime.jobgraph.SavepointRestoreSettings;

import cn.tenmg.flink.jobs.FlinkJobsLauncher.FlinkJobsInfo.State;
import cn.tenmg.flink.jobs.config.model.FlinkJobs;
import cn.tenmg.flink.jobs.launcher.context.FlinkJobsLauncherContext;

/**
 * REST群集客户端flink-jobs启动器。用于远程提交flink任务
 * 
 * @author June wjzhao@aliyun.com
 * 
 * @since 1.1.4
 */
public class RestClusterClientFlinkJobsLauncher extends AbstractFlinkJobsLauncher {

	private static final Configuration configuration = new Configuration();

	{
		setConfigString(configuration, JobManagerOptions.ADDRESS);// 必须配置
		setConfigString(configuration, JobManagerOptions.ARCHIVE_DIR);
		setConfigString(configuration, JobManagerOptions.BIND_HOST);
		setConfigString(configuration, JobManagerOptions.EXECUTION_FAILOVER_STRATEGY);
		setConfigInteger(configuration, JobManagerOptions.JOB_STORE_MAX_CAPACITY);
		setConfigLong(configuration, JobManagerOptions.JOB_STORE_CACHE_SIZE);
		setConfigLong(configuration, JobManagerOptions.JOB_STORE_EXPIRATION_TIME);
		setConfigInteger(configuration, JobManagerOptions.JOB_STORE_MAX_CAPACITY);
		setConfigBoolean(configuration, JobManagerOptions.JVM_DIRECT_MEMORY_LIMIT_ENABLED);
		setConfigMemorySize(configuration, JobManagerOptions.JVM_HEAP_MEMORY);
		setConfigMemorySize(configuration, JobManagerOptions.JVM_METASPACE);
		setConfigFloat(configuration, JobManagerOptions.JVM_OVERHEAD_FRACTION);
		setConfigMemorySize(configuration, JobManagerOptions.JVM_OVERHEAD_MAX);
		setConfigMemorySize(configuration, JobManagerOptions.JVM_OVERHEAD_MIN);
		setConfigInteger(configuration, JobManagerOptions.MAX_ATTEMPTS_HISTORY_SIZE);
		setConfigInteger(configuration, JobManagerOptions.MIN_PARALLELISM_INCREASE);
		setConfigMemorySize(configuration, JobManagerOptions.OFF_HEAP_MEMORY);
		setConfigBoolean(configuration, JobManagerOptions.PARTITION_RELEASE_DURING_JOB_EXECUTION);
		setConfigInteger(configuration, JobManagerOptions.PORT);// 必须配置
		setConfigDuration(configuration, JobManagerOptions.RESOURCE_STABILIZATION_TIMEOUT);
		setConfigDuration(configuration, JobManagerOptions.RESOURCE_WAIT_TIMEOUT);
		setConfigBoolean(configuration, JobManagerOptions.RETRIEVE_TASK_MANAGER_HOSTNAME);
		setConfigInteger(configuration, JobManagerOptions.RPC_BIND_PORT);
		setConfigSchedulerType(configuration, JobManagerOptions.SCHEDULER);
		setConfigSchedulerExecutionMode(configuration, JobManagerOptions.SCHEDULER_MODE);
		setConfigLong(configuration, JobManagerOptions.SLOT_IDLE_TIMEOUT);
		setConfigLong(configuration, JobManagerOptions.SLOT_REQUEST_TIMEOUT);
		setConfigMemorySize(configuration, JobManagerOptions.TOTAL_FLINK_MEMORY);
		setConfigMemorySize(configuration, JobManagerOptions.TOTAL_PROCESS_MEMORY);
		setConfigString(configuration, RestOptions.ADDRESS);
		setConfigLong(configuration, RestOptions.AWAIT_LEADER_TIMEOUT);

		setConfigString(configuration, RestOptions.BIND_ADDRESS);
		setConfigString(configuration, RestOptions.BIND_PORT);
		setConfigInteger(configuration, RestOptions.CLIENT_MAX_CONTENT_LENGTH);
		setConfigLong(configuration, RestOptions.CONNECTION_TIMEOUT);
		setConfigBoolean(configuration, RestOptions.ENABLE_FLAMEGRAPH);
		setConfigDuration(configuration, RestOptions.FLAMEGRAPH_CLEANUP_INTERVAL);
		setConfigDuration(configuration, RestOptions.FLAMEGRAPH_DELAY);
		setConfigInteger(configuration, RestOptions.FLAMEGRAPH_NUM_SAMPLES);
		setConfigDuration(configuration, RestOptions.FLAMEGRAPH_REFRESH_INTERVAL);
		setConfigInteger(configuration, RestOptions.FLAMEGRAPH_STACK_TRACE_DEPTH);
		setConfigLong(configuration, RestOptions.IDLENESS_TIMEOUT);
		setConfigInteger(configuration, RestOptions.PORT);
		setConfigLong(configuration, RestOptions.RETRY_DELAY);
		setConfigInteger(configuration, RestOptions.RETRY_MAX_ATTEMPTS);
		setConfigInteger(configuration, RestOptions.SERVER_MAX_CONTENT_LENGTH);
		setConfigInteger(configuration, RestOptions.SERVER_NUM_THREADS);
		setConfigInteger(configuration, RestOptions.SERVER_THREAD_PRIORITY);
	}

	private volatile RestClusterClient<StandaloneClusterId> restClusterClient;

	@Override
	public FlinkJobsInfo launch(FlinkJobs flinkJobs) throws Exception {
		RestClusterClient<StandaloneClusterId> client = null;
		FlinkJobsInfo appInfo = new FlinkJobsInfo();
		try {
			client = getRestClusterClient();
			File jarFile = new File(getJar(flinkJobs));
			SavepointRestoreSettings savepointRestoreSettings = SavepointRestoreSettings.none();

			Builder builder = PackagedProgram.newBuilder().setConfiguration(configuration)
					.setEntryPointClassName(getEntryPointClassName(flinkJobs)).setJarFile(jarFile)
					.setUserClassPaths(getUserClassPaths(flinkJobs))
					.setSavepointRestoreSettings(savepointRestoreSettings);
			String arguments = getArguments(flinkJobs);
			if (!isEmptyArguments(arguments)) {
				builder.setArguments(arguments);
			}
			PackagedProgram program = builder.build();
			JobGraph jobGraph = PackagedProgramUtils.createJobGraph(program, configuration, getParallelism(flinkJobs),
					Boolean.valueOf(FlinkJobsLauncherContext.getProperty("suppress.output", "false")));
			CompletableFuture<JobID> submited = client.submitJob(jobGraph);
			JobID jobId = submited.get();
			appInfo.setJobId(jobId.toString());
			JobStatus jobStatus = client.getJobStatus(jobId).get();
			if (JobStatus.INITIALIZING.equals(jobStatus)) {
				appInfo.setState(State.ACCEPTED);
			} else if (JobStatus.RUNNING.equals(jobStatus)) {
				appInfo.setState(State.RUNNING);
			} else if (JobStatus.FINISHED.equals(jobStatus)) {
				appInfo.setState(State.FINISHED);
			} else if (JobStatus.FAILED.equals(jobStatus)) {
				appInfo.setState(State.FAILED);
			} else if (JobStatus.CANCELED.equals(jobStatus)) {
				appInfo.setState(State.KILLED);
			} else {
				appInfo.setState(State.SUBMITTED);
			}
		} catch (Exception e) {
			if (client != null) {
				client.close();
			}
			throw e;
		}
		return appInfo;
	}

	@Override
	public String stop(String jobId) throws Exception {
		return getRestClusterClient().stopWithSavepoint(JobID.fromHexString(jobId), false,
				FlinkJobsLauncherContext.getProperty("state.savepoints.dir")).get();
	}

	private RestClusterClient<StandaloneClusterId> getRestClusterClient() throws Exception {
		if (restClusterClient == null) {
			synchronized (RestClusterClientFlinkJobsLauncher.class) {
				if (restClusterClient == null) {
					restClusterClient = new RestClusterClient<StandaloneClusterId>(configuration,
							StandaloneClusterId.getInstance());
				}
			}
		}
		return restClusterClient;
	}

	private static <T> String getConfigString(ConfigOption<T> configOption) {
		return FlinkJobsLauncherContext.getProperty(configOption.key());
	}

	private static void setConfigString(Configuration configuration, ConfigOption<String> configOption) {
		String config = getConfigString(configOption);
		if (config != null) {
			configuration.setString(configOption, config);
		}
	}

	private static void setConfigInteger(Configuration configuration, ConfigOption<Integer> configOption) {
		String config = getConfigString(configOption);
		if (config != null) {
			configuration.setInteger(configOption, Integer.parseInt(config));
		}
	}

	private static void setConfigLong(Configuration configuration, ConfigOption<Long> configOption) {
		String config = getConfigString(configOption);
		if (config != null) {
			configuration.setLong(configOption, Long.parseLong(config));
		}
	}

	private static void setConfigFloat(Configuration configuration, ConfigOption<Float> configOption) {
		String config = getConfigString(configOption);
		if (config != null) {
			configuration.setFloat(configOption, Float.parseFloat(config));
		}
	}

	private static void setConfigBoolean(Configuration configuration, ConfigOption<Boolean> configOption) {
		String config = getConfigString(configOption);
		if (config != null) {
			configuration.setBoolean(configOption, Boolean.parseBoolean(config));
		}
	}

	private static void setConfigMemorySize(Configuration configuration, ConfigOption<MemorySize> configOption) {
		String config = getConfigString(configOption);
		if (config != null) {
			configuration.set(configOption, MemorySize.parse(config));
		}
	}

	private static void setConfigDuration(Configuration configuration, ConfigOption<Duration> configOption) {
		String config = getConfigString(configOption);
		if (config != null) {
			configuration.set(configOption, Duration.parse(config));
		}
	}

	private static void setConfigSchedulerType(Configuration configuration, ConfigOption<SchedulerType> configOption) {
		String config = getConfigString(configOption);
		if (config != null) {
			configuration.set(configOption, SchedulerType.valueOf(config));
		}
	}

	private static void setConfigSchedulerExecutionMode(Configuration configuration,
			ConfigOption<SchedulerExecutionMode> configOption) {
		String config = getConfigString(configOption);
		if (config != null) {
			configuration.set(configOption, SchedulerExecutionMode.valueOf(config));
		}
	}

}
