package cn.tenmg.flink.jobs.launcher;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.alibaba.fastjson.JSON;

import cn.tenmg.flink.jobs.FlinkJobsLauncher;
import cn.tenmg.flink.jobs.launcher.config.model.FlinkJobs;
import cn.tenmg.flink.jobs.launcher.config.model.Option;
import cn.tenmg.flink.jobs.launcher.config.model.Options;
import cn.tenmg.flink.jobs.launcher.config.model.Params;
import cn.tenmg.flink.jobs.launcher.utils.Sets;

/**
 * 基于命令行的YARN flink-jobs应用程序启动器
 * 
 * @author 赵伟均 wjzhao@aliyun.com
 *
 */
public class CommandLineFlinkJobsLauncher implements FlinkJobsLauncher {

	private static final boolean isWindows = System.getProperty("os.name", "").toLowerCase().contains("windows");

	private static final char BLANK_SPACE = ' ';

	private static final String APPLICATION_ID_PREFFIX = "Submitting application master",
			JOB_ID_PREFIX = "Job with JobID ", JOB_ID_SUFFIX = " has finished",
			CURRENT_STATE_PREFFIX = "Deploying cluster, current state", RUNNING_LOG = "Starting execution of program",
			YARN_RUNNING_LOG = "YARN application has been deployed successfully",
			FINISHED_LOG = "Program execution finished",
			YARN_FINISHED_LOG = "YARN application has been finished successfully", EXCEPTION = "Exception";

	private static final Pattern JOB_ID_PATTERN = Pattern.compile(JOB_ID_PREFIX + "[\\S]+" + JOB_ID_SUFFIX);

	private static int APPLICATION_ID_PREFFIX_LEN = APPLICATION_ID_PREFFIX.length(),
			JOB_ID_PREFIX_LEN = JOB_ID_PREFIX.length(), JOB_ID_SUFFIX_LEN = JOB_ID_SUFFIX.length(),
			CURRENT_STATE_PREFFIX_LEN = CURRENT_STATE_PREFFIX.length();

	private static final Logger log = LogManager.getLogger(CommandLineFlinkJobsLauncher.class);

	private static Set<String> TARGETS = Sets.as("-t", "--target");

	private String flinkHome;

	private Action action = Action.RUN_APPLICATION;

	public String getFlinkHome() {
		return flinkHome;
	}

	public void setFlinkHome(String flinkHome) {
		this.flinkHome = flinkHome;
	}

	public Action getAction() {
		return action;
	}

	public void setAction(Action action) {
		this.action = action;
	}

	@Override
	public FlinkJobsApplicationInfo launch(FlinkJobs config) throws Exception {
		StringBuilder commandBuilder = new StringBuilder();
		commandBuilder.append(flinkHome + "/bin/flink").append(BLANK_SPACE).append(action.getName());
		attachOptions(commandBuilder, config.getOptions());
		String mainClass = config.getMainClass();
		if (mainClass != null) {
			commandBuilder.append(BLANK_SPACE).append("-c").append(BLANK_SPACE).append(mainClass);
		}
		commandBuilder.append(BLANK_SPACE).append(config.getJar());
		Params params = config.getParams();
		if (params != null) {
			commandBuilder.append(BLANK_SPACE).append(JSON.toJSONString(params).replaceAll("\"", "\\\""));
		}

		String command = commandBuilder.toString();
		log.info("Execute command: " + command);

		Process p = null;
		InputStream is = null;
		BufferedReader r = null;
		File temp = null;
		try {
			temp = File.createTempFile("flink-jobs/" + UUID.randomUUID().toString().replaceAll("-", ""),
					isWindows ? ".bat" : ".sh");
			temp.setExecutable(true);
			FileWriter w = null;
			String[] cmdarray;
			try {
				w = new FileWriter(temp);
				if (isWindows) {// Windows命令
					cmdarray = new String[] { "cmd", "/C", temp.getAbsolutePath() };
				} else {// Unix命令
					File profile = new File("/etc/profile");
					if (profile.exists()) {
						FileReader fr = null;
						try {
							fr = new FileReader(profile);
							int ch = 0;
							while ((ch = fr.read()) != -1) {
								w.write(ch);
							}
						} catch (IOException e) {
							e.printStackTrace();
						} finally {
							if (fr != null) {
								fr.close();
							}
						}
					}
					w.write("\n");
					cmdarray = new String[] { "/bin/sh", "-c", temp.getAbsolutePath() };
				}
				w.write(command);
			} catch (Exception e) {
				throw e;
			} finally {
				if (w != null) {
					w.close();
				}
			}
			p = Runtime.getRuntime().exec(cmdarray, null, new File(flinkHome.concat("/bin")));
			(new ErrorStreamCatcher(p.getErrorStream())).start();
			FlinkJobsApplicationInfo appInfo = new FlinkJobsApplicationInfo();
			(new InputStreamCatcher(p.getInputStream(), appInfo)).start();
			p.waitFor();
			return appInfo;
		} catch (Exception e) {
			throw e;
		} finally {
			if (is != null) {
				is.close();
			}
			if (r != null) {
				r.close();
			}
			if (p != null) {
				p.destroy();
			}
			if (temp != null) {// 终止后删除临时文件
				temp.deleteOnExit();
			}
		}
	}

	protected void attachOptions(StringBuilder commandBuilder, Options options) {
		String targetKey = null, target = null;
		if (options != null) {
			String prefix = options.getKeyPrefix(), key, value;
			List<Option> option = options.getOption();
			if (option != null) {
				for (Iterator<Option> it = option.iterator(); it.hasNext();) {
					Option entry = it.next();
					key = entry.getKey();
					if (!key.startsWith("-")) {
						key = prefix + key;
					}
					value = entry.getValue();
					if (TARGETS.contains(key)) {
						if (value != null && !value.isEmpty()) {
							targetKey = key;
							target = value;
						}
					} else {
						commandBuilder.append(BLANK_SPACE).append(key);
						if (value != null && !value.isEmpty()) {
							commandBuilder.append(BLANK_SPACE).append(value.replaceAll("\"", "\\\""));
						}
					}
				}
			}
		}
		if (target == null) {
			if (Action.RUN_APPLICATION.equals(action)) {
				commandBuilder.append(BLANK_SPACE).append("-t");
				commandBuilder.append(BLANK_SPACE).append("yarn-application");
			}
		} else {
			commandBuilder.append(BLANK_SPACE).append(targetKey);
			commandBuilder.append(BLANK_SPACE).append(target.replaceAll("\"", "\\\""));
		}
	}

	public static enum Action {
		RUN("run"), RUN_APPLICATION("run-application");

		private String name;

		private Action(String name) {
			this.name = name;
		}

		public String getName() {
			return name;
		}
	}

	public static class InputStreamCatcher extends Thread {

		private InputStream is;

		private FlinkJobsApplicationInfo appInfo;

		public InputStreamCatcher(InputStream is, FlinkJobsApplicationInfo appInfo) {
			this.is = is;
			this.appInfo = appInfo;
		}

		@Override
		public void run() {
			BufferedReader r = null;
			FlinkJobsApplicationInfo.State state = FlinkJobsApplicationInfo.State.SUBMITTED;
			String line, applicationId = null;
			try {
				r = new BufferedReader(new InputStreamReader(is));
				while ((line = r.readLine()) != null) {// 收集日志信息，并记录applicationId和state
					log.info(line);
					int index = line.indexOf(APPLICATION_ID_PREFFIX);
					if (index >= 0) {
						applicationId = line.substring(index + APPLICATION_ID_PREFFIX_LEN).trim();
					} else {
						index = line.indexOf(CURRENT_STATE_PREFFIX);
						if (index >= 0) {
							try {
								state = FlinkJobsApplicationInfo.State
										.valueOf(line.substring(index + CURRENT_STATE_PREFFIX_LEN).trim());
							} catch (Exception e) {
								state = FlinkJobsApplicationInfo.State.SUBMITTED;
							}
						} else if (line.contains(RUNNING_LOG) || line.contains(YARN_RUNNING_LOG)) {
							state = FlinkJobsApplicationInfo.State.RUNNING;
						} else if (line.contains(FINISHED_LOG) || line.contains(YARN_FINISHED_LOG)) {
							state = FlinkJobsApplicationInfo.State.FINISHED;
						} else {
							Matcher m = JOB_ID_PATTERN.matcher(line);
							if (m.find()) {
								String group = m.group();
								applicationId = group.substring(JOB_ID_PREFIX_LEN, group.length() - JOB_ID_SUFFIX_LEN);
							} else if (FlinkJobsApplicationInfo.State.SUBMITTED.equals(state)) {
								index = line.indexOf(EXCEPTION);
								if (index >= 0) {
									throw new Exception(line);
								}
							}
						}
					}
				}
			} catch (IOException e) {
				e.printStackTrace();
			} catch (Exception e) {
				e.printStackTrace();
			} finally {
				if (r != null) {
					try {
						r.close();
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}
			appInfo.setApplicationId(applicationId);
			appInfo.setState(state);
		}
	}

	public static class ErrorStreamCatcher extends Thread {

		private InputStream is;

		public ErrorStreamCatcher(InputStream is) {
			this.is = is;
		}

		@Override
		public void run() {
			BufferedReader r = null;
			try {
				r = new BufferedReader(new InputStreamReader(is));
				String line;
				while ((line = r.readLine()) != null) {
					log.info(line);
				}
			} catch (IOException e) {
				e.printStackTrace();
			} catch (Exception e) {
				e.printStackTrace();
			} finally {
				if (r != null) {
					try {
						r.close();
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}
		}

	}

	protected static class FlinkJobsApplicationInfo implements FlinkJobsLauncher.FlinkJobsApplicationInfo {

		private String applicationId;

		private State state;

		@Override
		public String getApplicationId() {
			return applicationId;
		}

		public void setApplicationId(String applicationId) {
			this.applicationId = applicationId;
		}

		@Override
		public State getState() {
			return state;
		}

		public void setState(State state) {
			this.state = state;
		}

	}

}
