package cn.tenmg.flink.jobs.launcher;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.alibaba.fastjson.JSON;

import cn.tenmg.flink.jobs.FlinkJobsLauncher;
import cn.tenmg.flink.jobs.FlinkJobsLauncher.FlinkJobsApplicationInfo.State;
import cn.tenmg.flink.jobs.launcher.config.model.FlinkJobs;
import cn.tenmg.flink.jobs.launcher.config.model.Option;
import cn.tenmg.flink.jobs.launcher.config.model.Options;
import cn.tenmg.flink.jobs.launcher.config.model.Params;
import cn.tenmg.flink.jobs.launcher.utils.Sets;

/**
 * 基于命令行的YARN flink-jobs应用程序启动器
 * 
 * @author 赵伟均 wjzhao@aliyun.com
 *
 */
public class CommandLineFlinkJobsLauncher implements FlinkJobsLauncher {

	private static final List<String> COMMAND_BEGIN = System.getProperty("os.name", "").toLowerCase()
			.contains("windows") ? Arrays.asList("cmd", "/C") : Arrays.asList("/bin/sh", "-c");

	private static final String APPLICATION_ID_PREFFIX = "Submitting application master",

			CURRENT_STATE_PREFFIX = "Deploying cluster, current state",

			RUNNING_LOG = "YARN application has been deployed successfully.",

			FINISHED_LOG = "YARN application has been finished successfully.";

	private static int APPLICATION_ID_PREFFIX_LEN = APPLICATION_ID_PREFFIX.length(),
			CURRENT_STATE_PREFFIX_LEN = CURRENT_STATE_PREFFIX.length();

	private static Set<String> TARGETS = Sets.as("-t", "--target");

	private String flinkHome;

	private Action action = Action.RUN_APPLICATION;

	public String getFlinkHome() {
		return flinkHome;
	}

	public void setFlinkHome(String flinkHome) {
		this.flinkHome = flinkHome;
	}

	public Action getAction() {
		return action;
	}

	public void setAction(Action action) {
		this.action = action;
	}

	@Override
	public FlinkJobsApplicationInfo launch(FlinkJobs config) throws Exception {
		List<String> commandWords = new ArrayList<String>();
		commandWords.addAll(COMMAND_BEGIN);
		commandWords.add(flinkHome + "/bin/flink");
		commandWords.add(action.getName());

		attachOptions(commandWords, config.getOptions());
		String mainClass = config.getMainClass();
		if (mainClass != null) {
			commandWords.add("-c");
			commandWords.add(mainClass);
		}
		commandWords.add(config.getJar());
		Params params = config.getParams();
		if (params != null) {
			commandWords.add(JSON.toJSONString(params));
		}

		String[] cmdarray = commandWords.toArray(new String[commandWords.size()]);
		System.out.println("Execute command: " + String.join(" ", commandWords));

		Process p = null;
		InputStream is = null;
		BufferedReader r = null;
		try {
			p = Runtime.getRuntime().exec(cmdarray, null, new File(flinkHome));
			is = p.getInputStream();
			r = new BufferedReader(new InputStreamReader(is));
			String line, applicationId = null;
			State state = State.SUBMITTED;
			while ((line = r.readLine()) != null) {// 收集日志信息，并记录applicationId和state
				System.out.println(line);
				int index = line.indexOf(APPLICATION_ID_PREFFIX);
				if (index >= 0) {
					applicationId = line.substring(index + APPLICATION_ID_PREFFIX_LEN).trim();
				} else {
					index = line.indexOf(CURRENT_STATE_PREFFIX);
					if (index >= 0) {
						try {
							state = State.valueOf(line.substring(index + CURRENT_STATE_PREFFIX_LEN).trim());
						} catch (Exception e) {
							state = State.SUBMITTED;
						}
					} else if (line.contains(RUNNING_LOG)) {
						state = State.RUNNING;
					} else if (line.contains(FINISHED_LOG)) {
						state = State.FINISHED;
					}
				}
			}
			p.waitFor();
			return FlinkJobsApplicationInfoBuilder.build(applicationId, state);
		} catch (Exception e) {
			throw e;
		} finally {
			if (is != null) {
				is.close();
			}
			if (r != null) {
				r.close();
			}
			if (p != null) {
				p.destroy();
			}
		}
	}

	protected void attachOptions(List<String> commandWords, Options options) {
		String targetKey = null, target = null;
		if (options != null) {
			String preffix = options.getKeyPreffix(), key, value;
			List<Option> option = options.getOption();
			if (option != null) {
				for (Iterator<Option> it = option.iterator(); it.hasNext();) {
					Option entry = it.next();
					key = entry.getKey();
					if (!key.startsWith("-")) {
						key = preffix + key;
					}
					value = entry.getValue();
					if (TARGETS.contains(key)) {
						if (value != null && !value.isEmpty()) {
							targetKey = key;
							target = value;
						}
					} else {
						commandWords.add(key);
						if (value != null && !value.isEmpty()) {
							commandWords.add(value);
						}
					}
				}
			}
		}
		if (target == null) {
			if (Action.RUN_APPLICATION.equals(action)) {
				commandWords.add("-t");
				commandWords.add("yarn-application");
			}
		} else {
			commandWords.add(targetKey);
			commandWords.add(target);
		}
	}

	public static enum Action {
		RUN("run"), RUN_APPLICATION("run-application");

		private String name;

		private Action(String name) {
			this.name = name;
		}

		public String getName() {
			return name;
		}
	}

}
