package cn.tenfell.common.redis.queue;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import cn.tenfell.common.core.SpringUtil;
import cn.tenfell.common.core.ThreadKeepActiveUtil;
import cn.tenfell.common.redis.emu.ReceiveStatus;
import cn.tenfell.common.redis.util.RedisUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.RedisTemplate;

import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
/**
 * redis消息队列
 * @author fs
 * @param <E> 消息元素类型
 */
@Slf4j
public class RedisQueue<E>  {
    /**
     * redis操作器
     */
    private static RedisTemplate redisTemplate;
    /**
     * 保活线程map
     * key 实现类的bean名称
     * val 保活线程id集合
     */
    private static Map<String,List<String>> threadMap = new ConcurrentHashMap<>();
    /**
     * RedisQueue享元池
     * key:topic:tag
     */
    private static Map<String,RedisQueue> queueMap = new ConcurrentHashMap<>();
    /**
     * redis操作器初始化
     */
    public static void init(){
        if(RedisQueue.redisTemplate != null){
            return;
        }
        RedisUtil.getRedisTemplate(new Consumer<RedisTemplate>() {
            @Override
            public void accept(RedisTemplate redisTemplate) {
                RedisQueue.redisTemplate = redisTemplate;
                RedisQueue.receiveInit();
            }
        });
    }

    /**
     * 消息队列接收器初始化
     */
    private static void receiveInit(){
        SpringUtil.asyncAction(new Consumer<ApplicationContext>() {
            @Override
            public void accept(ApplicationContext applicationContext) {
                String[] names = applicationContext.getBeanNamesForType(RedisMessageQueue.class);
                for(String name:names){
                    final RedisMessageQueue rmq = applicationContext.getBean(name, RedisMessageQueue.class);
                    final RedisQueue rq = rmq.getRedisQueue();
                    //消息队列接收数据
                    RedisQueueReceive rqr = new RedisQueueReceive(rmq,rq);
                    List<String> threads = RedisQueue.threadMap.get(name);
                    if( threads == null){
                        threads = new Vector<>();
                        int counts = rmq.count();
                        if(counts < 1){
                            counts = 1;
                        }
                        for(int i=0;i<counts;i++){
                            threads.add(IdUtil.simpleUUID());
                        }
                        RedisQueue.threadMap.put(name,threads);
                    }
                    for(String id:threads){
                        ThreadKeepActiveUtil.keepActive(id,rqr);
                    }
                }
            }
        });
    }

    /**
     * 消息接收器
     */
    private static class RedisQueueReceive implements Runnable{
        RedisMessageQueue redisMessageQueue;
        RedisQueue redisQueue;
        public RedisQueueReceive(RedisMessageQueue redisMessageQueue,RedisQueue redisQueue){
            this.redisMessageQueue = redisMessageQueue;
            this.redisQueue = redisQueue;
        }
        @Override
        public void run() {
            while(true){
                try{
                    Object data = this.redisQueue.poll(20L);
                    if(data == null){
                        continue;
                    }
                    ReceiveStatus receiveStatus = ReceiveStatus.FAILED;
                    try{
                        receiveStatus = this.redisMessageQueue.receive(data);
                    }catch (Exception e){
                        log.error("消费失败",e);
                    }
                    if(receiveStatus == ReceiveStatus.FAILED){
                        this.redisQueue.send(data);
                    }
                }catch (Exception e){
                    log.trace("redis队列接收阻塞:",e);
                }
            }
        }
    }
    String key;
    ListOperations<String,E> listOperations;
    public static RedisQueue getInstance(String topic,String tag){
        if(StrUtil.isBlank(topic)){
            topic = "queue";
        }
        if(StrUtil.isBlank(tag)){
            tag = "tag";
        }
        String key = topic+":"+tag;
        RedisQueue queue = queueMap.get(key);
        if(queue != null){
            return queue;
        }
        queue = new RedisQueue(key);
        queueMap.put(key,queue);
        return queue;
    }
    public static RedisQueue getInstance(String tag){
        return getInstance(null,tag);
    }

    /**
     * 构造方法1
     * @param key 消息线+标记
     */
    private RedisQueue(String key){
        RedisQueue that = this;
        that.key = key;
        that.listOperations = RedisQueue.redisTemplate.opsForList();
    }

    /**
     * 向队列中插入数据(非阻塞)
     * @param data 插入的数据
     * @return 返回的结果
     */
    public boolean send(E data) {
        if(StrUtil.isBlank(this.key) || data == null){
            return false;
        }
        Long length = this.listOperations.leftPush(this.key+":master", data);
        return length>0;
    }

    /**
     * 清空队列
     */
    public void empty() {
        if (StrUtil.isBlank(this.key)) {
            return;
        }
        RedisQueue.redisTemplate.delete(this.key);
    }

    /**
     * 从队列读数据阻塞(读取+删除)
     * @return 当前数据
     */
    private E poll(Long seconds) {
        if(StrUtil.isBlank(this.key)){
            return null;
        }
        if(seconds==null || seconds<5L){
            seconds = 5L;
        }
        E data = this.listOperations.rightPop(this.key+":master",seconds, TimeUnit.SECONDS);
        return data;
    }

    /**
     * 队列长度
     * @return 队列长度
     */
    public Long size(){
        if(StrUtil.isBlank(this.key)){
            return 0L;
        }
        Long length = this.listOperations.size(this.key+":master");
        if(length == null){
            length = 0L;
        }
        return length;
    }
}
