package cn.tenfell.common.redis;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.lang.Dict;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import cn.tenfell.common.core.SpringIocHolder;
import cn.tenfell.common.core.ThreadKeepActiveUtil;
import org.springframework.context.ApplicationContext;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.RedisTemplate;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
/**
 * redis消息队列
 * @author fs
 * @param <E> 消息元素类型
 */
public class RedisQueue<E>  {
    //类初始化--开始
    /**
     * redis操作器
     */
    private static RedisTemplate redisTemplate;
    /**
     * 保活线程map
     * key 实现类的bean名称
     * val 保活线程id集合
     */
    private static Map<String,List<String>> threadMap = new HashMap<>();
    static {
        init();
    }

    /**
     * redis操作器初始化
     */
    public static void init(){
        if(RedisQueue.redisTemplate != null){
            return;
        }
        RedisUtils.getRedisTemplate(new Consumer<RedisTemplate>() {
            @Override
            public void accept(RedisTemplate redisTemplate) {
                RedisQueue.redisTemplate = redisTemplate;
                RedisQueue.receiveInit();
            }
        });
    }

    /**
     * 消息队列接收器初始化
     */
    private static void receiveInit(){
        SpringIocHolder.asyncAction(new Consumer<ApplicationContext>() {
            @Override
            public void accept(ApplicationContext applicationContext) {
                String[] names = applicationContext.getBeanNamesForType(RedisMessageQueue.class);
                for(String name:names){
                    final RedisMessageQueue rmq = applicationContext.getBean(name, RedisMessageQueue.class);
                    final RedisQueue rq = rmq.getRedisQueue();
                    //消息队列接收数据
                    RedisQueueReceive rqr = new RedisQueueReceive(rmq,rq);
                    List<String> threads = RedisQueue.threadMap.get(name);
                    if( threads == null){
                        threads = new ArrayList<>();
                        int counts = rmq.count();
                        if(counts < 1){
                            counts = 1;
                        }
                        for(int i=0;i<counts;i++){
                            threads.add(IdUtil.simpleUUID());
                        }
                        RedisQueue.threadMap.put(name,threads);
                    }
                    for(String id:threads){
                        ThreadKeepActiveUtil.keepActive(id,rqr);
                    }
                    //将未消费的队列复制到master
                    RedisQueueOffer copyBakToMaster = new RedisQueueOffer(rq);
                    ThreadKeepActiveUtil.keepActive(IdUtil.simpleUUID(),copyBakToMaster);
                }
            }
        });
    }
    private static class RedisQueueReceive implements Runnable{
        RedisMessageQueue redisMessageQueue;
        RedisQueue redisQueue;
        public RedisQueueReceive(RedisMessageQueue redisMessageQueue,RedisQueue redisQueue){
            this.redisMessageQueue = redisMessageQueue;
            this.redisQueue = redisQueue;
        }
        @Override
        public void run() {
            while(true){
                try{
                    Object data;
                    String id;
                    if(this.redisMessageQueue.confirm()){
                        Dict dict = this.redisQueue.pollAndBak(30L);
                        data = dict.getBean("data");
                        id = dict.getStr("id");
                    }else{
                        data = this.redisQueue.poll(30L);
                        id = null;
                    }
                    this.redisMessageQueue.receive(data);
                    if(this.redisMessageQueue.confirm()){
                        this.redisQueue.delBak(id);
                    }
                }catch (Exception e){

                }
            }
        }
    }
    private static class RedisQueueOffer implements Runnable{
        RedisQueue redisQueue;
        public RedisQueueOffer(RedisQueue redisQueue){
            this.redisQueue = redisQueue;
        }
        @Override
        public void run() {
            Set<String> keys = RedisQueue.redisTemplate.keys(this.redisQueue.key+":bak:*");
            for(String key:keys){
                String[] sz = key.split(":");
                Long time = Convert.toLong(sz[sz.length-2]);
                if(System.currentTimeMillis() - time < 5*60*000){
                    //5分钟内的暂时不处理
                    continue;
                }
                this.redisQueue.operation.rightPopAndLeftPush(key,this.redisQueue.key+":master");
            }
        }
    }
    //类初始化--结束



    String key;
    ListOperations<String,E> operation;
    /**
     * 构造方法1
     * @param topic 消息线 不存在则用queue代替
     * @param tag 消息标记
     */
    public RedisQueue(String topic,String tag){
        RedisQueue that = this;
        if(StrUtil.isBlank(topic)){
            topic = "queue";
        }
        that.key = topic+":"+tag;
        that.operation = RedisQueue.redisTemplate.opsForList();
    }

    /**
     * 构造方法2
     * @param tag 消息标记
     */
    public RedisQueue(String tag){
        this(null,tag);
    }

    /**
     * 向队列中插入数据(非阻塞)
     * @param data 插入的数据
     * @return 返回的结果
     */
    public boolean offer(E data) {
        if(StrUtil.isBlank(this.key) || data == null){
            return false;
        }
        Long length = this.operation.leftPush(this.key+":master", data);
        return length>0;
    }

    /**
     * 清空队列
     */
    public void empty() {
        if (StrUtil.isBlank(this.key)) {
            return;
        }
        RedisQueue.redisTemplate.delete(this.key);
    }

    /**
     * 从队列读数据非阻塞(读取+删除)
     * @return 当前数据
     */
    public E poll() {
        if(StrUtil.isBlank(this.key)){
            return null;
        }
        E data = this.operation.rightPop(this.key+":master");
        return data;
    }
    /**
     * 从队列读数据阻塞(读取+删除)
     * @return 当前数据
     */
    public E poll(Long seconds) {
        if(StrUtil.isBlank(this.key)){
            return null;
        }
        if(seconds==null || seconds<5L){
            seconds = 30L;
        }
        E data = this.operation.rightPop(this.key+":master",seconds, TimeUnit.SECONDS);
        return data;
    }

    /**
     * 从队列读数据非阻塞(读取+删除+备份)
     * @return 当前数据 data:队列数据  id:备份的id
     */
    public Dict pollAndBak() {
        if(StrUtil.isBlank(this.key)){
            return null;
        }
        String id = System.currentTimeMillis() +":"+IdUtil.getSnowflake(1,1).nextIdStr();
        E data = this.operation.rightPopAndLeftPush(this.key+":master",this.key+":bak:"+id);
        return Dict.create().set("data",data).set("id",id);
    }
    /**
     * 从队列读数据阻塞(读取+删除+备份)
     * @return 当前数据 data:队列数据  id:备份的id
     */
    public Dict pollAndBak(Long seconds) {
        if(StrUtil.isBlank(this.key)){
            return null;
        }
        if(seconds==null || seconds<5L){
            seconds = 30L;
        }
        String id = System.currentTimeMillis() +":"+IdUtil.getSnowflake(1,1).nextIdStr();
        E data = this.operation.rightPopAndLeftPush(this.key+":master",this.key+":bak:"+id,seconds, TimeUnit.SECONDS);
        return Dict.create().set("data",data).set("id",id);
    }

    /**
     * 手动从备份队列中删除
     */
    public void delBak(String id){
        this.operation.rightPop(this.key+":bak:"+id);
    }

    /**
     * 手动确认数据,未确认的继续留在队列
     * @param consumer 确认的动作
     */
    public void ackPoll(Consumer<E> consumer){
        Dict dict = this.pollAndBak();
        if(dict == null){
            return;
        }
        E data = dict.getBean("data");
        String id = dict.getStr("id");
        if(data == null){
            return;
        }
        try{
            consumer.accept(data);
            //执行完成后删除
            delBak(id);
        }catch (Exception e){
            e.printStackTrace();
        }
    }

    /**
     * 队列长度
     * @return 队列长度
     */
    public long size(){
        if(StrUtil.isBlank(this.key)){
            return 0L;
        }
        Long length = this.operation.size(this.key+":master");
        if(length == null){
            length = 0L;
        }
        return length;
    }

    /**
     * 消息队列接收器
     * 实现此抽象类的Spring实例即可通过receive方法实时接收队列数据
     * @param <T>
     */
    public static abstract class RedisMessageQueue<T>{
        /**
         * 消息订阅
         * @param data 消息体
         */
        public abstract void receive(T data);

        /**
         * 是否需要确认 true需要确认,当receive异常的时候将数据放入队列,false不需要确认,receive无论是否异常均消费此数据
         */
        public abstract boolean confirm();
        /**
         * 线程线程数
         */
        public abstract int count();

        /**
         * 获取队列实体
         * @return
         */
        public abstract RedisQueue<T> getRedisQueue();
        /**
         * 消息发送
         */
        public boolean send(T data){
            RedisQueue<T> redisQueue = this.getRedisQueue();
            return redisQueue.offer(data);
        }
    }
}
