package cn.tdchain.cb.util;

import java.util.HashMap;
import java.util.Map;

import cn.tdchain.Trans;
import cn.tdchain.Trans.Category;
import cn.tdchain.cb.domain.BaseEntity;
import cn.tdchain.cipher.Cipher;
import cn.tdchain.cipher.Key;

/**
 * Utility for creating transaction.
 *
 * @version 1.0
 * @author Houmj 2019-01-31
 */
public class TransUtils {

    private TransUtils() {
    }

    /**
     * Generate transaction with new key.
     * 
     * @param entity data object
     * @param account application account
     * @param timestamp client time stamp
     * @return transaction
     */
    public static Trans createTrans(BaseEntity entity, String account,
                                    Long timestamp) {
        return createTrans(entity, account, timestamp, null, -1L);
    }

    /**
     * Generate transaction with exist key.
     * 
     * @param entity data object
     * @param account application account
     * @param timestamp client time stamp
     * @param oldVersion latest account version
     * @return transaction
     */
    public static Trans createTrans(BaseEntity entity, String account,
                                    Long timestamp, Long oldVersion) {
        return createTrans(entity, account, timestamp, null, oldVersion);
    }

    /**
     * Generate transaction with base map.
     * 
     * @param entity data object
     * @param account application account
     * @param timestamp client time stamp
     * @param baseMap base map
     * @return transaction
     */
    public static Trans createTrans(BaseEntity entity, String account,
                                    Long timestamp, Map<String, Long> baseMap) {
        return createTrans(entity, account, timestamp, baseMap, -1L);
    }

    /**
     * Generate transaction with base map and exist key.
     * 
     * @param entity data object
     * @param account application account
     * @param timestamp client time stamp
     * @param baseMap base map
     * @param oldVersion latest account version
     * @return transaction
     */
    public static Trans createTrans(BaseEntity entity, String account,
                                    Long timestamp, Map<String, Long> baseMap,
                                    Long oldVersion) {
        Trans tx = new Trans();
        tx.setAccount(account);
        tx.setCategory(Category.contract);
        tx.setKey(entity.getKey());
        tx.setType(entity.getType());
        tx.setData(entity.toString());
        if (CollectionUtils.isNotEmpty(baseMap)) {
            tx.setKeyVersionMap(new HashMap<String, Long>());
            tx.setKeyVersionMap(baseMap);
        }
        tx.setTimestamp(timestamp);
        tx.setVersion(oldVersion + 1);
        tx.upHash();
        return tx;
    }

    /**
     * Generate transaction with new key and signature.
     * 
     * @param entity data object
     * @param account application account
     * @param timestamp client time stamp
     * @param cipher Cipher
     * @param key Key
     * @return transaction
     */
    public static Trans createSignedTrans(BaseEntity entity, String account,
                                          Long timestamp, Cipher cipher,
                                          Key key) {
        return createSignedTrans(entity, account, timestamp, null, -1L, cipher,
                key);
    }

    /**
     * Generate transaction with exist key and signature.
     * 
     * @param entity data object
     * @param account application account
     * @param timestamp client time stamp
     * @param oldVersion latest account version
     * @param cipher Cipher
     * @param key Key
     * @return transaction
     */
    public static Trans createSignedTrans(BaseEntity entity, String account,
                                          Long timestamp, Long oldVersion,
                                          Cipher cipher, Key key) {
        return createSignedTrans(entity, account, timestamp, null, oldVersion,
                cipher, key);
    }

    /**
     * Generate transaction with base map and signature.
     * 
     * @param entity data object
     * @param account application account
     * @param timestamp client time stamp
     * @param baseMap base map
     * @param cipher Cipher
     * @param key Key
     * @return transaction
     */
    public static Trans createSignedTrans(BaseEntity entity, String account,
                                          Long timestamp,
                                          Map<String, Long> baseMap,
                                          Cipher cipher, Key key) {
        return createSignedTrans(entity, account, timestamp, baseMap, -1L,
                cipher, key);
    }

    /**
     * Generate transaction with base map, exist key and signature.
     * 
     * @param entity data object
     * @param account application account
     * @param timestamp client time stamp
     * @param baseMap base map
     * @param oldVersion latest account version
     * @param cipher Cipher
     * @param key Key
     * @return transaction
     */
    public static Trans createSignedTrans(BaseEntity entity, String account,
                                          Long timestamp,
                                          Map<String, Long> baseMap,
                                          Long oldVersion, Cipher cipher,
                                          Key key) {
        Trans tx = new Trans();
        tx.setAccount(account);
        tx.setCategory(Category.contract);
        tx.setKey(entity.getKey());
        tx.setType(entity.getType());
        tx.setData(entity.toString());
        if (CollectionUtils.isNotEmpty(baseMap)) {
            tx.setKeyVersionMap(new HashMap<String, Long>());
            tx.setKeyVersionMap(baseMap);
        }
        tx.setTimestamp(timestamp);
        tx.setVersion(oldVersion + 1);
        tx.upHash();
        String signature = cipher.signByPrivateKey(tx.getHash(),
                key.getPrivateKey());
        tx.setSignMap(new HashMap<String, String>());
        tx.getSignMap().put(key.getPublicKey(), signature);
        return tx;
    }

}
