/*
 * Copyright (c) 2017 Beijing Tiande Technology Co., Ltd.
 * All Rights Reserved.
 */
package cn.tdchain.jbcc;

import cn.tdchain.*;
import cn.tdchain.cipher.Cipher;
import cn.tdchain.cipher.CipherException;
import cn.tdchain.cipher.Key;
import cn.tdchain.cipher.utils.HashCheckUtil;
import cn.tdchain.jbcc.net.Net;
import cn.tdchain.jbcc.net.info.Node;
import cn.tdchain.jbcc.net.nio.NioNet;
import cn.tdchain.jbcc.rpc.RPCBatchResult;
import cn.tdchain.jbcc.rpc.RPCMessage;
import cn.tdchain.jbcc.rpc.RPCResult;
import cn.tdchain.tdmsp.Msp;
import cn.tdchain.tdmsp.accessctl.Permission;
import cn.tdchain.tdmsp.accessctl.SystemDefKey;
import cn.tdchain.tdmsp.util.MemberUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.security.KeyStore;
import java.security.cert.X509Certificate;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static cn.tdchain.jbcc.ConnectionFactory.DEFAULT_TIMEOUT;
import static cn.tdchain.jbcc.ConnectionFactory.MAX_TRANS_COUNT;
import static cn.tdchain.jbcc.ConnectionFactory.MAX_TRANS_HISTORY_COUNT;
import static cn.tdchain.jbcc.TransUtil.HASH_LENGTH;

/**
 * function：description
 * datetime：2019-03-27 13:07
 * author：warne
 */
public class Connection {

    /**
     * Description: 添加一笔交易
     *
     * @param trans
     * @return TransHead
     */
    public Result<TransHead> addTrans(Trans trans) {
        //# 判断交易null
        if (trans == null) {
            return new Result<>("trans is null");
        }
        BatchTrans<Trans> batch = new BatchTrans<>();
        batch.setConnectionId(this.connectionId);
        batch.addTransToBatch(trans);
        Result<BatchTrans<TransHead>> rpcResult = addBatchTrans(batch);
        Result<TransHead> result = new Result<>();
        result.setStatus(rpcResult.getStatus());
        result.setMsg(rpcResult.getMsg());
        result.setHashs(rpcResult.getHashs());
        if (!rpcResult.isSuccess()) {
            return result;
        }
        BatchTrans<TransHead> batchTrans = rpcResult.getEntity();
        result.setEntity(batchTrans.oneTrans());
        return result;
    }

    /**
     * Description: 添加一笔系统权限交(只有管理员证书才能使用)
     *
     * @param permission
     * @return TransHead
     */
    public Result<TransHead> addPermissionTrans(Permission permission) {
        //# 判断交易null
        if (permission == null) {
            return new Result<>("permission is null");
        }
        ChainTrans trans = new ChainTrans();
        permission.setSender(this.getAccount());
        trans.setKey(SystemDefKey.SYS_MEMBER_ACL_TACTIC_CTR.name());
        String data = JSON.toJSONString(permission);
        trans.setData(data);
        return this.addTrans(trans);
    }

    /**
     * Description:查询系统权限交易(管理员才能读取)
     *
     * @return
     */
    public Result<Trans> getPermissionTrans() {
        return getNewSystemTransByKey(SystemDefKey.SYS_MEMBER_ACL_TACTIC_CTR.name());
    }

    /**
     * Description: 根据key获取系统交易(只有管理员才能使用该方法,使用该方法只能查询系统交易,查询普通交易请使用getNewTransByKey)
     *
     * @param key
     * @return TransHead
     */
    public Result<Trans> getNewSystemTransByKey(String key) {
        Result<Trans> result = new Result<>();
        if (StringUtils.isBlank(key)) {
            result.setMsg("key is blank");
            return result;
        }
        key = key.trim();
        if (SQLCheckUtil.checkSQLError(key)) { //可能存在sql注入的key
            result.setMsg("key is invalid");
            return result;
        }
        Map<String, String> command = new HashMap<>();
        command.put("key", key);
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_SYSTEM_TRANS_KEY);

        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 5000);
        return batchResult.getResult(new TypeReference<Trans>() {
        });
    }

    /**
     * Description: 批量交易
     *
     * @param batch
     * @return BatchTrans<TransHead>
     */
    public Result<BatchTrans<TransHead>> addBatchTrans(BatchTrans<Trans> batch) {
        //# 判断批量交易null
        if (batch == null || CollectionUtils.isEmpty(batch.getTransSet())) {
            return new Result<>("batch is empty ");
        }
        batch.setConnectionId(this.connectionId);
        batch.setAccount(account);
        HashSet<Trans> transSet = batch.getTransSet();
        for (Trans trans : transSet) {
            trans.setAccount(account);
            trans.upHash();
        }
        batch.check();
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(batch.getId());
        msg.setMsg(batch.toJsonString());
        msg.setTargetType(RPCMessage.TargetType.TX_WAIT);

        // 异步将交易发送给请求搬运工
        net.request(msg);

        // 异步等待交易返回
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 12000);// 12秒超时，如果共识超时也能知道交易是否成功
        Result<BatchTrans<TransHead>> result = batchResult.getResult(new TypeReference<BatchTrans<TransHead>>() {
        });
        // 超时增加返回
        if (result.isTimeout()) {
            Set<String> collect = batch.getTransSet().stream().map(TransHead::getHash).collect(Collectors.toSet());
            result.setHashs(collect);
        }
        return result;
    }

    /**
     * Description: 根据块高度查询块信息
     *
     * @param height
     * @return Block<Trans>
     */
    public Result<Block<Trans>> getBlock(long height) {

        Map<String, String> command = new HashMap<>();
        command.put("height", String.valueOf(height));
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_BLOCK);

        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 10000);
        return batchResult.getResult(new TypeReference<Block<Trans>>() {
        });
    }

    /**
     * Description: 根据height查询block header
     *
     * @param height
     * @return Block
     */
    public Result<BlockHead> getBlockHead(long height) {
        Map<String, String> command = new HashMap<>();
        command.put("height", String.valueOf(height));

        RPCMessage msg = new RPCMessage(connectionId);
        msg.setCommand(command);
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setTargetType(RPCMessage.TargetType.GET_BLOCK_HEADER);

        net.request(msg);
        //resphonse
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 10000);
        return batchResult.getResult(new TypeReference<BlockHead>() {
        });
    }

    /**
     * Description: 根据块高度查询块信息
     *
     * @return Block<Trans>
     */
    public Result<List<BlockHead>> getBlockHeadList(long startHeight, long endHeight) {
        Map<String, String> command = new HashMap<>();
        command.put("startHeight", String.valueOf(startHeight));
        command.put("endHeight", String.valueOf(endHeight));
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_BLOCK_HEADERS);

        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 10000);
        return batchResult.getResult(new TypeReference<List<BlockHead>>() {
        });
    }

    /**
     * Description: 查询最大的块header,不包含交易
     *
     * @return Block
     */
    public Result<BlockHead> getMaxBlockHead() {
        return getBlockHead(-1l);
    }

    /**
     * Description: 查询最大块信息
     *
     * @return Block<Trans>
     */
    public Result<Block<Trans>> getMaxBlock() {
        return getBlock(-1L);
    }

    /**
     * Description: 根据hash查询交易信息
     *
     * @param hash
     * @return Trans
     */
    public Result<Trans> getTransByHash(String hash) {
        Result<Trans> result = new Result<>();
        if (!HashCheckUtil.hashCheck(hash) || hash.length() <= HASH_LENGTH) {
            result.setMsg("invalid param");
            result.setStatus(RPCResult.StatusType.fail);
            return result;
        }
        Map<String, String> command = new HashMap<>();
        command.put("hash", hash);
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_HASH);
        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 3000);
        return batchResult.getResult(new TypeReference<Trans>() {
        });
    }

    /**
     * Description: 根据hashlist查询交易
     *
     * @param hashList
     * @return List<Trans>
     */
    public Result<List<Trans>> getTransListByHashList(List<String> hashList) {

        Result<List<Trans>> result = new Result<>();
        if (CollectionUtils.isEmpty(hashList)) {
            result.setMsg("invalid param");
            return result;
        }
        if (hashList.size() > MAX_TRANS_COUNT) {
            result.setMsg("hashList is too large [ less than or equal to  " + "MAX_TRANS_COUNT" + " ]");
        }
        Map<String, String> command = new HashMap<>();
        command.put("hashList", JSONObject.toJSONString(hashList));

        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_LIST);

        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 6000);
        return batchResult.getResult(new TypeReference<List<Trans>>() {
        });
    }

    /**
     * Description: 根据key维度查询交易
     *
     * @param key
     * @return Trans
     */
    public Result<Trans> getNewTransByKey(String key) {
        Result<Trans> result = new Result<>();
        if (StringUtils.isBlank(key)) {
            result.setMsg("key is blank");
            return result;
        }
        key = key.trim();
        if (SQLCheckUtil.checkSQLError(key)) { //可能存在sql注入的key
            result.setMsg("key is invalid");
            return result;
        }
        Map<String, String> command = new HashMap<>();
        command.put("key", key);
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_KEY);

        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 3000);
        return batchResult.getResult(new TypeReference<Trans>() {
        });
    }

    /**
     * 根据交易类type查询交易
     *
     * @param type
     * @return List<Trans>
     */
    public Result<List<Trans>> getTransListByType(String type) {
        Result<List<Trans>> result = new Result<>();
        if (StringUtils.isBlank(type)) {
            result.setMsg("type is blank");
            return result;
        }
        if (HashCheckUtil.illegalCharacterCheck(type) || type == null || type.length() > 45) {
            result.setMsg("type have Illegal character or length too long.");
            return result;
        }
        Map<String, String> command = new HashMap<>();
        command.put("type", type);
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_LIST_BY_TYPE);
        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 6000);
        return batchResult.getResult(new TypeReference<List<Trans>>() {
        });
    }

    /**
     * 根据account 和key查询历史
     *
     * @param account
     * @param key
     * @return
     */
    public Result<List<Trans>> getTransHistoryByAccountAndKey(String account, String key, int startIndex, int endIndex) {
        Result<List<Trans>> result = new Result<>();
        if (StringUtils.isBlank(account)) {
            result.setMsg("account is blank");
            return result;
        }
        if (StringUtils.isBlank(key)) {
            result.setMsg("key is blank");
            return result;
        }
        if (SQLCheckUtil.checkSQLError(key)) {
            result.setMsg("key is sql limit");
            return result;
        }
        Map<String, String> command = new HashMap<>();
        command.put("account", account);
        command.put("key", key);
        command.put("startIndex", String.valueOf(startIndex));
        command.put("endIndex", String.valueOf(endIndex));
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_ACCOUNT_KEY_TRANS_HISTORY);

        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 8000);
        return batchResult.getResult(new TypeReference<List<Trans>>() {
        });
    }


    /**
     * Description:根据key维度查询交易历史
     *
     * @param key
     * @param startIndex
     * @param endIndex
     * @return List<Trans>
     */
    public Result<List<Trans>> getTransHistoryByKey(String key, int startIndex, int endIndex) {
        Result<List<Trans>> result = new Result<>();
        if (StringUtils.isBlank(key)) {
            result.setMsg("key is blank");
            return result;
        }
        if (startIndex < 0 || startIndex > endIndex || ((endIndex - startIndex) >= MAX_TRANS_HISTORY_COUNT)) {
            result.setMsg("startIndex or endIndex error ;endIndex - startIndex not >=" + MAX_TRANS_HISTORY_COUNT);
            return result;
        }
        if (SQLCheckUtil.checkSQLError(key)) {
            result.setMsg("key is sql limit");
            return result;
        }
        Map<String, String> command = new HashMap<>();
        command.put("key", key);
        command.put("startIndex", startIndex + "");
        command.put("endIndex", endIndex + "");
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_HISTORY);
        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 8000);
        return batchResult.getResult(new TypeReference<List<Trans>>() {
        });
    }

    /**
     * Description:获取当前区块链的连接数
     *
     * @return int
     */
    public Result<Integer> getConnectionCount() {
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setTargetType(RPCMessage.TargetType.GET_CONNECTION_COUNT);
        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 8000);
        return batchResult.getResult(new TypeReference<Integer>() {
        });
    }

    /**
     * Description:查询账户account下的交易总数
     *
     * @return int
     */
    public Result<Integer> getNewTransCountByAccount(String account) {
        Result<Integer> result = new Result<>();
        if (StringUtils.isBlank(account)) {
            result.setMsg("account is blank");
            return result;
        }
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setTargetType(RPCMessage.TargetType.GET_ACCOUNT_TRANS_COUNT);
        msg.getCommand().put("account", account);
        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 8000);
        return batchResult.getResult(new TypeReference<Integer>() {
        });
    }

    /**
     * Description:查询当前账户下的交易总数
     *
     * @return int
     */
    public Result<Integer> getNewTransCount() {
        return this.getNewTransCountByAccount(this.account);
    }

    /**
     * Description:查询账户account下的交易集合
     *
     * @return int
     */
    public Result<List<Trans>> getNewTransListByAccount(String account, int limit, int offset) {
        Result<List<Trans>> result = new Result<>();
        if (StringUtils.isBlank(account)) {
            result.setMsg("account is blank");
            return result;
        }
        if (limit > 30 | limit < 1) {
            result.setMsg("limit is in the range of 1 -- 30");
            return result;
        }
        if (offset < 0) {
            result.setMsg("offset params should > 0");
            return result;
        }
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setTargetType(RPCMessage.TargetType.GET_ACCOUNT_TRANS_LIST);
        Map<String, String> command = msg.getCommand();
        command.put("account", account);
        command.put("limit", String.valueOf(limit));
        command.put("offset", String.valueOf(offset));
        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 8000);
        return batchResult.getResult(new TypeReference<List<Trans>>() {
        });
    }

    /**
     * Description:查询当前账户下的交易集合
     *
     * @return int
     */
    public Result<List<Trans>> getNewTransList(int limit, int offset) {
        return this.getNewTransListByAccount(this.account, limit, offset);
    }

    /**
     * Description:开启事务
     *
     * @param keys
     * @return boolean
     */
    public boolean startTransaction(String[] keys) {
        if (keys != null && keys.length > 0) {
            //获取事务对象
            Transaction transaction = new Transaction(keys);
            //将事务提交到事务池，如果成功当前现在则锁住所有关于key的添加操作，在事务池中最多保留事务对象到stop time时间
            return ManagerTransactionPool.register(transaction, 6000);
        } else {
            return false;
        }
    }

    /**
     * Description:关闭事务
     *
     * @param keys
     */
    public void stopTransaction(String[] keys) {
        if (keys != null && keys.length > 0) {
            //获取事务对象
            Transaction transaction = new Transaction(keys);
            ManagerTransactionPool.destroy(transaction);
        }
    }

    /**
     * Description:查询区块链交易总数
     *
     * @return Long
     */
    public Result<Long> getBlockChainTransCount() {
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setTargetType(RPCMessage.TargetType.GET_TOTAL_TRANS_COUNT);

        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 5000);
        return batchResult.getResult(new TypeReference<Long>() {
        });
    }

    /**
     * Description:根据account和key获取最新交易
     *
     * @return Long
     */
    public Result<Trans> getNewTransByAccountAndKey(String account, String key) {
        RPCMessage msg = new RPCMessage(connectionId);
        msg.getCommand().put("account", account);
        msg.getCommand().put("key", key);
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setTargetType(RPCMessage.TargetType.GET_ACCOUNT_KEY_TRANS);

        net.request(msg);
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 5000);
        return batchResult.getResult(new TypeReference<Trans>() {
        });
    }

    /**
     * Description: 获取当前系统全部节点的node对象集合
     *
     * @return List<Node>
     */
    public List<Node> getBlockChainNodeList() {
        List<Node> nodes = net.getNodes();
        return nodes;
    }

    /**
     * Description: 获取当前系统全部节点的node对象集合
     *
     * @return List<TransHead>
     */
    public Result<List<TransHead>> getNewTransListByAccount(String account) {
        if (account == null) {
            return new Result<>("account is null");
        }
        RPCMessage msg = new RPCMessage(connectionId);
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setTargetType(RPCMessage.TargetType.GET_NEW_TRANS_LIST);

        Map<String, String> command = new HashMap<>();
        command.put("account", account);
        msg.setCommand(command);
        net.request(msg);

        RPCBatchResult batchResult = net.resphone(msg.getMessageId(), 5000);
        return batchResult.getResult(new TypeReference<List<TransHead>>() {
        });
    }

    /**
     * @return String 获取当前connection对象的id
     */
    public String getId() {
        return this.connectionId;
    }


    /**
     * Description: 客户端构造器 ， 默认超时为 DEFAULT_TIMEOUT
     *
     * @param ipArr
     * @param port
     * @param token
     * @param ksPath
     * @param ksPasswd
     */
    protected Connection(String[] ipArr, int port, String token, String ksPath, String ksPasswd) {
        this(ipArr, port, token, DEFAULT_TIMEOUT, ksPath, ksPasswd);
    }

    /**
     * @param ipTable
     * @param port
     * @param key
     * @param token
     * @param cipher
     */
    protected Connection(String[] ipTable, int port, Key key, String token, Cipher cipher) {
        this(ipTable, port, key, token, cipher, 30000);
    }

    /**
     * @param ipTable
     * @param port
     * @param key
     * @param token
     * @param cipher
     * @param timeout
     */
    protected Connection(String[] ipTable, int port, Key key, String token, Cipher cipher, long timeout) {
        this.key = key;
        this.cipher = cipher;
        this.timeout = timeout;
        /**
         * @Description: 根据初始化的iptable长度计算最小吻合指数。
         */
        this.minSucces = PBFT.getMinByCount(ipTable.length);

        this.role = MemberUtil.generateRole(key);
        this.account = MemberUtil.generateAccount(key);

        //开启net网络
        openNet(ipTable, port, token, cipher);

        //异步定时同步在线nodes
        asynAskNodes(DEFAULT_TIMEOUT);
    }

    /**
     * DEFAULT_TIMEOUT
     *
     * @param ipArr
     * @param port
     * @param token
     * @param timeout
     * @param ksPath
     * @param ksPasswd
     */
    protected Connection(String[] ipArr, int port, String token, long timeout, String ksPath, String ksPasswd) {

        this.minSucces = PBFT.getMinByCount(ipArr.length);

        //# 准备证书
        readyCert(ksPath, ksPasswd);

        //# 开启网络
        openNet(ipArr, port, token);

        //# 同步节点
        asynAskNodes(timeout);
    }

    /**
     * Description:开启网络连接
     *
     * @param ipArr
     * @param port
     * @param token
     */
    protected void openNet(String[] ipArr, int port, String token) {
        // 验证token不能为空
        if (StringUtils.isBlank(token)) {
            throw new ParameterException("token is empty ");
        }
        net = new NioNet(ipArr, port, cipher, token, key, connectionId);
        net.start();
        Long start = System.currentTimeMillis();
        while (true) {
            if (net.getTaskSize() >= net.getMinNodeSize()) {
                break;
            }
            if (start - System.currentTimeMillis() > this.timeout) {
                net.stop();
                throw new ConnectionTimeOutException("Connection time out, iptables=" + StringUtils.join(ipArr, ", ") + ",port=" + port + ",token=" + token);
            }
            try {
                Thread.sleep(20);
            } catch (InterruptedException e) {
            }

        }
    }

    protected void openNet(String[] ipArr, int port, String token, Cipher cipher) {
        this.cipher = cipher;
        openNet(ipArr, port, token);
    }

    /**
     * Description:异步同步节点信息
     *
     * @param timeout
     */
    protected void asynAskNodes(long timeout) {
        //# 每隔3秒同步一次
        scheduledService.scheduleAtFixedRate(() -> {
            try {
                RPCMessage msg = new RPCMessage(connectionId);
                msg.setTargetType(RPCMessage.TargetType.GET_NODE_LIST);
                msg.setMessageId(UUID.randomUUID().toString());
                //异步提交请求
                net.request(msg);
                //resphonse
                List<Node> nodes = new ArrayList<>();
                RPCBatchResult batchResult = net.resphone(msg.getMessageId(), timeout);
                if (batchResult.isFail()) {
                    batchResult.getResult(); // 抛出异常？
                    return;
                }
                List<Result<List<Node>>> rpcResultList = batchResult.buildList(new TypeReference<List<Node>>() {
                });
                for (Result<List<Node>> result : rpcResultList) {
                    List<Node> nodeList = result.getEntity();
                    if (nodeList != null && nodeList.size() > 0) {
                        if (nodeList.size() > nodes.size()) {
                            nodes = nodeList;
                        }
                    }
                }
                //获取在线的节点添加到net中
                if (nodes != null && nodes.size() > 0) {
                    for (Node node : nodes) {
                        if (SoutUtil.isOpenSout())
                            System.out.println("copy node id:" + node.getId() + " status=" + node.getStatus());
                        net.addNodeToNodes(node);
                    }
                }
                Thread.sleep(1); //# 释放权限
            } catch (Exception e) {
                e.printStackTrace();
            }
        }, 0, 3, TimeUnit.SECONDS);

    }

    /**
     * Description:准备证书
     *
     * @param ksPath
     * @param ksPasswd
     */
    protected void readyCert(String ksPath, String ksPasswd) {
        try {
            String alias = Msp.ORGANIZATION_ALIAS;
            // 读取key store
            KeyStore keyStore = Msp.getKeyStore(ksPath, ksPasswd);

            String privateKey = cipher.getPrivateKeyStringByKeyStore(keyStore, ksPasswd, alias);
            String publicKey = cipher.getPublicKeyStringByStore(keyStore, ksPasswd, alias);

            //获取本地证书,该证书必须时由root证书颁发，否则无法与server建立连接。
            X509Certificate cert = (X509Certificate) keyStore.getCertificate(alias);
            String certBase64String = Msp.certToBase64String(cert);

            key.setPrivateKey(privateKey); //缓存私钥字符串
            key.setPublicKey(publicKey); // 缓存公钥字符串
            key.setLocalCertBase64String(certBase64String); //缓存证书base64字符串

            this.role = MemberUtil.generateRole(key);
            this.account = MemberUtil.generateAccount(key);
        } catch (Exception e) {
            throw new CipherException("get private key by key store error:" + e.getMessage());
        }
    }


    public String getAccount() {
        return account;
    }

    public String getRole() {
        return role;
    }

    private int minSucces = 1;
    protected Key key = new Key();
    protected Net net;
    protected String connectionId = UUID.randomUUID().toString();
    protected Cipher cipher = new Cipher();
    protected String account;
    protected String role;
    protected long timeout = 24000;
    protected final static ScheduledExecutorService scheduledService = Executors.newSingleThreadScheduledExecutor();

}

