package cn.tbox.sdk.model.message;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;

import cn.tbox.sdk.core.exception.TboxServerException;
import cn.tbox.sdk.core.http.HttpResponseEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Message parser.
 * Used to parse HTTP SSE response messages.
 */
public class MessageParser {
    private static final Logger logger = LoggerFactory.getLogger("tbox.client");

    /**
     * Holds the answers from multiple responses.
     * Used to save the structure and type of multiple return contents.
     * answers ==> {
     * "lane_key" :{ // Holds the response lane, default is "default", in the workflow you can see which response keys
     * "type" : "text", // Holds the response type: text, images, object, etc.
     * "data": {}, // Holds the final aggregated data of various types of response values
     * "header" : {}, // Holds the response header for each lane
     * "messages" : [] // Holds all chunk type messages, saved here
     * }
     * }
     */
    private final Map<String, Map<String, Object>> answersHolder = new HashMap<>();

    /**
     * Checks if the response event needs to be parsed.
     *
     * @param responseEvent HTTP response event
     * @return true if needs parsing, false otherwise
     */
    public boolean needParse(HttpResponseEvent responseEvent) {
        return "message".equals(responseEvent.getEvent()) || "error".equals(responseEvent.getEvent());
    }

    /**
     * Parses a response event.
     *
     * @param responseEvent HTTP response event
     * @return Parsed data as a map
     * @throws TboxServerException if there's a server error
     */
    public Map<String, Object> parse(HttpResponseEvent responseEvent) throws TboxServerException {
        if ("error".equals(responseEvent.getEvent())) {
            parseError(responseEvent);
            // This will never be reached due to exception throw, but keeps compiler happy
            return new HashMap<>();
        }

        if ("message".equals(responseEvent.getEvent())) {
            try {
                JSONObject data = JSON.parseObject(responseEvent.getData());
                String type = data.getString("type");

                switch (type) {
                    case "meta":
                        return parseMetaMessage(data);
                    case "header":
                        return parseHeaderMessage(data);
                    case "chunk":
                        return parseChunkMessage(data);
                    case "revoke":
                        return parseRevokeMessage(data);
                    case "error":
                        return parseErrorMessage(data);
                    case "charge":
                        return parseChargeMessage(data);
                    case "end":
                        return parseEndMessage(data);
                    case "unknown":
                        return parseUnknownMessage(data);
                    case "followup":
                        return parseFollowupMessage(data);
                    default:
                        return convertJsonNodeToMap(data);
                }
            } catch (JSONException e) {
                logger.error("Failed to parse response data", e);
                Map<String, Object> errorResult = new HashMap<>();
                errorResult.put("error", "Failed to parse response data: " + e.getMessage());
                return errorResult;
            }
        }

        try {
            return JSON.parseObject(responseEvent.getData(), HashMap.class);
        } catch (JSONException e) {
            logger.error("Failed to parse response data", e);
            Map<String, Object> errorResult = new HashMap<>();
            errorResult.put("error", "Failed to parse response data: " + e.getMessage());
            return errorResult;
        }
    }

    /**
     * Parses a chunk message.
     *
     * @param data Message data
     * @return Processed data
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> parseChunkMessage(JSONObject data) {
        String lane = data.getString("lane");
        if (lane == null) {
            lane = "default";
        }
        Map<String, Object> result = convertJsonNodeToMap(data);

        // Parse payload
        JSONObject payloadNode = data.getJSONObject("payload");
        String payloadText = payloadNode != null ? payloadNode.toJSONString() : "{}";
        try {
            JSONObject payload = JSON.parseObject(payloadText);
            result.put("payload", convertJsonNodeToMap(payload));

            if (answersHolder.containsKey(lane)) {
                String mediaType = (String) answersHolder.get(lane).get("type");
                if ("text".equals(mediaType)) {
                    Map<String, Object> payloadMap = (Map<String, Object>) result.get("payload");
                    String text = (String) payloadMap.getOrDefault("text", "");
                    String currentData = (String) answersHolder.get(lane).get("data");
                    answersHolder.get(lane).put("data", currentData + text);
                }

                List<Map<String, Object>> messages = (List<Map<String, Object>>) answersHolder.get(lane).get("messages");
                messages.add(result);
            }
        } catch (JSONException e) {
            logger.error("Failed to parse payload", e);
        }

        return result;
    }

    /**
     * Parses a meta message.
     *
     * @param data Message data
     * @return Processed data
     */
    private Map<String, Object> parseMetaMessage(JSONObject data) {
        return convertJsonNodeToMap(data);
    }

    /**
     * Parses a header message.
     *
     * @param data Message data
     * @return Processed data
     */
    private Map<String, Object> parseHeaderMessage(JSONObject data) {
        Map<String, Object> result = convertJsonNodeToMap(data);

        String lane = data.getString("lane");
        if (lane == null) {
            lane = "default";
        }

        // Parse payload
        JSONObject payloadNode = data.getJSONObject("payload");
        String payloadText = payloadNode != null ? payloadNode.toJSONString() : "{}";
        try {
            JSONObject payload = JSON.parseObject(payloadText);
            result.put("payload", convertJsonNodeToMap(payload));

            Map<String, Object> payloadMap = convertJsonNodeToMap(payload);
            String mediaType = (String) payloadMap.get("mediaType");

            // Write content to answer holder
            if (!answersHolder.containsKey(lane)) {
                Map<String, Object> laneData = new HashMap<>();
                laneData.put("type", mediaType);
                laneData.put("header", result);
                laneData.put("data", "");
                laneData.put("messages", new ArrayList<>());
                answersHolder.put(lane, laneData);
            } else {
                answersHolder.get(lane).put("header", result);
                answersHolder.get(lane).put("type", mediaType);
            }
        } catch (JSONException e) {
            logger.error("Failed to parse payload", e);
        }

        return result;
    }

    /**
     * Parses a revoke message.
     *
     * @param data Message data
     * @return Processed data
     */
    private Map<String, Object> parseRevokeMessage(JSONObject data) {
        return convertJsonNodeToMap(data);
    }

    /**
     * Parses an error message.
     *
     * @param data Message data
     * @return Processed data
     */
    private Map<String, Object> parseErrorMessage(JSONObject data) {
        return convertJsonNodeToMap(data);
    }

    /**
     * Parses a charge message.
     *
     * @param data Message data
     * @return Processed data
     */
    private Map<String, Object> parseChargeMessage(JSONObject data) {
        return convertJsonNodeToMap(data);
    }

    /**
     * Parses an end message.
     *
     * @param data Message data
     * @return Processed data
     */
    private Map<String, Object> parseEndMessage(JSONObject data) {
        return convertJsonNodeToMap(data);
    }

    /**
     * Parses an unknown message.
     *
     * @param data Message data
     * @return Processed data
     */
    private Map<String, Object> parseUnknownMessage(JSONObject data) {
        return convertJsonNodeToMap(data);
    }

    /**
     * Parses a followup message.
     *
     * @param data Message data
     * @return Processed data
     */
    private Map<String, Object> parseFollowupMessage(JSONObject data) {
        return convertJsonNodeToMap(data);
    }

    /**
     * Parses an error event.
     *
     * @param responseEvent HTTP response event
     * @throws TboxServerException Always thrown with error details
     */
    private void parseError(HttpResponseEvent responseEvent) throws TboxServerException {
        try {
            JSONObject errorContext = JSON.parseObject(responseEvent.getData());
            String message = "unknown error";

            if (errorContext.containsKey("payload")) {
                JSONObject payload = errorContext.getJSONObject("payload");
                if (payload != null) {
                    message = payload.getString("errorMsg");
                    if (message == null) {
                        message = "unknown error";
                    }
                }
            } else if (errorContext.containsKey("errorMsg")) {
                // Direct errorMsg in errorContext
                message = errorContext.getString("errorMsg");
                if (message == null) {
                    message = "unknown error";
                }
            }

            TboxServerException exception = new TboxServerException(message);
            exception.setErrorContext(errorContext);
            throw exception;
        } catch (JSONException e) {
            throw new TboxServerException("Error parsing server error response", e);
        }
    }

    /**
     * Gets the answers holder.
     *
     * @return The answers holder map
     */
    public Map<String, Map<String, Object>> getAnswersHolder() {
        return answersHolder;
    }

    /**
     * Converts a JSONObject to a Map.
     *
     * @param node JSONObject to convert
     * @return Converted Map
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> convertJsonNodeToMap(JSONObject node) {
        try {
            return JSON.parseObject(node.toJSONString(), HashMap.class);
        } catch (Exception e) {
            logger.error("Error converting JSONObject to Map", e);
            return new HashMap<>();
        }
    }
} 