package cn.tbox.sdk.example;

import cn.tbox.sdk.TboxClient;
import cn.tbox.sdk.model.request.MessageListRequest;
import cn.tbox.sdk.model.response.TboxResponse;
import cn.tbox.sdk.model.response.MessageListResponse;
import cn.tbox.sdk.model.message.Message;
import cn.tbox.sdk.model.message.MessageFile;
import cn.tbox.sdk.core.exception.TboxClientConfigException;
import cn.tbox.sdk.core.exception.TboxHttpResponseException;

import java.util.List;

/**
 * Example demonstrating how to use the message list API.
 */
public class MessageListExample {
    
    public static void main(String[] args) {
        try {
            // Create a Tbox client with your authorization token
            TboxClient client = new TboxClient("your_authorization_token_here");
            
            // Example 1: Basic message list request
            MessageListRequest basicRequest = new MessageListRequest("2025*****4259");
            TboxResponse<MessageListResponse> basicResponse = client.getMessages(basicRequest);
            
            if (basicResponse.isSuccess()) {
                System.out.println("Basic request successful!");
                List<Message> messages = basicResponse.getData().getMessages();
                for (Message message : messages) {
                    System.out.println("Message ID: " + message.getMessageId());
                    System.out.println("Conversation ID: " + message.getConversationId());
                    System.out.println("App ID: " + message.getAppId());
                    System.out.println("Query: " + message.getQuery());
                    System.out.println("Status: " + message.getStatus());
                    System.out.println("Created at: " + message.getCreateAt());
                    System.out.println("Updated at: " + message.getUpdateAt());
                    
                    // Print answers
                    if (message.getAnswers() != null) {
                        System.out.println("Answers:");
                        for (Message.Answer answer : message.getAnswers()) {
                            System.out.println("  - Lane: " + answer.getLane());
                            System.out.println("  - Media Type: " + answer.getMediaType());
                            System.out.println("  - Text: " + answer.getText());
                            if (answer.getUrl() != null) {
                                System.out.println("  - URLs: " + answer.getUrl());
                            }
                            System.out.println("  - Expire At: " + answer.getExpireAt());
                        }
                    }
                    
                    // Print files
                    if (message.getFiles() != null) {
                        System.out.println("Files:");
                        for (MessageFile file : message.getFiles()) {
                            System.out.println("  - Type: " + file.getType());
                            System.out.println("  - URL: " + file.getUrl());
                            System.out.println("  - Expire At: " + file.getExpireAt() + "s");
                        }
                    }
                    System.out.println("---");
                }
            } else {
                System.out.println("Request failed: " + basicResponse.getErrorMsg());
            }
            
            // Example 2: Advanced message list request with pagination and sorting
            MessageListRequest advancedRequest = new MessageListRequest();
            advancedRequest.setConversationId("2025*****4259");
            advancedRequest.setPageNum(1);
            advancedRequest.setPageSize(20);
            advancedRequest.setSortOrder(MessageListRequest.SortOrder.DESC);
            
            TboxResponse<MessageListResponse> advancedResponse = client.getMessages(advancedRequest);
            
            if (advancedResponse.isSuccess()) {
                System.out.println("Advanced request successful!");
                List<Message> messages = advancedResponse.getData().getMessages();
                System.out.println("Found " + messages.size() + " messages");
                
                // Filter messages by status
                messages.stream()
                    .filter(msg -> Message.Status.SUCCESS.equals(msg.getStatus()))
                    .forEach(msg -> {
                        System.out.println("Successful message: " + msg.getMessageId());
                    });
                
                // Find messages with images
                messages.stream()
                    .filter(msg -> msg.getAnswers() != null)
                    .flatMap(msg -> msg.getAnswers().stream())
                    .filter(answer -> Message.Answer.MediaType.IMAGE.equals(answer.getMediaType()))
                    .forEach(answer -> {
                        System.out.println("Image answer found: " + answer.getText());
                    });
            } else {
                System.out.println("Advanced request failed: " + advancedResponse.getErrorMsg());
            }
            
        } catch (TboxClientConfigException e) {
            System.err.println("Configuration error: " + e.getMessage());
        } catch (TboxHttpResponseException e) {
            System.err.println("HTTP response error: " + e.getMessage());
        } catch (Exception e) {
            System.err.println("Unexpected error: " + e.getMessage());
            e.printStackTrace();
        }
    }
} 