package cn.tbox.sdk.example;

import cn.tbox.sdk.TboxClient;
import cn.tbox.sdk.model.request.ChatRequest;
import cn.tbox.sdk.model.request.MessageListRequest;
import cn.tbox.sdk.model.response.TboxResponse;
import cn.tbox.sdk.model.response.SyncChatResponse;
import cn.tbox.sdk.model.response.MessageListResponse;
import cn.tbox.sdk.core.exception.TboxClientConfigException;
import cn.tbox.sdk.core.exception.TboxHttpResponseException;

import java.util.List;

/**
 * Example demonstrating a complete conversation workflow:
 * 1. Create a conversation
 * 2. Send a message
 * 3. Query message list
 */
public class ConversationWorkflowExample {
    
    public static void main(String[] args) {
        try {
            // Create a Tbox client with your authorization token
            TboxClient client = new TboxClient("your_authorization_token_here");
            String appId = "2025*****4259";
            String userId = "test_user_123";
            
            System.out.println("=== Starting Conversation Workflow ===");
            
            // Step 1: Create a new conversation
            System.out.println("\n1. Creating a new conversation...");
            TboxResponse<String> createResponse = client.createConversation(appId);
            
            if (!createResponse.isSuccess()) {
                System.err.println("Failed to create conversation: " + createResponse.getErrorMsg());
                return;
            }
            
            String conversationId = createResponse.getData();
            System.out.println("✓ Conversation created successfully!");
            System.out.println("  Conversation ID: " + conversationId);
            
            // Step 2: Send a message in the conversation
            System.out.println("\n2. Sending a message...");
            ChatRequest chatRequest = new ChatRequest(appId, "Hello, this is a test message!", userId, conversationId);
            chatRequest.setStream(false); // Use sync mode for this example
            
            TboxResponse<SyncChatResponse> chatResponse = (TboxResponse<SyncChatResponse>) client.chat(chatRequest);
            
            if (!chatResponse.isSuccess()) {
                System.err.println("Failed to send message: " + chatResponse.getErrorMsg());
                return;
            }
            
            System.out.println("✓ Message sent successfully!");
            System.out.println("  Message ID: " + chatResponse.getData().getMessageId());
            System.out.println("  Request ID: " + chatResponse.getData().getRequestId());
            
            // Print the response content
            if (chatResponse.getData().getResult() != null) {
                System.out.println("  Response content:");
                for (SyncChatResponse.Answer answer : chatResponse.getData().getResult()) {
                    System.out.println("    - " + answer.getChunk());
                }
            }
            
            // Step 3: Query message list for the conversation
            System.out.println("\n3. Querying message list...");
            MessageListRequest messageListRequest = new MessageListRequest();
            messageListRequest.setConversationId(conversationId);
            messageListRequest.setPageNum(1);
            messageListRequest.setPageSize(10);
            messageListRequest.setSortOrder(MessageListRequest.SortOrder.DESC);
            
            TboxResponse<MessageListResponse> messageListResponse = client.getMessages(messageListRequest);
            
            if (!messageListResponse.isSuccess()) {
                System.err.println("Failed to query messages: " + messageListResponse.getErrorMsg());
                return;
            }
            
            System.out.println("✓ Message list retrieved successfully!");
            List<cn.tbox.sdk.model.message.Message> messages = messageListResponse.getData().getMessages();
            System.out.println("  Found " + messages.size() + " messages in the conversation");
            
            // Display message details
            for (cn.tbox.sdk.model.message.Message message : messages) {
                System.out.println("\n  Message Details:");
                System.out.println("    Message ID: " + message.getMessageId());
                System.out.println("    Query: " + message.getQuery());
                System.out.println("    Status: " + message.getStatus());
                System.out.println("    Created: " + message.getCreateAt());
                
                // Display answers
                if (message.getAnswers() != null && !message.getAnswers().isEmpty()) {
                    System.out.println("    Answers:");
                    for (cn.tbox.sdk.model.message.Message.Answer answer : message.getAnswers()) {
                        System.out.println("      - " + answer.getText());
                    }
                }
            }
            
            System.out.println("\n=== Conversation Workflow Completed Successfully ===");
            
        } catch (TboxClientConfigException e) {
            System.err.println("Configuration error: " + e.getMessage());
        } catch (TboxHttpResponseException e) {
            System.err.println("HTTP response error: " + e.getMessage());
        } catch (Exception e) {
            System.err.println("Unexpected error: " + e.getMessage());
            e.printStackTrace();
        }
    }
} 