package cn.tbox.sdk;

import java.util.HashMap;
import java.util.Map;

import cn.tbox.sdk.core.exception.TboxClientConfigException;
import cn.tbox.sdk.core.exception.TboxHttpResponseException;
import cn.tbox.sdk.core.exception.TboxServerException;
import cn.tbox.sdk.core.http.HttpClient;
import cn.tbox.sdk.core.http.HttpClientConfig;
import cn.tbox.sdk.core.http.HttpResponseEvent;
import cn.tbox.sdk.model.message.MessageParser;
import cn.tbox.sdk.model.request.ChatRequest;
import cn.tbox.sdk.model.request.CompletionRequest;
import cn.tbox.sdk.model.request.ConversationListRequest;
import cn.tbox.sdk.model.request.MessageListRequest;
import cn.tbox.sdk.model.response.ConversationListResponse;
import cn.tbox.sdk.model.response.MessageListResponse;
import cn.tbox.sdk.model.response.TboxResponse;
import cn.tbox.sdk.model.response.SyncChatResponse;
import cn.tbox.sdk.model.response.SyncCompletionResponse;

import com.alibaba.fastjson.TypeReference;

/**
 * Tbox client for making API requests.
 */
public class TboxClient {
    /**
     * Tbox client configuration.
     */
    private final HttpClientConfig httpClientConfig;

    /**
     * HTTP client.
     */
    private final HttpClient httpClient;

    /**
     * Creates a new Tbox client.
     *
     * @param httpClientConfig HTTP client configuration
     * @throws TboxClientConfigException if there's a configuration error
     */
    public TboxClient(HttpClientConfig httpClientConfig) throws TboxClientConfigException {
        if (httpClientConfig == null) {
            throw new TboxClientConfigException("httpClientConfig is null");
        }
        this.httpClientConfig = httpClientConfig;
        this.httpClient = createHttpClient(this.httpClientConfig);
    }

    /**
     * Creates a new Tbox client with authorization.
     *
     * @param authorization Authorization token
     * @throws TboxClientConfigException if there's a configuration error
     */
    public TboxClient(String authorization) throws TboxClientConfigException {
        this.httpClientConfig = new HttpClientConfig();
        if (authorization != null) {
            this.httpClientConfig.setAuthorization(authorization);
        }
        this.httpClient = createHttpClient(this.httpClientConfig);
    }

    /**
     * Creates the HTTP client instance.
     * This method is protected to allow mocking in tests.
     *
     * @param config HTTP client configuration
     * @return HTTP client instance
     * @throws TboxClientConfigException if there's a configuration error
     */
    protected HttpClient createHttpClient(HttpClientConfig config) throws TboxClientConfigException {
        return new HttpClient(config);
    }

    /**
     * Makes a chat request to the Tbox API.
     * Used to call Tbox chat-type applications.
     *
     * @param chatRequest Chat request
     * @return TboxResponse containing SyncChatResponse or Iterable for streaming
     * @throws TboxClientConfigException if there's a configuration error
     * @throws TboxHttpResponseException if there's an HTTP response error
     */
    public Object chat(
        ChatRequest chatRequest
    ) throws TboxClientConfigException, TboxHttpResponseException {
        Map<String, Object> data = new HashMap<>();
        data.put("appId", chatRequest.getAppId());
        data.put("query", chatRequest.getQuery());
        data.put("userId", chatRequest.getUserId());

        if (chatRequest.getConversationId() != null) {
            data.put("conversationId", chatRequest.getConversationId());
        }
        if (chatRequest.getRequestId() != null) {
            data.put("requestId", chatRequest.getRequestId());
        }
        if (chatRequest.getClientProperties() != null) {
            data.put("clientProperties", chatRequest.getClientProperties());
        }
        if (chatRequest.getFiles() != null) {
            data.put("files", chatRequest.getFiles());
        }
        
        data.put("stream", chatRequest.getStream());

        if (chatRequest.getStream()) {
            Iterable<HttpResponseEvent> responseIter = httpClient.postStream("/api/chat", data, null, 110);
            return stream(responseIter, chatRequest.getMessageParser());
        } else {
            String responseText = httpClient.post("/api/chat", data, null);
            return new TboxResponse<>(responseText, new TypeReference<SyncChatResponse>(){});
        }
    }

    /**
     * Makes a completion request to the Tbox API.
     *
     * @param completionRequest Completion request
     * @return TboxResponse containing SyncCompletionResponse or Iterable for streaming
     * @throws TboxClientConfigException if there's a configuration error
     * @throws TboxHttpResponseException if there's an HTTP response error
     */
    public Object completion(
        CompletionRequest completionRequest
    ) throws TboxClientConfigException, TboxHttpResponseException {
        Map<String, Object> data = new HashMap<>();
        data.put("appId", completionRequest.getAppId());
        data.put("userId", completionRequest.getUserId());

        if (completionRequest.getRequestId() != null) {
            data.put("requestId", completionRequest.getRequestId());
        }
        if (completionRequest.getInputs() != null) {
            data.put("inputs", completionRequest.getInputs());
        }
        if (completionRequest.getClientProperties() != null) {
            data.put("clientProperties", completionRequest.getClientProperties());
        }
        if (completionRequest.getFiles() != null) {
            data.put("files", completionRequest.getFiles());
        }

        data.put("stream", completionRequest.getStream());

        if (completionRequest.getStream()) {
            Iterable<HttpResponseEvent> responseIter = httpClient.postStream("/api/completion", data, null, 110);
            return stream(responseIter, completionRequest.getMessageParser());
        } else {
            String responseText = httpClient.post("/api/completion", data, null);
            return new TboxResponse<>(responseText, new TypeReference<SyncCompletionResponse>(){});
        }
    }

    /**
     * Queries the conversation list for a specified intelligent agent.
     *
     * @param conversationListRequest Conversation list request
     * @return ConversationListResponse
     * @throws TboxHttpResponseException if there's an HTTP response error
     */
    public TboxResponse<ConversationListResponse> getConversations(
        ConversationListRequest conversationListRequest
    ) throws TboxClientConfigException, TboxHttpResponseException {
        if (conversationListRequest == null) {
            throw new TboxClientConfigException("conversationListRequest is null");
        }
        if (conversationListRequest.getAppId() == null || conversationListRequest.getAppId().trim().isEmpty()) {
            throw new TboxClientConfigException("appId is required");
        }

        Map<String, Object> params = new HashMap<>();
        params.put("appId", conversationListRequest.getAppId());

        if (conversationListRequest.getUserId() != null) {
            params.put("userId", conversationListRequest.getUserId());
        }
        if (conversationListRequest.getSource() != null) {
            params.put("source", conversationListRequest.getSource());
        }
        if (conversationListRequest.getPageNum() != null) {
            params.put("pageNum", conversationListRequest.getPageNum());
        }
        if (conversationListRequest.getPageSize() != null) {
            params.put("pageSize", conversationListRequest.getPageSize());
        }
        if (conversationListRequest.getSortOrder() != null) {
            params.put("sortOrder", conversationListRequest.getSortOrder());
        }

        Map<String, String> stringParams = new HashMap<>();
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            if (entry.getValue() != null) {
                stringParams.put(entry.getKey(), entry.getValue().toString());
            }
        }

        try {
            String responseText = httpClient.get("/api/conversation/conversations", stringParams, null);
            return new TboxResponse<>(responseText, new TypeReference<ConversationListResponse>(){});
        } catch (Exception e) {
            throw new TboxHttpResponseException("Failed to parse response", e);
        }
    }

    /**
     * Queries the message list for a specified conversation.
     *
     * @param messageListRequest Message list request
     * @return MessageListResponse
     * @throws TboxClientConfigException if there's a configuration error
     * @throws TboxHttpResponseException if there's an HTTP response error
     */
    public TboxResponse<MessageListResponse> getMessages(
        MessageListRequest messageListRequest
    ) throws TboxClientConfigException, TboxHttpResponseException {
        if (messageListRequest == null) {
            throw new TboxClientConfigException("messageListRequest is null");
        }
        if (messageListRequest.getConversationId() == null || messageListRequest.getConversationId().trim().isEmpty()) {
            throw new TboxClientConfigException("conversationId is required");
        }

        Map<String, Object> params = new HashMap<>();
        params.put("conversationId", messageListRequest.getConversationId());

        if (messageListRequest.getPageNum() != null) {
            params.put("pageNum", messageListRequest.getPageNum());
        }
        if (messageListRequest.getPageSize() != null) {
            params.put("pageSize", messageListRequest.getPageSize());
        }
        if (messageListRequest.getSortOrder() != null) {
            params.put("sortOrder", messageListRequest.getSortOrder());
        }

        Map<String, String> stringParams = new HashMap<>();
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            if (entry.getValue() != null) {
                stringParams.put(entry.getKey(), entry.getValue().toString());
            }
        }

        try {
            String responseText = httpClient.get("/api/conversation/messages", stringParams, null);
            return new TboxResponse<>(responseText, new TypeReference<MessageListResponse>(){});
        } catch (Exception e) {
            throw new TboxHttpResponseException("Failed to parse response", e);
        }
    }

    /**
     * Creates a new conversation.
     *
     * @param appId appId
     * @return conversationId
     * @throws TboxClientConfigException if there's a configuration error
     * @throws TboxHttpResponseException if there's an HTTP response error
     */
    public TboxResponse<String> createConversation(
        String appId
    ) throws TboxClientConfigException, TboxHttpResponseException {
        if (appId == null || appId.trim().isEmpty()) {
            throw new TboxClientConfigException("appId is required");
        }

        Map<String, Object> data = new HashMap<>();
        data.put("appId", appId);

        try {
            String responseText = httpClient.post("/api/conversation/create", data, null);
            return new TboxResponse<>(responseText, String.class);
        } catch (Exception e) {
            throw new TboxHttpResponseException("Failed to parse response", e);
        }
    }

    /**
     * Uploads a file to the Tbox API.
     *
     * @param filePath File path to upload
     * @return TboxResponse containing fileId
     * @throws TboxClientConfigException if there's a configuration error
     * @throws TboxHttpResponseException if there's an HTTP response error
     */
    public TboxResponse<String> uploadFile(
        String filePath
    ) throws TboxClientConfigException, TboxHttpResponseException {
        if (filePath == null || filePath.trim().isEmpty()) {
            throw new TboxClientConfigException("filePath is required");
        }
        java.io.File file = new java.io.File(filePath);
        if (!file.exists()) {
            throw new TboxClientConfigException("file does not exist: " + filePath);
        }

        try {
            String responseText = httpClient.uploadFile("/api/file/upload", file, null);
            return new TboxResponse<>(responseText, String.class);
        } catch (Exception e) {
            throw new TboxHttpResponseException("Failed to upload file", e);
        }
    }

    /**
     * Processes a stream of response events.
     *
     * @param responseIter  Response event iterator
     * @param messageParser Message parser (optional)
     * @return Iterable of parsed responses
     */
    private Iterable<Map<String, Object>> stream(Iterable<HttpResponseEvent> responseIter, MessageParser messageParser) {
        final MessageParser parser = messageParser != null ? messageParser : new MessageParser();

        return () -> new java.util.Iterator<Map<String, Object>>() {
            private final java.util.Iterator<HttpResponseEvent> eventIterator = responseIter.iterator();
            private Map<String, Object> nextData = null;
            private boolean hasNextCalled = false;

            @Override
            public boolean hasNext() {
                if (hasNextCalled) {
                    return nextData != null;
                }

                while (eventIterator.hasNext()) {
                    HttpResponseEvent event = eventIterator.next();

                    // Parse response, if it's an error event an exception will be thrown
                    if (parser.needParse(event)) {
                        try {
                            Map<String, Object> data = parser.parse(event);
                            String type = (String) data.get("type");

                            if ("meta".equals(type)) {
                                nextData = data;
                                hasNextCalled = true;
                                return true;
                            } else {
                                nextData = data;
                                hasNextCalled = true;
                                return true;
                            }
                        } catch (TboxServerException e) {
                            throw new RuntimeException(e.getMessage(), e);
                        }
                    }
                }

                hasNextCalled = true;
                nextData = null;
                return false;
            }

            @Override
            public Map<String, Object> next() {
                if (!hasNextCalled) {
                    hasNext();
                }

                if (nextData == null) {
                    throw new java.util.NoSuchElementException();
                }

                Map<String, Object> result = nextData;
                nextData = null;
                hasNextCalled = false;
                return result;
            }
        };
    }
} 