package cn.tbox.sdk.core.http;

import java.io.IOException;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;

import cn.tbox.sdk.core.exception.TboxClientConfigException;
import cn.tbox.sdk.core.exception.TboxHttpResponseException;
import okhttp3.Headers;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.sse.EventSource;
import okhttp3.sse.EventSourceListener;
import okhttp3.sse.EventSources;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * HTTP client for making API requests.
 */
public class HttpClient {
    private static final Logger       logger          = LoggerFactory.getLogger("tbox.client");
    private static final MediaType    JSON_MEDIA_TYPE = MediaType.parse("application/json; charset=utf-8");
    private static final OkHttpClient okHttpClient    = new OkHttpClient.Builder()
            .connectTimeout(Duration.ofSeconds(30))
            .readTimeout(Duration.ofSeconds(120))
            .writeTimeout(Duration.ofSeconds(60))
            .build();

    /**
     * HTTP client configuration.
     */
    private final HttpClientConfig httpClientConfig;

    public HttpClient(HttpClientConfig httpClientConfig) throws TboxClientConfigException {
        if (httpClientConfig == null) {
            throw new TboxClientConfigException("httpClientConfig is null");
        }
        this.httpClientConfig = httpClientConfig;
    }

    /**
     * Makes a POST request to the specified path.
     *
     * @param path    API path
     * @param data    Request data
     * @param headers Additional headers (optional)
     * @return Response as string
     * @throws TboxClientConfigException if there's a configuration error
     * @throws TboxHttpResponseException if there's an HTTP response error
     */
    public String post(String path, Map<String, Object> data, Map<String, String> headers)
            throws TboxClientConfigException, TboxHttpResponseException {
        String url = buildUrl(path);

        try {
            String json = JSON.toJSONString(data);
            RequestBody body = RequestBody.create(json, JSON_MEDIA_TYPE);
            Request request = new Request.Builder()
                    .url(url)
                    .headers(Headers.of(generateHeaders(headers)))
                    .post(body)
                    .build();

            try (Response response = okHttpClient.newCall(request).execute()) {
                if (!response.isSuccessful()) {
                    String errorText = response.body() != null ? response.body().string() : "Unknown error";
                    logger.error("HTTP request failed, status_code: {}, text: {}", response.code(), errorText);
                    throw new TboxHttpResponseException(
                            response.code(),
                            String.format("HTTP request failed, status_code: %d, text: %s", response.code(), errorText)
                    );
                }

                String responseText = response.body() != null ? response.body().string() : "";
                logger.info("HTTP request success, status_code: {}, text: {}", response.code(), responseText);
                return responseText;
            }
        } catch (JSONException e) {
            throw new TboxClientConfigException("Failed to serialize request data", e);
        } catch (IOException e) {
            throw new TboxHttpResponseException("Failed to execute request", e);
        }
    }

    /**
     * Makes a streaming POST request to the specified path.
     *
     * @param path    API path
     * @param data    Request data
     * @param headers Additional headers (optional)
     * @param timeout Timeout in seconds
     * @return Iterable of HttpResponseEvent
     * @throws TboxClientConfigException if there's a configuration error
     */
    public Iterable<HttpResponseEvent> postStream(String path, Map<String, Object> data, Map<String, String> headers, int timeout)
            throws TboxClientConfigException {
        String url = buildUrl(path);

        try {
            String json = JSON.toJSONString(data);
            RequestBody body = RequestBody.create(json, JSON_MEDIA_TYPE);

            Request request = new Request.Builder()
                    .url(url)
                    .headers(Headers.of(generateHeaders(headers)))
                    .post(body)
                    .build();

            // Create a blocking queue to receive SSE events
            BlockingQueue<HttpResponseEvent> eventQueue = new LinkedBlockingQueue<>();

            // Set up event source
            EventSourceListener listener = new EventSourceListener() {
                @Override
                public void onEvent(EventSource eventSource, String id, String type, String data) {
                    // logger.info("Server response event id {}, event {}, data {}", id, type, data);
                    eventQueue.add(new HttpResponseEvent(type, data, id));
                }

                @Override
                public void onFailure(EventSource eventSource, Throwable t, Response response) {
                    String errorMessage = "Request TBox failed with HTTP code";
                    if (response != null) {
                        try {
                            errorMessage += ": " + response.code() + ", error message: " + Objects.requireNonNull(response.body()).string();
                        } catch (IOException e) {
                            logger.error("Failed to get response body", e);
                        }
                    }
                    logger.error(errorMessage, t);

                    Map<String, Object> errorContext = new HashMap<>();
                    if (t != null) {
                        errorContext.put("errorMsg", t.getMessage());
                    } else {
                        errorContext.put("errorMsg", errorMessage);
                    }

                    try {
                        eventQueue.add(new HttpResponseEvent("error", JSON.toJSONString(errorContext)));
                    } catch (JSONException e) {
                        eventQueue.add(new HttpResponseEvent("error", "{\"errorMsg\":\"Failed to serialize error context\"}"));
                    }
                    eventSource.cancel();
                }

                @Override
                public void onClosed(EventSource eventSource) {
                    eventQueue.add(new HttpResponseEvent("close", "Connection closed"));
                }
            };

            // Create a custom client with the specified timeout
            OkHttpClient customClient = okHttpClient.newBuilder()
                    .readTimeout(timeout, TimeUnit.SECONDS)
                    .build();

            // Build the event source
            EventSource.Factory factory = EventSources.createFactory(customClient);
            EventSource eventSource = factory.newEventSource(request, listener);

            // Return an iterable that allows consuming the events
            return () -> new java.util.Iterator<HttpResponseEvent>() {
                private HttpResponseEvent nextEvent;
                private boolean closed = false;

                @Override
                public boolean hasNext() {
                    if (closed) {
                        return false;
                    }

                    if (nextEvent != null) {
                        return true;
                    }

                    try {
                        nextEvent = eventQueue.poll(timeout, TimeUnit.SECONDS);
                        if (nextEvent == null || "close".equals(nextEvent.getEvent())) {
                            closed = true;
                            eventSource.cancel();
                            return false;
                        }
                        return true;
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        closed = true;
                        eventSource.cancel();
                        return false;
                    }
                }

                @Override
                public HttpResponseEvent next() {
                    if (!hasNext()) {
                        throw new java.util.NoSuchElementException();
                    }
                    HttpResponseEvent event = nextEvent;
                    nextEvent = null;
                    return event;
                }
            };
        } catch (JSONException e) {
            throw new TboxClientConfigException("Failed to serialize request data", e);
        }
    }

    /**
     * Makes a GET request to the specified URL.
     *
     * @param url     API URL
     * @param query   Query parameters
     * @param headers Additional headers (optional)
     * @return Response
     * @throws TboxHttpResponseException if there's an HTTP response error
     */
    public Response get(String url, Map<String, String> query, Map<String, String> headers)
            throws TboxHttpResponseException {
        HttpUrl.Builder httpBuilder = HttpUrl.parse(url).newBuilder();
        if (query != null) {
            for (Map.Entry<String, String> entry : query.entrySet()) {
                httpBuilder.addQueryParameter(entry.getKey(), entry.getValue());
            }
        }

        Request request = new Request.Builder()
                .url(httpBuilder.build())
                .headers(Headers.of(generateHeaders(headers)))
                .get()
                .build();

        try {
            return okHttpClient.newCall(request).execute();
        } catch (IOException e) {
            throw new TboxHttpResponseException("Failed to execute request", e);
        }
    }

    /**
     * Builds the complete URL from the path.
     *
     * @param path API path
     * @return Complete URL
     */
    private String buildUrl(String path) {
        if (!path.startsWith("/")) {
            path = "/" + path;
        }
        return String.format("%s://%s%s", httpClientConfig.getSchema(), httpClientConfig.getHost(), path);
    }

    /**
     * Generates HTTP headers.
     *
     * @param headers Additional headers (optional)
     * @return Complete headers map
     */
    private Map<String, String> generateHeaders(Map<String, String> headers) {
        Map<String, String> result = new HashMap<>();
        if (headers != null) {
            result.putAll(headers);
        }

        if (httpClientConfig.getAuthorization() != null) {
            result.put("Authorization", httpClientConfig.getAuthorization());
        }

        if (result.get("Content-Type") == null) {
            result.put("Content-Type", "application/json");
        }

        result.put("source", "AGENT_SDK");

        return result;
    }
} 