package cn.tbox.sdk;

import java.util.HashMap;
import java.util.Map;

import cn.tbox.sdk.core.exception.TboxClientConfigException;
import cn.tbox.sdk.core.exception.TboxServerException;
import cn.tbox.sdk.core.http.HttpClient;
import cn.tbox.sdk.core.http.HttpClientConfig;
import cn.tbox.sdk.core.http.HttpResponseEvent;
import cn.tbox.sdk.model.message.MessageParser;

/**
 * Tbox client for making API requests.
 */
public class TboxClient {
    /**
     * Tbox client configuration.
     */
    private final HttpClientConfig httpClientConfig;

    /**
     * HTTP client.
     */
    private final HttpClient httpClient;

    /**
     * Creates a new Tbox client.
     *
     * @param httpClientConfig HTTP client configuration
     * @throws TboxClientConfigException if there's a configuration error
     */
    public TboxClient(HttpClientConfig httpClientConfig) throws TboxClientConfigException {
        if (httpClientConfig == null) {
            throw new TboxClientConfigException("httpClientConfig is null");
        }
        this.httpClientConfig = httpClientConfig;
        this.httpClient = createHttpClient(this.httpClientConfig);
    }

    /**
     * Creates a new Tbox client with authorization.
     *
     * @param authorization Authorization token
     * @throws TboxClientConfigException if there's a configuration error
     */
    public TboxClient(String authorization) throws TboxClientConfigException {
        this.httpClientConfig = new HttpClientConfig();
        if (authorization != null) {
            this.httpClientConfig.setAuthorization(authorization);
        }
        this.httpClient = createHttpClient(this.httpClientConfig);
    }

    /**
     * Creates the HTTP client instance.
     * This method is protected to allow mocking in tests.
     *
     * @param config HTTP client configuration
     * @return HTTP client instance
     * @throws TboxClientConfigException if there's a configuration error
     */
    protected HttpClient createHttpClient(HttpClientConfig config) throws TboxClientConfigException {
        return new HttpClient(config);
    }

    /**
     * Makes a chat request to the Tbox API.
     * Used to call Tbox chat-type applications.
     * Returns a unified streaming response format.
     *
     * @param appId            Application ID
     * @param query            Query text
     * @param userId           User ID
     * @param conversationId   Conversation ID (optional)
     * @param requestId        Request ID (optional)
     * @param inputs           Input parameters (optional)
     * @param withMeta         Whether to include meta information (optional)
     * @param messageParser    Message parser (optional)
     * @param clientProperties Client properties (optional)
     * @return Iterable of parsed responses
     * @throws TboxClientConfigException if there's a configuration error
     */
    public Iterable<Map<String, Object>> chat(
            String appId,
            String query,
            String userId,
            String conversationId,
            String requestId,
            Map<String, Object> inputs,
            boolean withMeta,
            MessageParser messageParser,
            Map<String, Object> clientProperties
    ) throws TboxClientConfigException {
        Map<String, Object> data = new HashMap<>();
        data.put("appId", appId);
        data.put("query", query);

        if (conversationId != null) {
            data.put("conversationId", conversationId);
        }
        if (requestId != null) {
            data.put("requestId", requestId);
        }
        if (inputs != null) {
            data.put("inputs", inputs);
        }
        if (userId != null) {
            data.put("userId", userId);
        }
        if (clientProperties != null) {
            data.put("clientProperties", clientProperties);
        }

        Iterable<HttpResponseEvent> responseIter = httpClient.postStream("/api/chat", data, null, 110);
        return stream(responseIter, messageParser, withMeta);
    }

    /**
     * Makes a chat request to the Tbox API with default parameters.
     *
     * @param appId  Application ID
     * @param query  Query text
     * @param userId User ID
     * @return Iterable of parsed responses
     * @throws TboxClientConfigException if there's a configuration error
     */
    public Iterable<Map<String, Object>> chat(String appId, String query, String userId) throws TboxClientConfigException {
        return chat(appId, query, userId, null, null, null, false, null, null);
    }

    /**
     * Makes a synchronous chat request to the Tbox API.
     * Wraps the chat API into a synchronous interface.
     *
     * @param appId            Application ID
     * @param query            Query text
     * @param userId           User ID
     * @param conversationId   Conversation ID (optional)
     * @param requestId        Request ID (optional)
     * @param inputs           Input parameters (optional)
     * @param clientProperties Client properties (optional)
     * @return Map of response data
     * @throws TboxClientConfigException if there's a configuration error
     */
    public Map<String, Map<String, Object>> chatSync(
            String appId,
            String query,
            String userId,
            String conversationId,
            String requestId,
            Map<String, Object> inputs,
            Map<String, Object> clientProperties
    ) throws TboxClientConfigException {
        MessageParser parser = new MessageParser();

        Iterable<Map<String, Object>> events = chat(
                appId, query, userId, conversationId, requestId, inputs, false, parser, clientProperties
        );

        // Process all events (we need to iterate to consume the stream)
        for (Map<String, Object> event : events) {
            // Do nothing, just consume the events
        }

        return parser.getAnswersHolder();
    }

    /**
     * Makes a synchronous chat request to the Tbox API with default parameters.
     *
     * @param appId  Application ID
     * @param query  Query text
     * @param userId User ID
     * @return Map of response data
     * @throws TboxClientConfigException if there's a configuration error
     */
    public Map<String, Map<String, Object>> chatSync(String appId, String query, String userId) throws TboxClientConfigException {
        return chatSync(appId, query, userId, null, null, null, null);
    }

    /**
     * Makes a completion request to the Tbox API.
     *
     * @param appId            Application ID
     * @param userId           User ID
     * @param inputs           Input parameters (optional)
     * @param conversationId   Conversation ID (optional)
     * @param requestId        Request ID (optional)
     * @param withMeta         Whether to include meta information (optional)
     * @param messageParser    Message parser (optional)
     * @param clientProperties Client properties (optional)
     * @return Iterable of parsed responses
     * @throws TboxClientConfigException if there's a configuration error
     */
    public Iterable<Map<String, Object>> completion(
            String appId,
            Map<String, Object> inputs,
            String userId,
            String conversationId,
            String requestId,
            boolean withMeta,
            MessageParser messageParser,
            Map<String, Object> clientProperties
    ) throws TboxClientConfigException {
        Map<String, Object> data = new HashMap<>();
        data.put("appId", appId);

        if (conversationId != null) {
            data.put("conversationId", conversationId);
        }
        if (requestId != null) {
            data.put("requestId", requestId);
        }
        if (inputs != null) {
            data.put("inputs", inputs);
        }
        if (userId != null) {
            data.put("userId", userId);
        }
        if (clientProperties != null) {
            data.put("clientProperties", clientProperties);
        }

        Iterable<HttpResponseEvent> responseIter = httpClient.postStream("/api/completion", data, null, 110);
        return stream(responseIter, messageParser, withMeta);
    }

    /**
     * Makes a completion request to the Tbox API with default parameters.
     *
     * @param appId  Application ID
     * @param inputs Input parameters
     * @param userId User ID
     * @return Iterable of parsed responses
     * @throws TboxClientConfigException if there's a configuration error
     */
    public Iterable<Map<String, Object>> completion(String appId, Map<String, Object> inputs, String userId)
            throws TboxClientConfigException {
        return completion(appId, inputs, userId, null, null, false, null, null);
    }

    /**
     * Makes a synchronous completion request to the Tbox API.
     * Wraps the completion API into a synchronous interface.
     *
     * @param appId            Application ID
     * @param inputs           Input parameters
     * @param userId           User ID
     * @param conversationId   Conversation ID (optional)
     * @param requestId        Request ID (optional)
     * @param clientProperties Client properties (optional)
     * @return Map of response data
     * @throws TboxClientConfigException if there's a configuration error
     */
    public Map<String, Map<String, Object>> completionSync(
            String appId,
            Map<String, Object> inputs,
            String userId,
            String conversationId,
            String requestId,
            Map<String, Object> clientProperties
    ) throws TboxClientConfigException {
        MessageParser parser = new MessageParser();

        Iterable<Map<String, Object>> events = completion(
                appId, inputs, userId, conversationId, requestId, false, parser, clientProperties
        );

        // Process all events (we need to iterate to consume the stream)
        for (Map<String, Object> event : events) {
            // Do nothing, just consume the events
        }

        return parser.getAnswersHolder();
    }

    /**
     * Makes a synchronous completion request to the Tbox API with default parameters.
     *
     * @param appId  Application ID
     * @param inputs Input parameters
     * @param userId User ID
     * @return Map of response data
     * @throws TboxClientConfigException if there's a configuration error
     */
    public Map<String, Map<String, Object>> completionSync(String appId, Map<String, Object> inputs, String userId)
            throws TboxClientConfigException {
        return completionSync(appId, inputs, userId, null, null, null);
    }

    /**
     * Processes a stream of response events.
     *
     * @param responseIter  Response event iterator
     * @param messageParser Message parser (optional)
     * @param withMeta      Whether to include meta information
     * @return Iterable of parsed responses
     */
    private Iterable<Map<String, Object>> stream(Iterable<HttpResponseEvent> responseIter, MessageParser messageParser, boolean withMeta) {
        final MessageParser parser = messageParser != null ? messageParser : new MessageParser();

        return () -> new java.util.Iterator<Map<String, Object>>() {
            private final java.util.Iterator<HttpResponseEvent> eventIterator = responseIter.iterator();
            private Map<String, Object> nextData = null;
            private boolean hasNextCalled = false;

            @Override
            public boolean hasNext() {
                if (hasNextCalled) {
                    return nextData != null;
                }

                while (eventIterator.hasNext()) {
                    HttpResponseEvent event = eventIterator.next();

                    // Parse response, if it's an error event an exception will be thrown
                    if (parser.needParse(event)) {
                        try {
                            Map<String, Object> data = parser.parse(event);
                            String type = (String) data.get("type");

                            if ("meta".equals(type)) {
                                if (withMeta) {
                                    nextData = data;
                                    hasNextCalled = true;
                                    return true;
                                }
                            } else {
                                nextData = data;
                                hasNextCalled = true;
                                return true;
                            }
                        } catch (TboxServerException e) {
                            throw new RuntimeException(e.getMessage(), e);
                        }
                    }
                }

                hasNextCalled = true;
                nextData = null;
                return false;
            }

            @Override
            public Map<String, Object> next() {
                if (!hasNextCalled) {
                    hasNext();
                }

                if (nextData == null) {
                    throw new java.util.NoSuchElementException();
                }

                Map<String, Object> result = nextData;
                nextData = null;
                hasNextCalled = false;
                return result;
            }
        };
    }
} 