/*
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.feiliu.protogen.protobuf.compiler;

import org.apache.maven.repository.internal.MavenRepositorySystemUtils;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.connector.basic.BasicRepositoryConnectorFactory;
import org.eclipse.aether.impl.DefaultServiceLocator;
import org.eclipse.aether.repository.LocalRepository;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.ArtifactRequest;
import org.eclipse.aether.resolution.ArtifactResult;
import org.eclipse.aether.spi.connector.RepositoryConnectorFactory;
import org.eclipse.aether.spi.connector.transport.TransporterFactory;
import org.eclipse.aether.transport.file.FileTransporterFactory;
import org.eclipse.aether.transport.http.HttpTransporterFactory;

import java.io.File;
import java.util.Arrays;

public class MavenProtocResolver {
    private static final RepositorySystem  repositorySystem;
    private static RepositorySystemSession session;
    private static final RemoteRepository  mavenCentral;

    static {
        // 初始化 RepositorySystem
        DefaultServiceLocator locator = MavenRepositorySystemUtils.newServiceLocator();
        locator.addService(RepositoryConnectorFactory.class, BasicRepositoryConnectorFactory.class);
        locator.addService(TransporterFactory.class, FileTransporterFactory.class);
        locator.addService(TransporterFactory.class, HttpTransporterFactory.class);
        repositorySystem = locator.getService(RepositorySystem.class);

        // 创建会话
        DefaultRepositorySystemSession session = MavenRepositorySystemUtils.newSession();
        LocalRepository localRepo = new LocalRepository(System.getProperty("user.home") + "/.m2/repository");
        session.setLocalRepositoryManager(repositorySystem.newLocalRepositoryManager(session, localRepo));
        MavenProtocResolver.session = session;

        // 设置 Maven 中央仓库
        mavenCentral = new RemoteRepository.Builder("central", "default", "https://repo.maven.apache.org/maven2/")
            .build();
    }

    public static String resolveProtoc(String version) {
        try {
            String classifier = getProtocClassifier();
            ArtifactResult result = resolveArtifact("com.google.protobuf", "protoc", version, classifier, "exe");
            File file = result.getArtifact().getFile();
            file.setExecutable(true);
            return file.getAbsolutePath();
        } catch (Exception e) {
            throw new RuntimeException("Failed to resolve protoc from Maven repository", e);
        }
    }

    public static String resolveGrpcJavaPlugin(String version) {
        try {
            String classifier = getProtocClassifier();
            ArtifactResult result = resolveArtifact("io.grpc", "protoc-gen-grpc-java", version, classifier, "exe");
            File file = result.getArtifact().getFile();
            file.setExecutable(true);
            return file.getAbsolutePath();
        } catch (Exception e) {
            throw new RuntimeException("Failed to resolve grpc-java plugin from Maven repository", e);
        }
    }

    private static ArtifactResult resolveArtifact(String groupId, String artifactId, String version, String classifier,
                                                  String extension) throws Exception {
        DefaultArtifact artifact = new DefaultArtifact(groupId, artifactId, classifier, extension, version);
        ArtifactRequest request = new ArtifactRequest();
        request.setArtifact(artifact);
        request.setRepositories(Arrays.asList(mavenCentral));
        return repositorySystem.resolveArtifact(session, request);
    }

    private static String getProtocClassifier() {
        String os = System.getProperty("os.name").toLowerCase();
        String arch = System.getProperty("os.arch").toLowerCase();

        if (os.contains("windows")) {
            return "windows-x86_64";
        } else if (os.contains("linux")) {
            return "linux-x86_64";
        } else if (os.contains("mac")) {
            if (arch.contains("aarch64") || arch.contains("arm64")) {
                return "osx-aarch_64";
            } else {
                return "osx-x86_64";
            }
        }
        throw new RuntimeException("Unsupported operating system: " + os);
    }

    public static String resolveGoogleProtos() {
        try {
            ArtifactResult result = resolveArtifact("com.google.protobuf", "protobuf-java", "3.19.2", null, "jar");

            // 返回包含 proto 文件的目录路径
            File jarFile = result.getArtifact().getFile();
            String protoPath = extractProtoFiles(jarFile);
            return protoPath;
        } catch (Exception e) {
            throw new RuntimeException("Failed to resolve Google Protobuf standard types", e);
        }
    }

    private static String extractProtoFiles(File jarFile) throws Exception {
        String tempDir = System.getProperty("java.io.tmpdir") + "/google-protos-" + System.currentTimeMillis();
        File outputDir = new File(tempDir);
        outputDir.mkdirs();

        // 解压 jar 文件中的 proto 文件
        try (java.util.jar.JarFile jar = new java.util.jar.JarFile(jarFile)) {
            java.util.Enumeration<java.util.jar.JarEntry> entries = jar.entries();
            while (entries.hasMoreElements()) {
                java.util.jar.JarEntry entry = entries.nextElement();
                if (entry.getName().endsWith(".proto")) {
                    File outFile = new File(outputDir, entry.getName());
                    outFile.getParentFile().mkdirs();
                    try (java.io.InputStream is = jar.getInputStream(entry);
                            java.io.FileOutputStream fos = new java.io.FileOutputStream(outFile)) {
                        byte[] buffer = new byte[1024];
                        int count;
                        while ((count = is.read(buffer)) != -1) {
                            fos.write(buffer, 0, count);
                        }
                    }
                }
            }
        }
        return outputDir.getAbsolutePath();
    }
}