package cn.takujo.takujoframework.mybatis.sqlhelper.find;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.takujo.takujoframework.mybatis.sqlhelper.base.Condition;
import cn.takujo.takujoframework.mybatis.sqlhelper.base.Form;
import cn.takujo.takujoframework.mybatis.sqlhelper.base.Rule;
import lombok.Getter;

/**
 * 搜索规则，需要继承或匿名实现
 * 
 * @author wzx
 *
 * @param <T>
 *            Form的实现类
 */
public abstract class FindRule<T extends Form> extends Condition implements Rule {

	public static final String DESC = " DESC ";
	public static final String ASC = " ASC ";

	@Getter
	private String table;
	@Getter
	private String sort;
	@Getter
	private String sortBy;

	private T form;
	private Class<? extends Form> cla;

	private List<FindField> fields;

	/**
	 * @param table
	 *            表名
	 * @param form
	 *            Form实现类
	 */
	public FindRule(String table, T form) {
		this.table = table;
		this.form = form;
		this.cla = form.getClass();
	}

	/**
	 * @param table
	 *            表名
	 * @param sort
	 *            要实现排序的列名
	 * @param sortBy
	 *            排序方式: 降序 FindRule.DESC,升序 FindRule.ASC
	 * @param form
	 *            Form实现类
	 */
	public FindRule(String table, String sort, String sortBy, T form) {
		this.table = table;
		this.sort = sort;
		this.sortBy = sortBy;
		this.form = form;
		this.cla = form.getClass();
	}

	/**
	 * 重要！ 用来生成查询条件,重写时return的最外层一定是and()或or()
	 * 
	 * @return 条件sql
	 */
	@Override
	public String generate() {
		return " ";
	}

	/**
	 * 基础查询
	 * 
	 * @param fields
	 *            要查询的字段的集合
	 * @return 基础sql
	 */
	public String base(List<FindField> fields) {
		this.fields = fields;
		String filed = null;
		if (fields == null) {
			filed = "* ";
		} else {
			int length = fields.size();
			if (length == 0) {
				filed = "* ";
			} else if (length == 1) {
				filed = fields.get(0).result() + " ";
			} else {
				StringBuffer sb = new StringBuffer();
				for (int i = 0; i < length; i++) {
					String s = fields.get(i).result();
					if (i != length - 1) {
						sb.append(s + " , ");
					} else {
						sb.append(s + " ");
					}
				}
				filed = sb.toString();
			}
		}
		return "SELECT " + filed + "FROM " + table + " ";
	}

	@Override
	public String commit() {
		String base = base(fields);
		String generate = generate();
		if (!generate.trim().isEmpty()) {
			generate = "WHERE " + generate;
		}
		if (sort != null && !sort.trim().isEmpty()) {
			return base + generate + "ORDER BY " + sort + sortBy;
		}
		return base + generate;
	}

	/**
	 * and
	 * 
	 * @param con
	 *            条件必须是以下方法的返回值，且任意数量个（可以为零）：
	 *            like、greaterThan、greaterOrEqual、equal、lessThan、lessOrEqual、nesting
	 * @return 多条件
	 */
	public String and(String... con) {
		int length = con.length;
		if (length == 0) {
			return " ";
		} else if (length == 1) {
			String name = con[0];
			Object o = fieldValueIsExist(name);
			if (o != null) {
				return name + " ";
			} else {
				return " ";
			}
		} else {
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < length; i++) {
				String s = con[i];
				Object o = fieldValueIsExist(s);
				if (o != null) {
					sb.append(s + " AND ");
				}
			}
			int lastIndexOf = sb.lastIndexOf("AND");
			if (lastIndexOf != -1) {
				sb.delete(lastIndexOf, lastIndexOf + 3);
			}
			return sb.toString();
		}
	}

	/**
	 * or
	 * 
	 * @param con
	 *            条件必须是以下方法的返回值，且任意数量个（可以为零）：
	 *            like、greaterThan、greaterOrEqual、equal、lessThan、lessOrEqual、nesting
	 * @return 多条件
	 */
	public String or(String... con) {
		int length = con.length;
		if (length == 0) {
			return " ";
		} else if (length == 1) {
			String name = con[0];
			Object o = fieldValueIsExist(name);
			if (o != null) {
				return name + " ";
			} else {
				return " ";
			}
		} else {
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < length; i++) {
				String s = con[i];
				Object o = fieldValueIsExist(s);
				if (o != null) {
					sb.append(s + " OR ");
				}
			}
			int lastIndexOf = sb.lastIndexOf("OR");
			if (lastIndexOf != -1) {
				sb.delete(lastIndexOf, lastIndexOf + 3);
			}
			return sb.toString();
		}
	}

	/**
	 * and和or混用
	 * 
	 * @param con
	 *            条件必须是以下方法的返回值，且任意数量个（可以为零）：
	 *            and、or（必须用attachAndSign或attachOrSign方法连接and或or）
	 * @return 多条件
	 */
	public String andor(String... con) {
		int length = con.length;
		if (length == 0) {
			return " ";
		} else if (length == 1) {
			String name = con[0];
			String trim = name.trim();
			if (trim.equals("AND") || trim.equals("OR")) {
				return " ";
			} else {
				return name + " ";
			}
		} else {
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < length; i++) {
				String s = con[i];
				if (i != length - 1) {
					sb.append(s);
				} else {
					String trim = s.trim();
					if (trim.equals("AND") || trim.equals("OR")) {
						continue;
					} else {
						sb.append(s);
					}
				}
			}
			return sb.toString();
		}
	}

	@Override
	public Map<String, Object> toMap() {
		Map<String, Object> map = new HashMap<>();
		map.put("myRule", this);
		Field[] declaredFields = cla.getDeclaredFields();
		for (Field field : declaredFields) {
			String name = field.getName();
			int modifiers = field.getModifiers();
			if (!isSerializableField(modifiers)) {
				Object o = fieldGetMethodInvoke(name);
				if (o != null) {
					map.put(name, o);
				}
			}
		}
		return map;
	}

	/**
	 * 判断是否为序列化字段
	 * 
	 * @param modifiers
	 * @return true：是 false：否
	 */
	private boolean isSerializableField(int modifiers) {
		if (Modifier.isStatic(modifiers) && Modifier.isFinal(modifiers)) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 调用某个字段的get方法
	 * 
	 * @param field
	 *            字段名
	 * @return 调用结果为空则返回null
	 */
	private Object fieldGetMethodInvoke(String field) {
		try {
			Method method = cla.getMethod("get" + field.substring(0, 1).toUpperCase() + field.substring(1));
			return method.invoke(form);
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		} catch (SecurityException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * form中字段值是否存在
	 * 
	 * @return 如果有则返回值，没有则返回null
	 */
	private Object fieldValueIsExist(String field) {
		int indexOf = field.indexOf("{");
		int lastIndexOf = field.lastIndexOf("}");
		return fieldGetMethodInvoke(field.substring(indexOf + 1, lastIndexOf));
	}

}
