package cn.takujo.takujoframework.mybatis.sqlhelper.update;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.takujo.takujoframework.mybatis.sqlhelper.base.Condition;
import cn.takujo.takujoframework.mybatis.sqlhelper.base.Form;
import cn.takujo.takujoframework.mybatis.sqlhelper.base.Rule;
import lombok.Getter;

/**
 * 更新规则，需要继承或匿名实现
 * 
 * @author wzx
 *
 * @param <T>
 *            Form的实现类
 */
public abstract class UpdateRule<T extends Form> extends Condition implements Rule {

	@Getter
	private String table;

	private T form;

	private Class<? extends Form> cla;

	/**
	 * @param table
	 *            表名
	 * @param form
	 *            Form实现类
	 */
	public UpdateRule(String table, T form) {
		this.table = table;
		this.form = form;
		this.cla = form.getClass();
	}

	/**
	 * 重要！ 用来生成要更新的值,重写时return的最外层一定是values方法 默认是更新全部
	 * 
	 * @return 要set的字段
	 */
	public String set() {
		return values();
	}

	/**
	 * values
	 * 
	 * @param value
	 *            条件必须是以下方法的返回值，且任意数量个（可以为零）： value
	 * @return 多个值
	 */
	public String values(String... value) {
		int length = value.length;
		if (length == 0) {
			return lengthEqualZero();
		} else if (length == 1) {
			String name = value[0];
			if (isId(name)) {
				return " ";
			}
			Object o = fieldValueIsExist(name);
			if (o != null) {
				return name + " ";
			} else {
				return " ";
			}
		} else {
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < length; i++) {
				String s = value[i];
				if (isId(s)) {
					continue;
				}
				Object o = fieldValueIsExist(s);
				if (o != null) {
					sb.append(s + " , ");
				}
			}
			int lastIndexOf = sb.lastIndexOf(",");
			if (lastIndexOf != -1) {
				sb.deleteCharAt(lastIndexOf);
			}
			return sb.toString();
		}
	}

	/**
	 * set一个值
	 * 
	 * @param column
	 *            数据库列名
	 * @param property
	 *            映射java对象属性名
	 * @return 一个值
	 */
	public static String value(String column, String property) {
		return column + "=" + "#{" + property + "}";
	}

	private String lengthEqualZero() {
		Class<? extends Form> cla = form.getClass();
		List<String> p = new ArrayList<>();
		Field[] declaredFields = cla.getDeclaredFields();
		for (Field field : declaredFields) {
			String name = field.getName();
			int modifiers = field.getModifiers();
			if (!isSerializableField(modifiers) && !name.equals("id")) {
				Object o = fieldGetMethodInvoke(name);
				if (o != null) {
					String value = "=#{" + name + "}";
					char[] array = name.toCharArray();
					StringBuffer sb = new StringBuffer();
					for (char c : array) {
						if (Character.isUpperCase(c)) {
							sb.append('_');
							sb.append(Character.toLowerCase(c));
						} else {
							sb.append(c);
						}
					}
					p.add(sb.toString() + value);
				}
			}
		}
		int size = p.size();
		if (size != 0) {
			String[] ss = new String[size];
			p.toArray(ss);
			return values(ss);
		} else {
			return " ";
		}
	}

	/**
	 * @return 基础sql
	 */
	public String base() {
		return "UPDATE " + table + " SET ";
	}

	/**
	 * 重要！ 用来生成更新条件,重写时return的最外层一定是and()或or() 默认是按id作为更新条件
	 * 
	 * @return 条件sql
	 */
	@Override
	public String generate() {
		return and(equal("id", "id"));
	}

	@Override
	public String commit() {
		String generate = generate();
		if (!generate.trim().isEmpty()) {
			generate = "WHERE " + generate;
		}
		return base() + set() + generate;
	}

	/**
	 * and
	 * 
	 * @param con
	 *            条件必须是以下方法的返回值，且任意数量个（可以为零）：
	 *            like、greaterThan、greaterOrEqual、equal、lessThan、lessOrEqual、nesting
	 * @return 多条件
	 */
	public String and(String... con) {
		int length = con.length;
		if (length == 0) {
			return " ";
		} else if (length == 1) {
			String name = con[0];
			Object o = fieldValueIsExist(name);
			if (o != null) {
				return name + " ";
			} else {
				return " ";
			}
		} else {
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < length; i++) {
				String s = con[i];
				Object o = fieldValueIsExist(s);
				if (o != null) {
					sb.append(s + " AND ");
				}
			}
			int lastIndexOf = sb.lastIndexOf("AND");
			if (lastIndexOf != -1) {
				sb.delete(lastIndexOf, lastIndexOf + 3);
			}
			return sb.toString();
		}
	}

	/**
	 * or
	 * 
	 * @param con
	 *            条件必须是以下方法的返回值，且任意数量个（可以为零）：
	 *            like、greaterThan、greaterOrEqual、equal、lessThan、lessOrEqual、nesting
	 * @return 多条件
	 */
	public String or(String... con) {
		int length = con.length;
		if (length == 0) {
			return " ";
		} else if (length == 1) {
			String name = con[0];
			Object o = fieldValueIsExist(name);
			if (o != null) {
				return name + " ";
			} else {
				return " ";
			}
		} else {
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < length; i++) {
				String s = con[i];
				Object o = fieldValueIsExist(s);
				if (o != null) {
					sb.append(s + " OR ");
				}
			}
			int lastIndexOf = sb.lastIndexOf("OR");
			if (lastIndexOf != -1) {
				sb.delete(lastIndexOf, lastIndexOf + 3);
			}
			return sb.toString();
		}
	}

	/**
	 * and和or混用
	 * 
	 * @param con
	 *            条件必须是以下方法的返回值，且任意数量个（可以为零）：
	 *            and、or（必须用attachAndSign或attachOrSign方法连接and或or）
	 * @return 多条件
	 */
	public String andor(String... con) {
		int length = con.length;
		if (length == 0) {
			return " ";
		} else if (length == 1) {
			String name = con[0];
			String trim = name.trim();
			if (trim.equals("AND") || trim.equals("OR")) {
				return " ";
			} else {
				return name + " ";
			}
		} else {
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < length; i++) {
				String s = con[i];
				if (i != length - 1) {
					sb.append(s);
				} else {
					String trim = s.trim();
					if (trim.equals("AND") || trim.equals("OR")) {
						continue;
					} else {
						sb.append(s);
					}
				}
			}
			return sb.toString();
		}
	}

	@Override
	public Map<String, Object> toMap() {
		Class<? extends Form> cla = form.getClass();
		Map<String, Object> map = new HashMap<>();
		map.put("myRule", this);
		Field[] declaredFields = cla.getDeclaredFields();
		for (Field field : declaredFields) {
			String name = field.getName();
			int modifiers = field.getModifiers();
			if (!isSerializableField(modifiers)) {
				Object o = fieldGetMethodInvoke(name);
				if (o != null) {
					map.put(name, o);
				}
			}
		}
		return map;
	}

	/**
	 * 判断是否为序列化字段
	 * 
	 * @param modifiers
	 * @return true：是 false：否
	 */
	private boolean isSerializableField(int modifiers) {
		if (Modifier.isStatic(modifiers) && Modifier.isFinal(modifiers)) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 调用某个字段的get方法
	 * 
	 * @param field
	 *            字段名
	 * @return 调用结果为空则返回null
	 */
	private Object fieldGetMethodInvoke(String field) {
		try {
			Method method = cla.getMethod("get" + field.substring(0, 1).toUpperCase() + field.substring(1));
			return method.invoke(form);
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		} catch (SecurityException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * form中字段值是否存在
	 * 
	 * @return 如果有则返回值，没有则返回null
	 */
	private Object fieldValueIsExist(String field) {
		int indexOf = field.indexOf("{");
		int lastIndexOf = field.lastIndexOf("}");
		return fieldGetMethodInvoke(field.substring(indexOf + 1, lastIndexOf));
	}

	/**
	 * 判断value是否为id
	 * 
	 * @param value
	 * @return 默认为false
	 */
	private boolean isId(String value) {
		int indexOf = value.indexOf("=");
		int lastIndexOf = value.lastIndexOf("}");
		String column = value.substring(0, indexOf);
		String property = value.substring(indexOf + 3, lastIndexOf);
		if (column.equals("id") || property.equals("id")) {
			return true;
		}
		return false;
	}

}
