package cn.takujo.common_api.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.DeleteObjectsRequest;
import com.aliyun.oss.model.DeleteObjectsResult;
import com.aliyun.oss.model.ListObjectsRequest;
import com.aliyun.oss.model.OSSObjectSummary;
import com.aliyun.oss.model.ObjectListing;

import cn.takujo.common_api.exception.OssException;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * 阿里云OSS存储
 * 
 * @author wzx
 *
 */

@Slf4j
@AllArgsConstructor
@Data
public final class AliyunOssUtil {

	private String endpoint;
	private String accessKeyId;
	private String accessKeySecret;
	private String bucketName;

	/**
	 * 上传单个文件到OSS
	 * 
	 * @param objectName
	 *            存储后的文件名（要包含路径信息）
	 * @param input
	 *            文件输入流
	 * @throws OssException
	 *             oss上传服务异常
	 */
	public void uploadFile(String objectName, InputStream input) throws OssException {
		OSSClient ossClient = new OSSClient(endpoint, accessKeyId, accessKeySecret);
		try {
			if (ossClient.doesBucketExist(bucketName)) {
				ossClient.putObject(bucketName, objectName, input);
			}
		} catch (OSSException e) {
			e.printStackTrace();
			throw new OssException("oss01", "上传到OSS->" + "失败");
		} catch (ClientException e) {
			e.printStackTrace();
			throw new OssException("oss01", "上传到OSS->" + "失败");
		} finally {
			ossClient.shutdown();
			if (input != null) {
				try {
					input.close();
				} catch (IOException e) {
					e.printStackTrace();
					log.warn("aliyun oss upload: error -> stream close failure");
				}
			}
		}
	}

	/**
	 * 上传单个文件到OSS
	 * 
	 * @param objectName
	 *            存储后的文件名（要包含路径信息）
	 * @param file
	 *            上传的文件
	 * @throws OssException
	 *             oss上传服务异常
	 */
	public void uploadFile(String objectName, File file) throws OssException {
		OSSClient ossClient = new OSSClient(endpoint, accessKeyId, accessKeySecret);
		try {
			if (ossClient.doesBucketExist(bucketName)) {
				ossClient.putObject(bucketName, objectName, file);
			}
		} catch (OSSException e) {
			e.printStackTrace();
			throw new OssException("oss02", "上传到OSS->" + "失败");
		} catch (ClientException e) {
			e.printStackTrace();
			throw new OssException("oss02", "上传到OSS->" + "失败");
		} finally {
			ossClient.shutdown();
		}
	}

	/**
	 * 删除OSS上指定的文件夹（包含此文件夹内所有文件） 注：删除数量有限制，最大100个
	 * 
	 * @param directory
	 *            文件夹在OSS的绝对路径
	 * @return 删除结果集合，无删除则返回空集合
	 * @throws OssException
	 *             oss删除目录服务异常
	 */
	public List<String> deleteDirectory(String directory) throws OssException {
		OSSClient ossClient = new OSSClient(endpoint, accessKeyId, accessKeySecret);
		try {
			if (ossClient.doesBucketExist(bucketName)) {
				ObjectListing objectListing = ossClient.listObjects(bucketName, directory + "/");
				List<OSSObjectSummary> objectSummary = objectListing.getObjectSummaries();
				List<String> keys = new ArrayList<String>();
				for (OSSObjectSummary object : objectSummary) {
					keys.add(object.getKey());
				}
				DeleteObjectsResult deleteObjectsResult = ossClient
						.deleteObjects(new DeleteObjectsRequest(bucketName).withKeys(keys));
				return deleteObjectsResult.getDeletedObjects();
			}
			return new ArrayList<>();
		} catch (OSSException e) {
			e.printStackTrace();
			throw new OssException("oss03", "OSS上删除目录失败");
		} catch (ClientException e) {
			e.printStackTrace();
			throw new OssException("oss03", "OSS上删除目录失败");
		} finally {
			ossClient.shutdown();
		}
	}

	/**
	 * 删除OSS上单个文件
	 * 
	 * @param file
	 *            文件的在OSS的绝对路径
	 * @throws OssException
	 *             oss删除单个文件服务异常
	 */
	public void deleteFile(String file) throws OssException {
		OSSClient ossClient = new OSSClient(endpoint, accessKeyId, accessKeySecret);
		try {
			if (ossClient.doesBucketExist(bucketName)) {
				ossClient.deleteObject(bucketName, file);
			}
		} catch (OSSException e) {
			e.printStackTrace();
			throw new OssException("oss04", "OSS上删除单个文件失败");
		} catch (ClientException e) {
			e.printStackTrace();
			throw new OssException("oss04", "OSS上删除单个文件失败");
		} finally {
			ossClient.shutdown();
		}
	}

	/**
	 * 生成OSS上某个对象的url
	 * 
	 * @param objectName
	 *            OSS上某个对象的完整地址
	 * @param validityPeriod
	 *            有效期，单位秒
	 * @return url 带有效期的url
	 * @throws OssException
	 *             oss生成url服务异常
	 */
	public String url(String objectName, long validityPeriod) throws OssException {
		OSSClient ossClient = new OSSClient(endpoint, accessKeyId, accessKeySecret);
		try {
			Date expiration = new Date(new Date().getTime() + validityPeriod * 1000);
			return ossClient.generatePresignedUrl(bucketName, objectName, expiration).toString();
		} catch (ClientException e) {
			e.printStackTrace();
			throw new OssException("oss05", "OSS生成url失败");
		}
	}

	/**
	 * 返回前缀一样的key的总数
	 * 
	 * @param keyPrefix
	 *            key的前缀
	 * @return 总数
	 */
	public int keyNums(String keyPrefix) {
		OSSClient ossClient = new OSSClient(endpoint, accessKeyId, accessKeySecret);
		int i = 0;
		try {
			if (ossClient.doesBucketExist(bucketName)) {
				final int maxKeys = 200;
				String nextMarker = null;
				ObjectListing objectListing;
				do {
					objectListing = ossClient.listObjects(new ListObjectsRequest(bucketName).withPrefix(keyPrefix)
							.withMarker(nextMarker).withMaxKeys(maxKeys));
					List<OSSObjectSummary> sums = objectListing.getObjectSummaries();
					i += sums.size();
					nextMarker = objectListing.getNextMarker();
				} while (objectListing.isTruncated());
			}
		} catch (OSSException e) {
			e.printStackTrace();
		} catch (ClientException e) {
			e.printStackTrace();
		} finally {
			ossClient.shutdown();
		}
		return i;
	}

}
