package cn.takujo.common_api.util;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.takujo.common_api.exception.HttpException;
import cn.takujo.common_api.exception.JsonException;
import lombok.AllArgsConstructor;
import lombok.Data;

/**
 * 微信小程序服务
 * 
 * @author wzx
 *
 */
@AllArgsConstructor
@Data
public final class WeiXinUtil {

	private String appId;
	private String appSecret;

	public String accessToken() throws HttpException, JsonException {
		String url = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=" + appId + "&secret="
				+ appSecret;
		String tokenId = HttpUtil.get(url);
		return (String) JsonUtil.toMap(tokenId).get("access_token");
	}

	/**
	 * 发送模板消息
	 * 
	 * @param map
	 *            调用templateMessageParamMap方法生成参数集，也可以参考小程序文档自定义
	 * 
	 * @return true是发送成功，false是发送失败
	 */
	public boolean sendTemplateMessage(Map<String, Object> map) {
		try {
			String accessToken = accessToken();
			String url = "https://api.weixin.qq.com/cgi-bin/message/wxopen/template/send?access_token=" + accessToken;
			String post = HttpUtil.post(url, JsonUtil.toString(map));
			return (int) JsonUtil.toMap(post).get("errcode") == 0;
		} catch (HttpException | JsonException e) {
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * 模板消息参数map
	 * 
	 * @param touser
	 *            接收者（用户）的 openid
	 * @param templateId
	 *            模板消息的id
	 * @param page
	 *            点击模板卡片后的跳转页面，仅限本小程序内的页面。
	 * @param formId
	 *            表单提交场景下，为 submit 事件带上的 formId；支付场景下，为本次支付的 prepay_id
	 * @param emphasisKeyword
	 *            模板需要放大的关键词，为null则无放大
	 * @param keywords
	 *            调用templateMessageKeyword方法生成后放入集合
	 * @return map 返回模板消息参数集
	 */
	public Map<String, Object> templateMessageParamMap(String touser, String templateId, String page, String formId,
			String emphasisKeyword, List<Map<String, Object>> keywords) {
		Map<String, Object> map = new HashMap<>();
		map.put("touser", touser);
		map.put("template_id", templateId);
		map.put("page", page);
		map.put("form_id", formId);
		Map<String, Object> data = new HashMap<>();
		for (int i = 0; i < keywords.size(); i++) {
			data.put("keyword" + (i + 1), keywords.get(i));
		}
		map.put("data", data);
		if (emphasisKeyword != null) {
			map.put("emphasis_keyword", emphasisKeyword);
		}
		return map;
	}

	/**
	 * 模板消息中keyword对象
	 * 
	 * @param value
	 *            所传入的值
	 * @return map keyword对象
	 */
	public Map<String, Object> templateMessageKeyword(String value) {
		Map<String, Object> map = new HashMap<>();
		map.put("value", value);
		return map;
	}

	/**
	 * 获取openid
	 * 
	 * @param code
	 *            微信小程序传来的code
	 * @return openid 返回openid
	 * @throws HttpException
	 *             http请求异常
	 * @throws JsonException
	 *             json解析异常
	 */
	public String getOpenid(String code) throws HttpException, JsonException {
		String url = "https://api.weixin.qq.com/sns/jscode2session?appid=" + appId + "&secret=" + appSecret
				+ "&js_code=" + code + "&grant_type=authorization_code";
		String data = HttpUtil.get(url);
		return JsonUtil.toMap(data).get("openid").toString();
	}

}
