/*
 * Decompiled with CFR 0.152.
 */
package infra.session;

import infra.lang.Assert;
import infra.lang.Nullable;
import infra.logging.Logger;
import infra.logging.LoggerFactory;
import infra.session.AbstractWebSession;
import infra.session.SerializableSession;
import infra.session.SessionEventDispatcher;
import infra.session.SessionIdGenerator;
import infra.session.SessionRepository;
import infra.session.TooManyActiveSessionsException;
import infra.session.WebSession;
import infra.util.LogFormatUtils;
import infra.util.StringUtils;
import java.io.IOException;
import java.io.NotSerializableException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.io.WriteAbortedException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;

public class InMemorySessionRepository
implements SessionRepository {
    private static final Logger log = LoggerFactory.getLogger(InMemorySessionRepository.class);
    private int maxSessions = 10000;
    private Clock clock = Clock.systemUTC();
    private boolean notifyBindingListenerOnUnchangedValue;
    private boolean notifyAttributeListenerOnUnchangedValue = true;
    private Duration maxIdleTime = Duration.ofMinutes(30L);
    private final SessionIdGenerator idGenerator;
    private final SessionEventDispatcher eventDispatcher;
    private final ExpiredSessionChecker expiredSessionChecker = new ExpiredSessionChecker();
    private final ConcurrentHashMap<String, InMemoryWebSession> sessions = new ConcurrentHashMap();

    public InMemorySessionRepository(SessionEventDispatcher eventDispatcher, SessionIdGenerator idGenerator) {
        Assert.notNull((Object)idGenerator, (String)"SessionIdGenerator is required");
        Assert.notNull((Object)eventDispatcher, (String)"SessionEventDispatcher is required");
        this.idGenerator = idGenerator;
        this.eventDispatcher = eventDispatcher;
    }

    public void setMaxSessions(int maxSessions) {
        this.maxSessions = maxSessions;
    }

    public int getMaxSessions() {
        return this.maxSessions;
    }

    public void setSessionMaxIdleTime(@Nullable Duration timeout) {
        this.maxIdleTime = timeout == null ? Duration.ofMinutes(30L) : timeout;
    }

    public void setNotifyBindingListenerOnUnchangedValue(boolean notifyBindingListenerOnUnchangedValue) {
        this.notifyBindingListenerOnUnchangedValue = notifyBindingListenerOnUnchangedValue;
    }

    public void setNotifyAttributeListenerOnUnchangedValue(boolean notifyAttributeListenerOnUnchangedValue) {
        this.notifyAttributeListenerOnUnchangedValue = notifyAttributeListenerOnUnchangedValue;
    }

    public void setClock(Clock clock) {
        Assert.notNull((Object)clock, (String)"Clock is required");
        this.clock = clock;
        this.removeExpiredSessions();
    }

    public Clock getClock() {
        return this.clock;
    }

    @Override
    public int getSessionCount() {
        return this.sessions.size();
    }

    @Override
    public String[] getIdentifiers() {
        return StringUtils.toStringArray((Collection)this.sessions.keySet());
    }

    public Map<String, WebSession> getSessions() {
        return Collections.unmodifiableMap(this.sessions);
    }

    @Override
    public WebSession createSession() {
        return this.createSession(this.idGenerator.generateId());
    }

    @Override
    public WebSession createSession(String id) {
        Instant now = this.clock.instant();
        this.expiredSessionChecker.checkIfNecessary(now);
        return new InMemoryWebSession(id, now, this.maxIdleTime);
    }

    @Override
    public WebSession retrieveSession(String id) {
        Instant now = this.clock.instant();
        this.expiredSessionChecker.checkIfNecessary(now);
        InMemoryWebSession session = this.sessions.get(id);
        if (session == null) {
            return null;
        }
        if (session.isExpired(now)) {
            this.sessions.remove(id);
            return null;
        }
        session.lastAccessTime = now;
        return session;
    }

    @Override
    public WebSession removeSession(String id) {
        return this.sessions.remove(id);
    }

    @Override
    public void updateLastAccessTime(WebSession session) {
        session.setLastAccessTime(this.clock.instant());
    }

    @Override
    public boolean contains(String id) {
        return this.sessions.containsKey(id);
    }

    public void removeExpiredSessions() {
        this.expiredSessionChecker.removeExpiredSessions(this.clock.instant());
    }

    private final class ExpiredSessionChecker {
        private static final int CHECK_PERIOD = 60000;
        private final ReentrantLock lock = new ReentrantLock();
        private Instant checkTime;

        private ExpiredSessionChecker() {
            this.checkTime = InMemorySessionRepository.this.clock.instant().plus(60000L, ChronoUnit.MILLIS);
        }

        public void checkIfNecessary(Instant now) {
            if (this.checkTime.isBefore(now)) {
                this.removeExpiredSessions(now);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void removeExpiredSessions(Instant now) {
            if (!InMemorySessionRepository.this.sessions.isEmpty() && this.lock.tryLock()) {
                try {
                    Iterator<InMemoryWebSession> iterator = InMemorySessionRepository.this.sessions.values().iterator();
                    while (iterator.hasNext()) {
                        InMemoryWebSession session = iterator.next();
                        if (!session.isExpired(now)) continue;
                        iterator.remove();
                        session.invalidate();
                    }
                }
                finally {
                    this.checkTime = now.plus(60000L, ChronoUnit.MILLIS);
                    this.lock.unlock();
                }
            }
        }
    }

    final class InMemoryWebSession
    extends AbstractWebSession
    implements WebSession,
    Serializable,
    SerializableSession {
        private static final long serialVersionUID = 1L;
        private Instant creationTime;
        private volatile Duration maxIdleTime;
        private volatile Instant lastAccessTime;
        private final AtomicReference<String> id;
        private final AtomicReference<State> state;

        InMemoryWebSession(String id, Instant creationTime, Duration maxIdleTime) {
            super(InMemorySessionRepository.this.eventDispatcher);
            this.state = new AtomicReference<State>(State.NEW);
            this.id = new AtomicReference<String>(id);
            this.maxIdleTime = maxIdleTime;
            this.lastAccessTime = this.creationTime = creationTime;
        }

        @Override
        public String getId() {
            return this.id.get();
        }

        @Override
        public Instant getCreationTime() {
            return this.creationTime;
        }

        @Override
        public Instant getLastAccessTime() {
            return this.lastAccessTime;
        }

        @Override
        public void setLastAccessTime(Instant lastAccessTime) {
            this.lastAccessTime = lastAccessTime;
        }

        @Override
        public void changeSessionId() {
            InMemorySessionRepository.this.sessions.remove(this.getId());
            String newId = InMemorySessionRepository.this.idGenerator.generateId();
            this.id.set(newId);
            InMemorySessionRepository.this.sessions.put(newId, this);
        }

        @Override
        protected void doInvalidate() {
            this.state.set(State.EXPIRED);
            InMemorySessionRepository.this.sessions.remove(this.getId());
        }

        @Override
        public void save() {
            this.checkMaxSessionsLimit();
            if (this.hasAttributes()) {
                this.state.compareAndSet(State.NEW, State.STARTED);
            }
            if (this.isStarted()) {
                InMemorySessionRepository.this.sessions.put(this.getId(), this);
                if (this.state.get().equals((Object)State.EXPIRED)) {
                    InMemorySessionRepository.this.sessions.remove(this.getId());
                    throw new IllegalStateException("Session was invalidated");
                }
            }
        }

        @Override
        public void setMaxIdleTime(Duration maxIdleTime) {
            this.maxIdleTime = maxIdleTime;
        }

        @Override
        public Duration getMaxIdleTime() {
            return this.maxIdleTime;
        }

        @Override
        public void start() {
            this.state.compareAndSet(State.NEW, State.STARTED);
            this.eventDispatcher.onSessionCreated(this);
        }

        @Override
        public boolean isStarted() {
            return this.state.get().equals((Object)State.STARTED) || this.attributes != null;
        }

        @Override
        protected boolean attributeBinding(Object value, @Nullable Object oldValue) {
            return oldValue != value || InMemorySessionRepository.this.notifyBindingListenerOnUnchangedValue;
        }

        @Override
        protected boolean allowAttributeReplaced(Object value, @Nullable Object oldValue) {
            return value != oldValue || InMemorySessionRepository.this.notifyAttributeListenerOnUnchangedValue;
        }

        private void checkMaxSessionsLimit() {
            if (InMemorySessionRepository.this.sessions.size() >= InMemorySessionRepository.this.maxSessions) {
                InMemorySessionRepository.this.expiredSessionChecker.removeExpiredSessions(InMemorySessionRepository.this.clock.instant());
                if (InMemorySessionRepository.this.sessions.size() >= InMemorySessionRepository.this.maxSessions) {
                    throw new TooManyActiveSessionsException("Max sessions limit reached: " + InMemorySessionRepository.this.sessions.size(), InMemorySessionRepository.this.maxSessions);
                }
            }
        }

        @Override
        public boolean isExpired() {
            return this.isExpired(InMemorySessionRepository.this.clock.instant());
        }

        private boolean isExpired(Instant now) {
            if (this.state.get().equals((Object)State.EXPIRED)) {
                return true;
            }
            if (this.checkExpired(now)) {
                this.state.set(State.EXPIRED);
                return true;
            }
            return false;
        }

        private boolean checkExpired(Instant currentTime) {
            return this.isStarted() && !this.maxIdleTime.isNegative() && currentTime.minus(this.maxIdleTime).isAfter(this.lastAccessTime);
        }

        @Override
        public void writeObjectData(ObjectOutputStream stream) throws IOException {
            stream.writeObject(this.creationTime);
            stream.writeObject(this.lastAccessTime);
            stream.writeObject(this.maxIdleTime);
            stream.writeObject((Object)this.state.get());
            if (this.attributes != null) {
                if (log.isDebugEnabled()) {
                    log.debug("writeObject() [{}]", (Object)this.attributes);
                }
                ArrayList<String> saveNames = new ArrayList<String>();
                ArrayList saveValues = new ArrayList();
                for (Map.Entry entry : this.attributes.entrySet()) {
                    String key = (String)entry.getKey();
                    Object value = entry.getValue();
                    if (!(value instanceof Serializable)) continue;
                    saveNames.add(key);
                    saveValues.add(value);
                }
                int n = saveNames.size();
                stream.writeInt(n);
                for (int i = 0; i < n; ++i) {
                    String name = (String)saveNames.get(i);
                    stream.writeObject(name);
                    try {
                        Object object = saveValues.get(i);
                        stream.writeObject(object);
                        LogFormatUtils.traceDebug((Logger)log, traceOn -> LogFormatUtils.formatValue((Object)"  storing attribute '%s' with value '%s'".formatted(name, object), (traceOn == false ? 1 : 0) != 0));
                        continue;
                    }
                    catch (NotSerializableException e) {
                        log.warn("Cannot serialize session attribute [{}] for session [{}]", new Object[]{name, this.id, e});
                    }
                }
            } else {
                stream.writeInt(0);
            }
        }

        @Override
        public void readObjectData(ObjectInputStream stream) throws ClassNotFoundException, IOException {
            int size;
            this.creationTime = (Instant)stream.readObject();
            this.lastAccessTime = (Instant)stream.readObject();
            this.maxIdleTime = (Duration)stream.readObject();
            this.state.set((State)((Object)stream.readObject()));
            if (log.isDebugEnabled()) {
                log.debug("readObject() loading session {}", this.id);
            }
            if ((size = stream.readInt()) > 0) {
                for (int i = 0; i < size; ++i) {
                    Object value;
                    String name = (String)stream.readObject();
                    try {
                        value = stream.readObject();
                    }
                    catch (WriteAbortedException wae) {
                        if (wae.getCause() instanceof NotSerializableException) {
                            if (log.isDebugEnabled()) {
                                log.debug("Cannot deserialize session attribute [{}] for session [{}]", new Object[]{name, this.id, wae});
                                continue;
                            }
                            log.warn("Cannot deserialize session attribute [{}] for session [{}]", (Object)name, this.id);
                            continue;
                        }
                        throw wae;
                    }
                    this.setAttribute(name, value);
                    LogFormatUtils.traceDebug((Logger)log, traceOn -> LogFormatUtils.formatValue((Object)"  loading attribute '%s' with value '%s'".formatted(name, value), (traceOn == false ? 1 : 0) != 0));
                }
            }
            InMemorySessionRepository.this.sessions.put(this.getId(), this);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            InMemoryWebSession that = (InMemoryWebSession)o;
            return Objects.equals(this.id.get(), that.id.get()) && Objects.equals((Object)this.state.get(), (Object)that.state.get()) && Objects.equals(this.creationTime, that.creationTime) && Objects.equals(this.maxIdleTime, that.maxIdleTime) && Objects.equals(this.lastAccessTime, that.lastAccessTime);
        }

        @Override
        public int hashCode() {
            return Objects.hash(super.hashCode(), this.id, this.creationTime, this.lastAccessTime, this.maxIdleTime, this.state);
        }
    }

    private static enum State {
        NEW,
        STARTED,
        EXPIRED;

    }
}

