/*
 * Decompiled with CFR 0.152.
 */
package infra.http.codec.multipart;

import infra.core.codec.DecodingException;
import infra.core.io.buffer.DataBuffer;
import infra.core.io.buffer.DataBufferLimitException;
import infra.core.io.buffer.DataBufferUtils;
import infra.http.DefaultHttpHeaders;
import infra.http.HttpHeaders;
import infra.http.codec.multipart.MultipartUtils;
import infra.lang.Nullable;
import infra.logging.Logger;
import infra.logging.LoggerFactory;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.reactivestreams.Subscription;
import reactor.core.CoreSubscriber;
import reactor.core.publisher.BaseSubscriber;
import reactor.core.publisher.Flux;
import reactor.core.publisher.FluxSink;
import reactor.util.context.Context;
import reactor.util.context.ContextView;

final class MultipartParser
extends BaseSubscriber<DataBuffer> {
    private static final Logger log = LoggerFactory.getLogger(MultipartParser.class);
    private static final byte CR = 13;
    private static final byte LF = 10;
    private static final byte[] CR_LF = new byte[]{13, 10};
    private static final byte[] DOUBLE_CR_LF = new byte[]{13, 10, 13, 10};
    private static final byte HYPHEN = 45;
    private static final byte[] TWO_HYPHENS = new byte[]{45, 45};
    private static final String HEADER_ENTRY_SEPARATOR = "\\r\\n";
    private final FluxSink<Token> sink;
    private final AtomicReference<State> state;
    private final byte[] boundary;
    private final int maxHeadersSize;
    private final Charset headersCharset;
    private final AtomicBoolean requestOutstanding = new AtomicBoolean();

    private MultipartParser(FluxSink<Token> sink, byte[] boundary, int maxHeadersSize, Charset headersCharset) {
        this.sink = sink;
        this.boundary = boundary;
        this.maxHeadersSize = maxHeadersSize;
        this.headersCharset = headersCharset;
        this.state = new AtomicReference<PreambleState>(new PreambleState());
    }

    public static Flux<Token> parse(Flux<DataBuffer> buffers, byte[] boundary, int maxHeadersSize, Charset headersCharset) {
        return Flux.create(sink -> {
            MultipartParser parser = new MultipartParser((FluxSink<Token>)sink, boundary, maxHeadersSize, headersCharset);
            sink.onCancel(parser::onSinkCancel);
            sink.onRequest(n -> parser.requestBuffer());
            buffers.subscribe((CoreSubscriber)parser);
        });
    }

    public Context currentContext() {
        return Context.of((ContextView)this.sink.contextView());
    }

    protected void hookOnSubscribe(Subscription subscription) {
        this.requestBuffer();
    }

    protected void hookOnNext(DataBuffer value) {
        this.requestOutstanding.set(false);
        this.state.get().onNext(value);
    }

    protected void hookOnComplete() {
        this.state.get().onComplete();
    }

    protected void hookOnError(Throwable throwable) {
        State oldState = this.state.getAndSet(DisposedState.INSTANCE);
        oldState.dispose();
        this.sink.error(throwable);
    }

    private void onSinkCancel() {
        State oldState = this.state.getAndSet(DisposedState.INSTANCE);
        oldState.dispose();
        this.cancel();
    }

    boolean changeState(State oldState, State newState, @Nullable DataBuffer remainder) {
        if (this.state.compareAndSet(oldState, newState)) {
            if (log.isTraceEnabled()) {
                log.trace("Changed state: {} -> {}", (Object)oldState, (Object)newState);
            }
            oldState.dispose();
            if (remainder != null) {
                if (remainder.readableByteCount() > 0) {
                    newState.onNext(remainder);
                } else {
                    DataBufferUtils.release((DataBuffer)remainder);
                    this.requestBuffer();
                }
            }
            return true;
        }
        DataBufferUtils.release((DataBuffer)remainder);
        return false;
    }

    void emitHeaders(HttpHeaders headers) {
        if (log.isTraceEnabled()) {
            log.trace("Emitting headers: {}", (Object)headers);
        }
        this.sink.next((Object)new HeadersToken(headers));
    }

    void emitBody(DataBuffer buffer, boolean last) {
        if (log.isTraceEnabled()) {
            log.trace("Emitting body: {}", (Object)buffer);
        }
        this.sink.next((Object)new BodyToken(buffer, last));
    }

    void emitError(Throwable t) {
        this.cancel();
        this.sink.error(t);
    }

    void emitComplete() {
        this.cancel();
        this.sink.complete();
    }

    private void requestBuffer() {
        if (this.upstream() != null && !this.sink.isCancelled() && this.sink.requestedFromDownstream() > 0L && this.requestOutstanding.compareAndSet(false, true)) {
            this.request(1L);
        }
    }

    private final class PreambleState
    extends State {
        private final DataBufferUtils.Matcher firstBoundary;

        public PreambleState() {
            this.firstBoundary = DataBufferUtils.matcher((byte[])MultipartUtils.concat(TWO_HYPHENS, MultipartParser.this.boundary));
        }

        @Override
        public void onNext(DataBuffer buf) {
            int endIdx = this.firstBoundary.match(buf);
            if (endIdx != -1) {
                if (log.isTraceEnabled()) {
                    log.trace("First boundary found @{} in {}", (Object)endIdx, (Object)buf);
                }
                DataBuffer preambleBuffer = buf.split(endIdx + 1);
                DataBufferUtils.release((DataBuffer)preambleBuffer);
                MultipartParser.this.changeState(this, new HeadersState(), buf);
            } else {
                DataBufferUtils.release((DataBuffer)buf);
                MultipartParser.this.requestBuffer();
            }
        }

        @Override
        public void onComplete() {
            if (MultipartParser.this.changeState(this, DisposedState.INSTANCE, null)) {
                MultipartParser.this.emitError((Throwable)new DecodingException("Could not find first boundary"));
            }
        }

        public String toString() {
            return "PREAMBLE";
        }
    }

    private static abstract class State {
        private State() {
        }

        abstract void onNext(DataBuffer var1);

        abstract void onComplete();

        void dispose() {
        }
    }

    private static final class DisposedState
    extends State {
        public static final DisposedState INSTANCE = new DisposedState();

        private DisposedState() {
        }

        @Override
        public void onNext(DataBuffer buf) {
            DataBufferUtils.release((DataBuffer)buf);
        }

        @Override
        public void onComplete() {
        }

        public String toString() {
            return "DISPOSED";
        }
    }

    public static final class HeadersToken
    extends Token {
        private final HttpHeaders headers;

        public HeadersToken(HttpHeaders headers) {
            this.headers = headers;
        }

        @Override
        public HttpHeaders getHeaders() {
            return this.headers;
        }

        @Override
        public DataBuffer getBuffer() {
            throw new IllegalStateException();
        }

        @Override
        public boolean isLast() {
            return false;
        }
    }

    public static final class BodyToken
    extends Token {
        private final boolean last;
        private final DataBuffer buffer;

        public BodyToken(DataBuffer buffer, boolean last) {
            this.buffer = buffer;
            this.last = last;
        }

        @Override
        public HttpHeaders getHeaders() {
            throw new IllegalStateException();
        }

        @Override
        public DataBuffer getBuffer() {
            return this.buffer;
        }

        @Override
        public boolean isLast() {
            return this.last;
        }
    }

    private final class BodyState
    extends State {
        private final DataBufferUtils.Matcher boundary;
        private final int boundaryLength;
        private final Deque<DataBuffer> queue = new ConcurrentLinkedDeque<DataBuffer>();

        public BodyState() {
            byte[] delimiter = MultipartUtils.concat(CR_LF, TWO_HYPHENS, MultipartParser.this.boundary);
            this.boundary = DataBufferUtils.matcher((byte[])delimiter);
            this.boundaryLength = delimiter.length;
        }

        @Override
        public void onNext(DataBuffer buffer) {
            int endIdx = this.boundary.match(buffer);
            if (endIdx != -1) {
                DataBuffer boundaryBuffer;
                int len;
                if (log.isTraceEnabled()) {
                    log.trace("Boundary found @{} in {}", (Object)endIdx, (Object)buffer);
                }
                if ((len = endIdx - this.boundaryLength + 1 - (boundaryBuffer = buffer.split(endIdx + 1)).readPosition()) > 0) {
                    DataBuffer body = boundaryBuffer.split(len);
                    DataBufferUtils.release((DataBuffer)boundaryBuffer);
                    this.enqueue(body);
                    this.flush();
                } else if (len < 0) {
                    DataBuffer prev;
                    DataBufferUtils.release((DataBuffer)boundaryBuffer);
                    while ((prev = this.queue.pollLast()) != null) {
                        int prevByteCount = prev.readableByteCount();
                        int prevLen = prevByteCount + len;
                        if (prevLen >= 0) {
                            DataBuffer body = prev.split(prevLen + prev.readPosition());
                            DataBufferUtils.release((DataBuffer)prev);
                            this.enqueue(body);
                            this.flush();
                            break;
                        }
                        DataBufferUtils.release((DataBuffer)prev);
                        len += prevByteCount;
                    }
                } else {
                    DataBufferUtils.release((DataBuffer)boundaryBuffer);
                    this.flush();
                }
                MultipartParser.this.changeState(this, new HeadersState(), buffer);
            } else {
                this.enqueue(buffer);
                MultipartParser.this.requestBuffer();
            }
        }

        private void enqueue(DataBuffer buf) {
            this.queue.add(buf);
            int len = 0;
            ArrayDeque<DataBuffer> emit = new ArrayDeque<DataBuffer>();
            Iterator<DataBuffer> iterator = this.queue.descendingIterator();
            while (iterator.hasNext()) {
                DataBuffer previous = iterator.next();
                if (len > this.boundaryLength) {
                    emit.addFirst(previous);
                    iterator.remove();
                }
                len += previous.readableByteCount();
            }
            for (DataBuffer buffer : emit) {
                MultipartParser.this.emitBody(buffer, false);
            }
        }

        private void flush() {
            Iterator<DataBuffer> iterator = this.queue.iterator();
            while (iterator.hasNext()) {
                DataBuffer buffer = iterator.next();
                boolean last = !iterator.hasNext();
                MultipartParser.this.emitBody(buffer, last);
            }
            this.queue.clear();
        }

        @Override
        public void onComplete() {
            if (MultipartParser.this.changeState(this, DisposedState.INSTANCE, null)) {
                String msg = "Could not find end of body (\u240d\u240a--" + new String(MultipartParser.this.boundary, StandardCharsets.UTF_8) + ")";
                MultipartParser.this.emitError((Throwable)new DecodingException(msg));
            }
        }

        @Override
        public void dispose() {
            this.queue.forEach(DataBufferUtils::release);
            this.queue.clear();
        }

        public String toString() {
            return "BODY";
        }
    }

    private final class HeadersState
    extends State {
        private final DataBufferUtils.Matcher endHeaders = DataBufferUtils.matcher((byte[])DOUBLE_CR_LF);
        private final AtomicInteger byteCount = new AtomicInteger();
        private final ArrayList<DataBuffer> buffers = new ArrayList();

        private HeadersState() {
        }

        @Override
        public void onNext(DataBuffer buf) {
            if (this.isLastBoundary(buf)) {
                if (log.isTraceEnabled()) {
                    log.trace("Last boundary found in {}", (Object)buf);
                }
                if (MultipartParser.this.changeState(this, DisposedState.INSTANCE, buf)) {
                    MultipartParser.this.emitComplete();
                }
                return;
            }
            int endIdx = this.endHeaders.match(buf);
            if (endIdx != -1) {
                long count;
                if (log.isTraceEnabled()) {
                    log.trace("End of headers found @{} in {}", (Object)endIdx, (Object)buf);
                }
                if (this.belowMaxHeaderSize(count = (long)this.byteCount.addAndGet(endIdx))) {
                    DataBuffer headerBuf = buf.split(endIdx + 1);
                    this.buffers.add(headerBuf);
                    MultipartParser.this.emitHeaders(this.parseHeaders());
                    MultipartParser.this.changeState(this, new BodyState(), buf);
                }
            } else {
                long count = this.byteCount.addAndGet(buf.readableByteCount());
                if (this.belowMaxHeaderSize(count)) {
                    this.buffers.add(buf);
                    MultipartParser.this.requestBuffer();
                }
            }
        }

        private boolean isLastBoundary(DataBuffer buf) {
            return this.buffers.isEmpty() && buf.readableByteCount() >= 2 && buf.getByte(0) == 45 && buf.getByte(1) == 45 || this.buffers.size() == 1 && this.buffers.get(0).readableByteCount() == 1 && this.buffers.get(0).getByte(0) == 45 && buf.readableByteCount() >= 1 && buf.getByte(0) == 45;
        }

        private boolean belowMaxHeaderSize(long count) {
            if (count <= (long)MultipartParser.this.maxHeadersSize) {
                return true;
            }
            MultipartParser.this.emitError((Throwable)new DataBufferLimitException("Part headers exceeded the memory usage limit of " + MultipartParser.this.maxHeadersSize + " bytes"));
            return false;
        }

        private HttpHeaders parseHeaders() {
            if (this.buffers.isEmpty()) {
                return HttpHeaders.empty();
            }
            DataBuffer joined = this.buffers.get(0).factory().join(this.buffers);
            this.buffers.clear();
            String string = joined.toString(MultipartParser.this.headersCharset);
            DataBufferUtils.release((DataBuffer)joined);
            DefaultHttpHeaders result = HttpHeaders.forWritable();
            for (String line : string.split(MultipartParser.HEADER_ENTRY_SEPARATOR)) {
                int idx = line.indexOf(58);
                if (idx == -1) continue;
                String name = line.substring(0, idx);
                String value = line.substring(idx + 1);
                while (value.startsWith(" ")) {
                    value = value.substring(1);
                }
                ((HttpHeaders)result).add(name, value);
            }
            return result;
        }

        @Override
        public void onComplete() {
            if (MultipartParser.this.changeState(this, DisposedState.INSTANCE, null)) {
                MultipartParser.this.emitError((Throwable)new DecodingException("Could not find end of headers"));
            }
        }

        @Override
        public void dispose() {
            for (DataBuffer buffer : this.buffers) {
                DataBufferUtils.release((DataBuffer)buffer);
            }
        }

        public String toString() {
            return "HEADERS";
        }
    }

    public static abstract class Token {
        public abstract DataBuffer getBuffer();

        public abstract HttpHeaders getHeaders();

        public abstract boolean isLast();
    }
}

