/*
 * Decompiled with CFR 0.152.
 */
package infra.http.client;

import infra.http.HttpHeaders;
import infra.http.HttpMethod;
import infra.http.StreamingHttpOutputMessage;
import infra.http.client.AbstractStreamingClientHttpRequest;
import infra.http.client.ClientHttpResponse;
import infra.http.client.JdkClientHttpResponse;
import infra.http.client.OutputStreamPublisher;
import infra.lang.Nullable;
import infra.util.StreamUtils;
import infra.util.StringUtils;
import infra.util.concurrent.Future;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpTimeoutException;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

class JdkClientHttpRequest
extends AbstractStreamingClientHttpRequest {
    private static final OutputStreamPublisher.ByteMapper<ByteBuffer> BYTE_MAPPER = new ByteBufferMapper();
    private static final Set<String> DISALLOWED_HEADERS = JdkClientHttpRequest.disallowedHeaders();
    private final HttpClient httpClient;
    private final HttpMethod method;
    private final URI uri;
    private final Executor executor;
    @Nullable
    private final Duration timeout;

    private static Set<String> disallowedHeaders() {
        TreeSet<String> headers = new TreeSet<String>(String.CASE_INSENSITIVE_ORDER);
        headers.addAll(Set.of("connection", "content-length", "expect", "host", "upgrade"));
        String headersToAllow = System.getProperty("jdk.httpclient.allowRestrictedHeaders");
        if (headersToAllow != null) {
            Set toAllow = StringUtils.commaDelimitedListToSet((String)headersToAllow);
            headers.removeAll(toAllow);
        }
        return Collections.unmodifiableSet(headers);
    }

    public JdkClientHttpRequest(HttpClient httpClient, URI uri, HttpMethod method, Executor executor, @Nullable Duration readTimeout) {
        this.httpClient = httpClient;
        this.uri = uri;
        this.method = method;
        this.executor = executor;
        this.timeout = readTimeout;
    }

    @Override
    public HttpMethod getMethod() {
        return this.method;
    }

    @Override
    public URI getURI() {
        return this.uri;
    }

    @Override
    protected ClientHttpResponse executeInternal(HttpHeaders headers, @Nullable StreamingHttpOutputMessage.Body body) throws IOException {
        CompletableFuture<HttpResponse<InputStream>> responseFuture = null;
        try {
            HttpRequest request = this.buildRequest(headers, body);
            responseFuture = this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofInputStream());
            if (this.timeout != null) {
                TimeoutHandler timeoutHandler = new TimeoutHandler(responseFuture, this.timeout);
                HttpResponse<InputStream> response = responseFuture.get();
                InputStream inputStream = timeoutHandler.wrapInputStream(response);
                return new JdkClientHttpResponse(response, inputStream);
            }
            HttpResponse<InputStream> response = responseFuture.get();
            return new JdkClientHttpResponse(response, response.body());
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            responseFuture.cancel(true);
            throw new IOException("Request was interrupted: " + ex.getMessage(), ex);
        }
        catch (ExecutionException ex) {
            Throwable cause = ex.getCause();
            if (cause instanceof CancellationException) {
                throw new HttpTimeoutException("Request timed out");
            }
            if (cause instanceof UncheckedIOException) {
                UncheckedIOException uioEx = (UncheckedIOException)cause;
                throw uioEx.getCause();
            }
            if (cause instanceof RuntimeException) {
                RuntimeException rtEx = (RuntimeException)cause;
                throw rtEx;
            }
            if (cause instanceof IOException) {
                IOException ioEx = (IOException)cause;
                throw ioEx;
            }
            throw new IOException(cause.getMessage(), cause);
        }
    }

    @Override
    protected Future<ClientHttpResponse> asyncInternal(HttpHeaders headers, @Nullable StreamingHttpOutputMessage.Body body, @Nullable Executor executor) {
        HttpRequest request = this.buildRequest(headers, body);
        Future responseFuture = Future.forAdaption(this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofInputStream()), (Executor)executor);
        if (this.timeout != null) {
            responseFuture = responseFuture.timeout(this.timeout);
        }
        return responseFuture.map(JdkClientHttpResponse::new);
    }

    private HttpRequest buildRequest(HttpHeaders headers, @Nullable StreamingHttpOutputMessage.Body body) {
        HttpRequest.Builder builder = HttpRequest.newBuilder().uri(this.uri);
        for (Map.Entry entry : headers.entrySet()) {
            String headerName = (String)entry.getKey();
            if (DISALLOWED_HEADERS.contains(headerName.toLowerCase(Locale.ROOT))) continue;
            for (String headerValue : (List)entry.getValue()) {
                builder.header(headerName, headerValue);
            }
        }
        builder.method(this.method.name(), this.bodyPublisher(headers, body));
        return builder.build();
    }

    private HttpRequest.BodyPublisher bodyPublisher(HttpHeaders headers, @Nullable StreamingHttpOutputMessage.Body body) {
        if (body != null) {
            OutputStreamPublisher<ByteBuffer> publisher = new OutputStreamPublisher<ByteBuffer>(os -> body.writeTo(StreamUtils.nonClosing((OutputStream)os)), BYTE_MAPPER, this.executor, null);
            long contentLength = headers.getContentLength();
            if (contentLength > 0L) {
                return HttpRequest.BodyPublishers.fromPublisher(publisher, contentLength);
            }
            if (contentLength == 0L) {
                return HttpRequest.BodyPublishers.noBody();
            }
            return HttpRequest.BodyPublishers.fromPublisher(publisher);
        }
        return HttpRequest.BodyPublishers.noBody();
    }

    private static final class TimeoutHandler {
        private final CompletableFuture<Void> timeoutFuture;

        private TimeoutHandler(CompletableFuture<HttpResponse<InputStream>> future, Duration timeout) {
            this.timeoutFuture = new CompletableFuture<Object>().completeOnTimeout(null, timeout.toMillis(), TimeUnit.MILLISECONDS);
            this.timeoutFuture.thenRun(() -> {
                if (future.cancel(true) || future.isCompletedExceptionally() || !future.isDone()) {
                    return;
                }
                try {
                    ((InputStream)((HttpResponse)future.get()).body()).close();
                }
                catch (Exception exception) {
                    // empty catch block
                }
            });
        }

        @Nullable
        public InputStream wrapInputStream(HttpResponse<InputStream> response) {
            InputStream body = response.body();
            if (body == null) {
                return null;
            }
            return new FilterInputStream(body){

                @Override
                public void close() throws IOException {
                    timeoutFuture.cancel(false);
                    super.close();
                }
            };
        }
    }

    private static final class ByteBufferMapper
    implements OutputStreamPublisher.ByteMapper<ByteBuffer> {
        private ByteBufferMapper() {
        }

        @Override
        public ByteBuffer map(int b) {
            ByteBuffer byteBuffer = ByteBuffer.allocate(1);
            byteBuffer.put((byte)b);
            byteBuffer.flip();
            return byteBuffer;
        }

        @Override
        public ByteBuffer map(byte[] b, int off, int len) {
            ByteBuffer byteBuffer = ByteBuffer.allocate(len);
            byteBuffer.put(b, off, len);
            byteBuffer.flip();
            return byteBuffer;
        }
    }
}

