/*
 * Copyright 2017 - 2024 the original author or authors.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see [https://www.gnu.org/licenses/]
 */

package infra.web.service.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import infra.aot.hint.annotation.Reflective;
import infra.core.annotation.AliasFor;
import infra.core.io.Resource;
import infra.http.HttpEntity;
import infra.web.service.invoker.RequestPartArgumentResolver;
import infra.web.util.UriBuilderFactory;

/**
 * Annotation to declare a method on an HTTP service interface as an HTTP
 * endpoint. The endpoint details are defined statically through attributes of
 * the annotation, as well as through the input method argument types.
 *
 * <p>Supported at the type level to express common attributes, to be inherited
 * by all methods, such as a base URL path.
 *
 * <p>At the method level, it's more common to use one of the following HTTP method
 * specific, shortcut annotations, each of which is itself <em>meta-annotated</em>
 * with {@code HttpExchange}:
 *
 * <ul>
 * <li>{@link GetExchange}
 * <li>{@link PostExchange}
 * <li>{@link PutExchange}
 * <li>{@link PatchExchange}
 * <li>{@link DeleteExchange}
 * </ul>
 *
 * <p>Supported method arguments:
 * <table border="1">
 * <tr>
 * <th>Method Argument</th>
 * <th>Description</th>
 * <th>Resolver</th>
 * </tr>
 * <tr>
 * <td>{@link java.net.URI URI}</td>
 * <td>Dynamically set the URL for the request, overriding the annotation's
 * {@link #url()} attribute</td>
 * <td>{@link infra.web.service.invoker.UrlArgumentResolver}</td>
 * </tr>
 * <td>{@link UriBuilderFactory}</td>
 * <td>Dynamically set the {@code base URI} for the request, overriding the
 * one from the annotation's {@link #url()} attribute, while keeping the
 * subsequent path segments as defined there</td>
 * <td>{@link infra.web.service.invoker.UriBuilderFactoryArgumentResolver}</td>
 * </tr>
 * <tr>
 * <td>{@link infra.http.HttpMethod HttpMethod}</td>
 * <td>Dynamically set the HTTP method for the request, overriding the annotation's
 * {@link #method()} attribute</td>
 * <td>{@link infra.web.service.invoker.HttpMethodArgumentResolver
 * HttpMethodArgumentResolver}</td>
 * </tr>
 * <tr>
 * <td>{@link infra.web.annotation.RequestHeader @RequestHeader}</td>
 * <td>Add a request header</td>
 * <td>{@link infra.web.service.invoker.RequestHeaderArgumentResolver
 * RequestHeaderArgumentResolver}</td>
 * </tr>
 * <tr>
 * <td>{@link infra.web.annotation.PathVariable @PathVariable}</td>
 * <td>Add a path variable for the URI template</td>
 * <td>{@link infra.web.service.invoker.PathVariableArgumentResolver
 * PathVariableArgumentResolver}</td>
 * </tr>
 * <tr>
 * <td>{@link infra.web.annotation.RequestBody @RequestBody}</td>
 * <td>Set the body of the request</td>
 * <td>{@link infra.web.service.invoker.RequestBodyArgumentResolver
 * RequestBodyArgumentResolver}</td>
 * </tr>
 * <tr>
 * <td>{@link infra.web.annotation.RequestParam @RequestParam}</td>
 * <td>Add a request parameter, either form data if {@code "Content-Type"} is
 * {@code "application/x-www-form-urlencoded"} or query params otherwise</td>
 * <td>{@link infra.web.service.invoker.RequestParamArgumentResolver
 * RequestParamArgumentResolver}</td>
 * </tr>
 * <tr>
 * <td>{@link infra.web.annotation.RequestPart @RequestPart}</td>
 * <td>Add a request part, which may be a String (form field),
 * {@link Resource} (file part), Object (entity to be
 * encoded, e.g. as JSON), {@link HttpEntity} (part content and headers), a
 * {@link infra.http.codec.multipart.Part}, or a
 * {@link org.reactivestreams.Publisher} of any of the above.
 * (</td>
 * <td>{@link RequestPartArgumentResolver
 * RequestPartArgumentResolver}</td>
 * </tr>
 * <tr>
 * <td>{@link infra.web.annotation.CookieValue @CookieValue}</td>
 * <td>Add a cookie</td>
 * <td>{@link infra.web.service.invoker.CookieValueArgumentResolver
 * CookieValueArgumentResolver}</td>
 * </tr>
 * </table>
 *
 * @author Rossen Stoyanchev
 * @author <a href="https://github.com/TAKETODAY">Harry Yang</a>
 * @since 4.0
 */
@Documented
@Retention(RetentionPolicy.RUNTIME)
@Target({ ElementType.TYPE, ElementType.METHOD })
@Reflective(HttpExchangeReflectiveProcessor.class)
public @interface HttpExchange {

  /**
   * This is an alias for {@link #url}.
   */
  @AliasFor("url")
  String value() default "";

  /**
   * The URL for the request, either a full URL or a path only that is relative
   * to a URL declared in a type-level {@code @HttpExchange}, and/or a globally
   * configured base URL.
   * <p>By default, this is empty.
   */
  @AliasFor("value")
  String url() default "";

  /**
   * The HTTP method to use.
   * <p>Supported at the type level as well as at the method level.
   * When used at the type level, all method-level mappings inherit this value.
   * <p>By default, this is empty.
   */
  String method() default "";

  /**
   * The media type for the {@code "Content-Type"} header.
   * <p>Supported at the type level as well as at the method level, in which
   * case the method-level values override type-level values.
   * <p>By default, this is empty.
   */
  String contentType() default "";

  /**
   * The media types for the {@code "Accept"} header.
   * <p>Supported at the type level as well as at the method level, in which
   * case the method-level values override type-level values.
   * <p>By default, this is empty.
   */
  String[] accept() default {};

  /**
   * The additional headers to use, as an array of {@code name=value} pairs.
   * <p>Multiple comma-separated values are accepted, and placeholders are
   * supported in these values. However, Accept and Content-Type headers are
   * ignored: see {@link #accept()} and {@link #contentType()}.
   * <p>Supported at the type level as well as at the method level, in which
   * case the method-level values override type-level values.
   * <p>By default, this is empty.
   *
   * @since 5.0
   */
  String[] headers() default {};

}
