/*
 * Original Author -> Harry Yang (taketoday@foxmail.com) https://taketoday.cn
 * Copyright © Harry Yang & 2017 - 2023 All Rights Reserved.
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see [http://www.gnu.org/licenses/]
 */
package cn.taketoday.bytecode.core;

/**
 * The <code>GeneratorStrategy</code>. is responsible for taking a
 * {@link ClassGenerator} and producing a byte array containing the
 * data for the generated <code>Class</code>. By providing your own strategy you
 * may examine or modify the generated class before it is loaded. Typically this
 * will be accomplished by subclassing {@link DefaultGeneratorStrategy} and
 * overriding the appropriate protected method.
 *
 * @see AbstractClassGenerator#setStrategy
 */
@FunctionalInterface
public interface GeneratorStrategy {

  /**
   * Generate the class.
   *
   * @param cg a class generator on which you can call
   * {@link ClassGenerator#generateClass}
   * @return a byte array containing the bits of a valid Class
   */
  byte[] generate(ClassGenerator cg) throws Exception;

  /**
   * The <code>GeneratorStrategy</code> in use does not currently, but may in the
   * future, affect the caching of classes generated by
   * {@link AbstractClassGenerator}, so this is a reminder that you should
   * correctly implement <code>equals</code> and <code>hashCode</code> to avoid
   * generating too many classes.
   */
  boolean equals(Object o);
}
