/*
 * Copyright 2017 - 2025 the original author or authors.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see [https://www.gnu.org/licenses/]
 */

package infra.context.support;

import org.jspecify.annotations.Nullable;

import java.util.Locale;

import infra.beans.BeansException;
import infra.context.ApplicationContext;

/**
 * {@link infra.context.ApplicationContext} implementation
 * which supports programmatic registration of beans and messages,
 * rather than reading bean definitions from external configuration sources.
 * Mainly useful for testing.
 *
 * @author Rod Johnson
 * @author Juergen Hoeller
 * @author <a href="https://github.com/TAKETODAY">Harry Yang</a>
 * @see #registerSingleton
 * @see #registerPrototype
 * @see #registerBeanDefinition
 * @see #refresh
 */
public class StaticApplicationContext extends GenericApplicationContext {

  private final StaticMessageSource staticMessageSource;

  /**
   * Create a new StaticApplicationContext.
   *
   * @see #registerSingleton
   * @see #registerPrototype
   * @see #registerBeanDefinition
   * @see #refresh
   */
  public StaticApplicationContext() throws BeansException {
    this(null);
  }

  /**
   * Create a new StaticApplicationContext with the given parent.
   *
   * @see #registerSingleton
   * @see #registerPrototype
   * @see #registerBeanDefinition
   * @see #refresh
   */
  public StaticApplicationContext(@Nullable ApplicationContext parent) throws BeansException {
    super(parent);

    // Initialize and register a StaticMessageSource.
    this.staticMessageSource = new StaticMessageSource();
    getBeanFactory().registerSingleton(MESSAGE_SOURCE_BEAN_NAME, this.staticMessageSource);
  }

  /**
   * Overridden to turn it into a no-op, to be more lenient towards test cases.
   */
  @Override
  protected void assertBeanFactoryActive() { }

  /**
   * Return the internal StaticMessageSource used by this context.
   * Can be used to register messages on it.
   *
   * @see #addMessage
   */
  public final StaticMessageSource getStaticMessageSource() {
    return this.staticMessageSource;
  }

  /**
   * Associate the given message with the given code.
   *
   * @param code lookup code
   * @param locale the locale message should be found within
   * @param defaultMessage message associated with this lookup code
   * @see #getStaticMessageSource
   */
  public void addMessage(String code, Locale locale, String defaultMessage) {
    getStaticMessageSource().addMessage(code, locale, defaultMessage);
  }

}
